package com.biz.crm.mdm.business.position.local.service.internal;

import com.biz.crm.mdm.business.position.local.entity.PositionRoleEntity;
import com.biz.crm.mdm.business.position.local.repository.PositionRoleRepository;
import com.biz.crm.mdm.business.position.local.service.PositionRoleService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.rbac.sdk.service.RoleVoCacheService;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 职位-角色关联表接口实现
 *
 * @author ning.zhang
 * @date 2021/10/13
 */
@Service
public class PositionRoleServiceImpl implements PositionRoleService {

  @Autowired(required = false)
  private PositionRoleRepository positionRoleRepository;
  @Autowired(required = false)
  private RoleVoCacheService roleVoCacheService;

  @Override
  public List<PositionRoleEntity> findByPositionCode(String positionCode) {
    if (StringUtils.isEmpty(positionCode)) {
      return Lists.newArrayList();
    }
    return positionRoleRepository.findByPositionCode(positionCode, TenantUtils.getTenantCode());
  }

  @Override
  public List<PositionRoleEntity> findByPositionCodes(List<String> positionCodes) {
    if (CollectionUtils.isEmpty(positionCodes)) {
      return Lists.newArrayList();
    }
    return positionRoleRepository.findByPositionCodes(positionCodes, TenantUtils.getTenantCode());
  }

  @Override
  public void unbindByPositionCodes(List<String> positionCodes) {
    Validate.isTrue(CollectionUtils.isNotEmpty(positionCodes), "职位编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    List<PositionRoleEntity> entities = positionRoleRepository.findByPositionCodes(positionCodes, tenantCode);
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    this.positionRoleRepository.deleteByPositionCodes(positionCodes, tenantCode);
    //角色关联职位信息发生改变,清理角色缓存
    entities.forEach(entity -> {
      this.roleVoCacheService.clearCache(tenantCode, entity.getRoleCode());
    });
  }

  @Override
  public void bindByPositionCodeAndRoleCodes(String positionCode, List<String> roleCodes) {
    Validate.notBlank(positionCode, "职位级别编码不能为空");
    //传入roleCodes为空,视为不绑定关系
    if (CollectionUtils.isEmpty(roleCodes)) {
      return;
    }
    String tenantCode = TenantUtils.getTenantCode();
    List<PositionRoleEntity> roleEntities = roleCodes.stream().map(roleCode -> {
      PositionRoleEntity entity = new PositionRoleEntity();
      entity.setRoleCode(roleCode);
      entity.setPositionCode(positionCode);
      entity.setTenantCode(tenantCode);
      return entity;
    }).collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(roleEntities)) {
      this.positionRoleRepository.saveBatch(roleEntities);
      //角色关联职位之别信息发生改变,清理角色缓存
      roleCodes.forEach(roleCode -> {
        this.roleVoCacheService.clearCache(tenantCode, roleCode);
      });
    }
  }

  @Override
  public List<PositionRoleEntity> findByRoleCodes(List<String> roleCodes) {
    if (CollectionUtils.isEmpty(roleCodes)) {
      return Lists.newArrayList();
    }
    return this.positionRoleRepository.findByRoleCodes(roleCodes, TenantUtils.getTenantCode());
  }

  @Override
  public void bindByPositionCodesAndRoleCode(List<String> positionCodes, String roleCode) {
    Validate.isTrue(CollectionUtils.isNotEmpty(positionCodes), "缺失职位编码");
    Validate.notBlank(roleCode, "缺失角色编码");
    String tenantCode = TenantUtils.getTenantCode();
    List<PositionRoleEntity> list = this.positionRoleRepository.findByPositionCodesAndRoleCode(positionCodes, roleCode, tenantCode);
    //过滤掉已经存在的关联
    if (CollectionUtils.isNotEmpty(list)) {
      List<String> existPositionCodes = list.stream().map(PositionRoleEntity::getPositionCode).collect(Collectors.toList());
      positionCodes = positionCodes.stream().filter(s -> !existPositionCodes.contains(s)).collect(Collectors.toList());
    }
    //已经没有需要绑定的职位和角色关系,直接返回
    if (CollectionUtils.isEmpty(positionCodes)) {
      return;
    }
    List<PositionRoleEntity> entities = positionCodes.stream().map(positionCode -> {
      PositionRoleEntity entity = new PositionRoleEntity();
      entity.setTenantCode(tenantCode);
      entity.setRoleCode(roleCode);
      entity.setPositionCode(positionCode);
      return entity;
    }).collect(Collectors.toList());
    this.positionRoleRepository.saveBatch(entities);
  }

  @Override
  public void unbindByPositionCodesAndRoleCode(List<String> positionCodes, String roleCode) {
    Validate.isTrue(CollectionUtils.isNotEmpty(positionCodes), "缺失职位编码");
    Validate.notBlank(roleCode, "缺失角色编码");
    List<PositionRoleEntity> list = this.positionRoleRepository.findByPositionCodesAndRoleCode(positionCodes, roleCode, TenantUtils.getTenantCode());
    //没有需要解绑的数据,直接返回
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    List<String> positionRoleIds = list.stream().map(PositionRoleEntity::getId).collect(Collectors.toList());
    this.positionRoleRepository.removeByIds(positionRoleIds);
  }
}
