package com.biz.crm.mdm.business.price.local.service.internal;

import com.biz.crm.mdm.business.price.sdk.constant.PriceConstant;
import com.biz.crm.mdm.business.price.sdk.register.PriceDimensionRegister;
import com.biz.crm.mdm.business.price.sdk.service.PriceDimensionContainerService;
import com.biz.crm.mdm.business.price.sdk.vo.CombineDimensionVo;
import com.biz.crm.mdm.business.price.sdk.vo.PriceDimensionDictVo;
import com.biz.crm.mdm.business.price.sdk.vo.PriceDimensionSelectVo;
import com.google.common.collect.Lists;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 价格维度sdk接口实现
 *
 * @author sunx
 * @date 2021/12/31
 */
@Service
public class PriceDimensionContainerServiceImpl
    implements PriceDimensionContainerService, InitializingBean {

  @Autowired(required = false)
  private List<PriceDimensionRegister> registers;

  /** 启动注册器属性校验 */
  @Override
  public void afterPropertiesSet() throws Exception {
    if (CollectionUtils.isEmpty(registers)) {
      return;
    }
    Map<String, List<String>> mapCode =
        registers.stream()
            .collect(
                Collectors.groupingBy(
                    PriceDimensionRegister::getCode,
                    Collectors.mapping(PriceDimensionRegister::getDesc, Collectors.toList())));
    Map<Integer, List<String>> mapSort =
        registers.stream()
            .collect(
                Collectors.groupingBy(
                    PriceDimensionRegister::sort,
                    Collectors.mapping(PriceDimensionRegister::getDesc, Collectors.toList())));
    List<String> list = Lists.newLinkedList();
    for (Entry<String, List<String>> item : mapCode.entrySet()) {
      if (item.getValue().size() > 1) {
        list.add(item.getValue().stream().collect(Collectors.joining(",")) + "维度编码重复");
      }
    }
    for (Entry<Integer, List<String>> item : mapSort.entrySet()) {
      if (item.getValue().size() > 1) {
        list.add(item.getValue().stream().collect(Collectors.joining(",")) + "排序重复");
      }
    }
    if (CollectionUtils.isNotEmpty(list)) {
      Validate.isTrue(
          false,
          list.stream().collect(Collectors.joining(",")) + ";为保证系统稳定，询价维度编码或排序不允许重复，请检查！！！!");
    }
  }

  @Override
  public List<PriceDimensionDictVo> findDimensionSelect() {
    if (CollectionUtils.isEmpty(registers)) {
      return Lists.newLinkedList();
    }
    List<PriceDimensionDictVo> re = Lists.newArrayList();
    final List<PriceDimensionRegister> list =
        registers.stream()
            .sorted(
                Comparator.comparing(PriceDimensionRegister::sort)
                    .thenComparing(Comparator.comparing(PriceDimensionRegister::getCode)))
            .collect(Collectors.toList());
    for (PriceDimensionRegister item : list) {
      PriceDimensionDictVo cur = new PriceDimensionDictVo();
      cur.setCode(item.getCode());
      cur.setName(item.getDesc());
      cur.setSort(item.sort());
      re.add(cur);
    }
    return re;
  }

  @Override
  public CombineDimensionVo getCombineDimensionVo(Set<String> dimensionCodeSet) {
    if (CollectionUtils.isEmpty(dimensionCodeSet)) {
      return null;
    }
    List<PriceDimensionDictVo> list = this.findDimensionSelect();
    if (CollectionUtils.isEmpty(list)) {
      return null;
    }
    List<String> codeList = Lists.newArrayList();
    List<String> nameList = Lists.newArrayList();
    for (PriceDimensionDictVo item : list) {
      if (dimensionCodeSet.contains(item.getCode())) {
        codeList.add(item.getCode());
        nameList.add(item.getName());
      }
    }
    CombineDimensionVo re = new CombineDimensionVo();
    re.setDimensionKey(StringUtils.join(codeList, PriceConstant.SEPARATOR));
    re.setDimensionValue(StringUtils.join(nameList, PriceConstant.NAME_SEPARATOR));
    return re;
  }

  @Override
  public List<PriceDimensionSelectVo> findSelectSourceDataByCodeAndKeyword(
      String code, String keyword) {
    return this.getRegister(code).findSelectVoByKeyword(keyword);
  }

  @Override
  public String findDimensionNameByTypeDetailCode(String typeDetailCode) {
    if (StringUtils.isBlank(typeDetailCode) || !typeDetailCode.contains("-")) {
      return StringUtils.EMPTY;
    }
    final String[] arr = StringUtils.split(typeDetailCode, "-");
    if (arr.length < 2) {
      return StringUtils.EMPTY;
    }
    List<String> list = Lists.newArrayList();
    for (int i = 1; i < arr.length; i++) {
      String dimensionCode = arr[i];
      final PriceDimensionRegister register = this.getRegister(dimensionCode);
      if (Objects.isNull(register)) {
        continue;
      }
      list.add(register.getDesc());
    }
    if (CollectionUtils.isEmpty(list)) {
      return StringUtils.EMPTY;
    }
    return list.stream().collect(Collectors.joining("/"));
  }

  /**
   * 根据价格维度系统编码获取具体的维度注册信息
   *
   * @param code
   * @return
   */
  private PriceDimensionRegister getRegister(String code) {
    Validate.notBlank(code, "价格维度系统编码不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(registers), "无可用的价格维度注册信息");
    PriceDimensionRegister register = null;
    for (PriceDimensionRegister cur : registers) {
      if (cur.getCode().equals(code)) {
        register = cur;
        break;
      }
    }
    Validate.notNull(register, "未获取到匹配的价格维度注册信息");
    return register;
  }
}
