package com.biz.crm.mdm.business.price.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.price.local.entity.Price;
import com.biz.crm.mdm.business.price.local.service.PriceService;
import com.biz.crm.mdm.business.price.sdk.dto.PricePaginationDto;
import com.biz.crm.mdm.business.price.sdk.dto.PriceSearchDimensionDto;
import com.biz.crm.mdm.business.price.sdk.service.PriceDimensionContainerService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceDimensionSelectVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import java.util.List;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 价格维护主信息(Price)表控制层
 *
 * @author sunx
 * @date 2021-12-30 17:46:07
 */
@Slf4j
@Api(tags = "价格维护主信息: Price: 价格维护主信息")
@RestController
@RequestMapping(value = {"/v1/price/price"})
public class PriceController {

  @Autowired(required = false) private PriceService priceService;

  @Autowired(required = false) private PriceDimensionContainerService priceDimensionContainerService;

  /**
   * 查询分页列表
   *
   * @param pageable
   * @param paginationDto
   * @return
   */
  @ApiOperation(value = "查询分页列表")
  @GetMapping(value = {"/findByConditions"})
  public Result<Page<Price>> findByConditions(
      @PageableDefault(50) Pageable pageable, PricePaginationDto paginationDto) {
    try {
      Page<Price> result = this.priceService.findByConditions(pageable, paginationDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 详情
   *
   * @param id
   * @return
   */
  @ApiOperation(value = "详情")
  @GetMapping(value = {"/findDetailById"})
  public Result<Price> findDetailById(@RequestParam("id") String id) {
    try {
      Price price = this.priceService.findDetailById(id);
      return Result.ok(price);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 创建
   *
   * @param price
   * @return
   */
  @ApiOperation(value = "创建")
  @PostMapping(value = "")
  public Result<Price> create(@RequestBody Price price) {
    try {
      Price current = this.priceService.create(price);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 创建
   *
   * @param price
   * @return
   */
  @ApiOperation(value = "更新")
  @PatchMapping(value = "")
  public Result<Price> update(@RequestBody Price price) {
    try {
      Price current = this.priceService.update(price);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 逻辑删除
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "逻辑删除")
  @DeleteMapping("/delete")
  public Result<?> delete(@RequestParam("ids") List<String> ids) {
    try {
      this.priceService.updateDelFlagByIds(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据维度配置信息获取历史数据
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "根据维度配置信息获取历史数据")
  @GetMapping(value = {"/findDetailByPriceSearchDimensionDto"})
  public Result<Price> findDetailByPriceSearchDimensionDto(PriceSearchDimensionDto dto) {
    try {
      Price price = this.priceService.findDetailByPriceSearchDimensionDto(dto);
      return Result.ok(price);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据维度编码和keyword获取对应的维度数据源下拉
   *
   * @param dimensionCode 维度编码
   * @param keyword 模糊搜索关键字
   * @return
   */
  @ApiOperation(value = "根据维度编码和keyword获取对应的维度数据源下拉")
  @GetMapping("/findSelectSourceDataByKeywordAndCode")
  public Result<List<PriceDimensionSelectVo>> findSelectSourceDataByKeywordAndCode(
      @RequestParam("dimensionCode") String dimensionCode,
      @RequestParam(value = "keyword", required = false, defaultValue = "") String keyword) {
    try {
      List<PriceDimensionSelectVo> list =
          this.priceDimensionContainerService.findSelectSourceDataByCodeAndKeyword(
              dimensionCode, keyword);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
