package com.biz.crm.mdm.business.price.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.price.local.entity.Price;
import com.biz.crm.mdm.business.price.local.entity.PriceDimension;
import com.biz.crm.mdm.business.price.local.entity.PriceLog;
import com.biz.crm.mdm.business.price.local.entity.PriceType;
import com.biz.crm.mdm.business.price.local.entity.PriceTypeDetailItem;
import com.biz.crm.mdm.business.price.local.repository.PriceLogRepository;
import com.biz.crm.mdm.business.price.local.repository.PriceRepository;
import com.biz.crm.mdm.business.price.local.repository.PriceTypeRepository;
import com.biz.crm.mdm.business.price.local.service.PriceDimensionService;
import com.biz.crm.mdm.business.price.local.service.PriceService;
import com.biz.crm.mdm.business.price.local.service.PriceTypeDetailItemService;
import com.biz.crm.mdm.business.price.sdk.constant.PriceConstant;
import com.biz.crm.mdm.business.price.sdk.dto.PriceEventDto;
import com.biz.crm.mdm.business.price.sdk.dto.PricePaginationDto;
import com.biz.crm.mdm.business.price.sdk.dto.PriceSearchDimensionDto;
import com.biz.crm.mdm.business.price.sdk.enums.EffectiveStatusEnum;
import com.biz.crm.mdm.business.price.sdk.event.PriceLogEventListener;
import com.biz.crm.mdm.business.price.sdk.service.PriceDimensionContainerService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceDimensionDictVo;
import com.biz.crm.mdm.business.price.sdk.vo.PriceVo;
import com.biz.crm.mdm.business.product.sdk.dto.ProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 价格维护主信息(Price)表服务实现类
 *
 * @author sunx
 * @date 2021-12-30 17:46:06
 */
@Service("priceService")
public class PriceServiceImpl implements PriceService {

  @Autowired(required = false)
  private PriceRepository priceRepository;

  @Autowired(required = false)
  private PriceTypeRepository priceTypeRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private PriceDimensionService priceDimensionService;

  @Autowired(required = false)
  private PriceTypeDetailItemService priceTypeDetailItemService;

  @Autowired(required = false)
  private PriceDimensionContainerService priceDimensionContainerService;

  @Autowired(required = false)
  private PriceLogRepository priceLogRepository;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Override
  public Page<Price> findByConditions(Pageable pageable, PricePaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new PricePaginationDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<Price> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());

    if (!StringUtils.isAllBlank(dto.getProductCode(), dto.getProductName())) {
      final ProductQueryDto queryDto = new ProductQueryDto();
      queryDto.setProductCode(dto.getProductCode());
      queryDto.setProductName(dto.getProductName());
      queryDto.setTenantCode(TenantUtils.getTenantCode());
      final List<ProductVo> productVoList = this.productVoService.findByProductQueryDto(queryDto);
      if (CollectionUtils.isEmpty(productVoList)) {
        return new Page<>();
      }
      dto.setProductCodeSet(
          productVoList.stream()
              .map(ProductVo::getProductCode)
              .limit(500)
              .collect(Collectors.toSet()));
    }
    Page<Price> pageResult = this.priceRepository.findByConditions(page, dto);
    if (Objects.nonNull(pageResult) && CollectionUtils.isNotEmpty(pageResult.getRecords())) {
      Set<String> priceCodeSet =
          pageResult.getRecords().stream()
              .filter(a -> StringUtils.isNotBlank(a.getPriceCode()))
              .map(Price::getPriceCode)
              .collect(Collectors.toSet());
      List<PriceDimension> list =
          priceDimensionService.findByPriceCodes(Lists.newArrayList(priceCodeSet));
      Map<String, List<PriceDimension>> map =
          list.stream()
              .filter(a -> StringUtils.isNotBlank(a.getPriceCode()))
              .collect(Collectors.groupingBy(PriceDimension::getPriceCode));
      for (Price price : pageResult.getRecords()) {
        price.setDimensionList(map.get(price.getPriceCode()));
        price.setEffectiveStatus(this.findEffectiveStatus(price));
      }
    }
    return pageResult;
  }

  @Override
  public Price findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Price price = this.priceRepository.findById(id);
    if (Objects.isNull(price)) {
      return null;
    }
    if (StringUtils.isNotBlank(price.getTypeCode())) {
      PriceType priceType = this.priceTypeRepository.findByTypeCode(price.getTypeCode());
      if (Objects.nonNull(priceType)) {
        price.setTypeName(priceType.getTypeName());
      }
    }
    List<PriceDimension> dimensionList =
        priceDimensionService.findByPriceCodes(Lists.newArrayList(price.getPriceCode()));
    price.setDimensionList(dimensionList);
    return price;
  }

  @Override
  @Transactional
  public Price create(Price price) {
    this.createValidation(price);
    price.setTenantCode(TenantUtils.getTenantCode());
    price.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    price.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    if (StringUtils.isBlank(price.getPriceCode())) {
      price.setPriceCode(
          this.generateCodeService.generateCode(PriceConstant.PRICE_CODE_PREFIX, 1).get(0));
    } else {
      Integer count = this.priceRepository.countByTypeCode(price.getPriceCode());
      Validate.isTrue(null == count || 1 > count, price.getPriceCode() + "价格编码已存在");
    }
    this.saveDimensionInfo(price);
    this.priceRepository.saveOrUpdate(price);
    PriceLog priceLog =
        nebulaToolkitService.copyObjectByWhiteList(
            price, PriceLog.class, HashSet.class, ArrayList.class);
    priceLog.setId(null);
    this.priceLogRepository.save(priceLog);
    // 业务日志创建
    PriceEventDto priceEventDto = new PriceEventDto();
    PriceVo nowVo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            price, PriceVo.class, HashSet.class, ArrayList.class);
    priceEventDto.setNewest(nowVo);
    priceEventDto.setOriginal(null);
    SerializableBiConsumer<PriceLogEventListener, PriceEventDto> onCreate =
        PriceLogEventListener::onCreate;
    this.nebulaNetEventClient.publish(priceEventDto, PriceLogEventListener.class, onCreate);

    return price;
  }

  @Override
  @Transactional
  public Price update(Price price) {
    this.updateValidation(price);
    String currentId = price.getId();
    Price current = priceRepository.findById(currentId);
    Validate.notNull(current, "修改信息不存在");
    price.setTenantCode(TenantUtils.getTenantCode());
    Validate.isTrue(price.getPriceCode().equals(current.getPriceCode()), "价格编码不能修改");
    this.saveDimensionInfo(price);
    this.priceRepository.saveOrUpdate(price);
    PriceLog priceLog =
        nebulaToolkitService.copyObjectByWhiteList(
            price, PriceLog.class, HashSet.class, ArrayList.class);
    priceLog.setId(null);
    this.priceLogRepository.save(priceLog);
    // 业务日志更新
    PriceEventDto priceEventDto = new PriceEventDto();
    PriceVo oldVO =
        this.nebulaToolkitService.copyObjectByWhiteList(
            current, PriceVo.class, HashSet.class, ArrayList.class);
    PriceVo nowVo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            price, PriceVo.class, HashSet.class, ArrayList.class);
    priceEventDto.setNewest(nowVo);
    priceEventDto.setOriginal(oldVO);
    SerializableBiConsumer<PriceLogEventListener, PriceEventDto> onUpdate =
        PriceLogEventListener::onUpdate;
    this.nebulaNetEventClient.publish(priceEventDto, PriceLogEventListener.class, onUpdate);
    return price;
  }

  /**
   * 保存价格设置维度配置明细
   *
   * @param price
   */
  @Transactional
  public void saveDimensionInfo(Price price) {
    Validate.notNull(price, "价格信息不能为空");
    List<PriceTypeDetailItem> priceTypeDetailItemList =
        priceTypeDetailItemService.findByTypeDetailCodes(
            Lists.newArrayList(price.getTypeDetailCode()));
    Validate.isTrue(CollectionUtils.isNotEmpty(priceTypeDetailItemList), "定价维度对应的配置不存在");
    Validate.isTrue(CollectionUtils.isNotEmpty(price.getDimensionList()), "定价维度配置不能为空");
    Set<String> set =
        price.getDimensionList().stream()
            .filter(a -> StringUtils.isNotBlank(a.getDimensionCode()))
            .map(PriceDimension::getDimensionCode)
            .collect(Collectors.toSet());
    for (PriceTypeDetailItem item : priceTypeDetailItemList) {
      if (!set.contains(item.getDimensionCode())) {
        Validate.isTrue(false, item.getDimensionCode() + "对应的维度配置信息缺失");
      }
    }
    // k-dimensionCode,v-relateCode
    Map<String, String> map =
        price.getDimensionList().stream()
            .collect(
                Collectors.toMap(
                    PriceDimension::getDimensionCode, PriceDimension::getRelateCode, (a, b) -> a));
    List<PriceDimensionDictVo> dimensionSelect =
        this.priceDimensionContainerService.findDimensionSelect();
    List<String> joinCodeList = Lists.newArrayList();
    joinCodeList.add(price.getTypeDetailCode());
    for (PriceDimensionDictVo item : dimensionSelect) {
      if (map.keySet().contains(item.getCode())) {
        joinCodeList.add(map.get(item.getCode()));
      }
    }
    price.setRelateCodeJoin(
        joinCodeList.stream().collect(Collectors.joining(PriceConstant.SEPARATOR)));
    for (PriceDimension item : price.getDimensionList()) {
      item.setPriceCode(price.getPriceCode());
      item.setTypeCode(price.getTypeCode());
      item.setTypeDetailCode(price.getTypeDetailCode());
    }
    this.priceDimensionService.saveBatch(price.getDimensionList(), price.getPriceCode());
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<Price> byIds = this.priceRepository.findByIds(ids);
    Validate.notEmpty(byIds, "未查询到相关信息");
    Validate.isTrue(byIds.size() == ids.size(), "id集合与查询出的实体集合不相等");
    this.priceRepository.updateDelFlagByIds(ids);
    // 业务日志删除
    for (Price byId : byIds) {
      PriceEventDto priceEventDto = new PriceEventDto();
      PriceVo oldVO =
          this.nebulaToolkitService.copyObjectByWhiteList(
              byId, PriceVo.class, HashSet.class, ArrayList.class);
      priceEventDto.setNewest(null);
      priceEventDto.setOriginal(oldVO);
      SerializableBiConsumer<PriceLogEventListener, PriceEventDto> onDelete =
          PriceLogEventListener::onDelete;
      this.nebulaNetEventClient.publish(priceEventDto, PriceLogEventListener.class, onDelete);
    }
  }

  @Override
  public Boolean checkExistsPriceByTypeDetailCodes(Set<String> typeDetailCodeSet) {
    if (CollectionUtils.isEmpty(typeDetailCodeSet)) {
      return false;
    }
    Integer count = this.priceRepository.countByTypeDetailCodes(typeDetailCodeSet);
    return count != null && count > 0;
  }

  @Override
  public Price findDetailByPriceSearchDimensionDto(PriceSearchDimensionDto dto) {
    if (Objects.isNull(dto)
        || StringUtils.isAnyBlank(dto.getTypeCode(), dto.getTypeDetailCode())
        || Objects.isNull(dto.getMap())) {
      return null;
    }
    List<PriceDimensionDictVo> dimensionSelect =
        this.priceDimensionContainerService.findDimensionSelect();
    if (CollectionUtils.isEmpty(dimensionSelect)) {
      return null;
    }
    List<String> list = Lists.newLinkedList();
    list.add(dto.getTypeDetailCode());
    for (PriceDimensionDictVo item : dimensionSelect) {
      if (dto.getMap().containsKey(item.getCode())) {
        list.add(dto.getMap().get(item.getCode()));
      }
    }
    if (CollectionUtils.isEmpty(list)) {
      return null;
    }
    String relateCodeJoin = list.stream().collect(Collectors.joining(PriceConstant.SEPARATOR));

    return this.priceRepository.findByTypeCodeAndDetailCodeAndRelateCodeJoin(
        dto.getTypeCode(), dto.getTypeDetailCode(), relateCodeJoin);
  }

  /**
   * 通过编码查询
   *
   * @param typeCodeList
   * @return
   */
  @Override
  public List<Price> findByTypeCodes(List<String> typeCodeList) {
    if (CollectionUtils.isEmpty(typeCodeList)) {
      return Lists.newArrayList();
    }
    return priceRepository.findByTypeCodes(typeCodeList);
  }

  private void createValidation(Price price) {
    price.setId(null);
    this.validation(price);
  }

  private void updateValidation(Price price) {
    this.validation(price);
  }

  private void validation(Price price) {
    Validate.notBlank(price.getTypeCode(), "请选择价格类型信息");
    Validate.notBlank(price.getTypeDetailCode(), "请选择定价维度信息");
    Validate.notNull(price.getBeginTime(), "有效开始时间不能为空");
    Validate.notNull(price.getEndTime(), "有效截止时间不能为空");
    Validate.isTrue(price.getEndTime().compareTo(price.getBeginTime()) > 0, "截止时间需大于开始时间");
    Validate.isTrue(price.getEndTime().compareTo(new Date()) > 0, "截止时间需大于当前时间");
    price.setPrice(Optional.ofNullable(price.getPrice()).orElse(BigDecimal.ZERO));
    Validate.isTrue(price.getPrice().compareTo(BigDecimal.ZERO) > 0, "请录入大于0的价格数据");
  }

  /**
   * 获取展示的生效状态
   *
   * @param price
   * @return
   */
  private String findEffectiveStatus(Price price) {
    if (Objects.isNull(price)
        || Objects.isNull(price.getBeginTime())
        || Objects.isNull(price.getEndTime())) {
      return EffectiveStatusEnum.DEFAULT.getDictCode();
    }
    Date now = new Date();
    Date begin = price.getBeginTime();
    Date end = price.getEndTime();
    if (begin.compareTo(now) > 0) {
      return EffectiveStatusEnum.DEFAULT.getDictCode();
    }
    if (begin.compareTo(now) <= 0 && end.compareTo(now) > 0) {
      return EffectiveStatusEnum.ACTIVE.getDictCode();
    }
    if (end.compareTo(now) <= 0) {
      return EffectiveStatusEnum.OVERDUE.getDictCode();
    }
    return EffectiveStatusEnum.DEFAULT.getDictCode();
  }
}
