package com.biz.crm.mdm.business.price.local.service.internal;

import com.biz.crm.mdm.business.price.local.entity.PriceTypeDetail;
import com.biz.crm.mdm.business.price.local.entity.PriceTypeDetailItem;
import com.biz.crm.mdm.business.price.local.repository.PriceTypeDetailRepository;
import com.biz.crm.mdm.business.price.local.service.PriceTypeDetailItemService;
import com.biz.crm.mdm.business.price.local.service.PriceTypeDetailService;
import com.biz.crm.mdm.business.price.sdk.service.PriceDimensionContainerService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceDimensionDictVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 价格类型维度信息(PriceTypeDetail)表服务实现类
 *
 * @author sunx
 * @date 2021-12-30 17:46:13
 */
@Service("priceTypeDetailService")
public class PriceTypeDetailServiceImpl implements PriceTypeDetailService {

  @Autowired(required = false) private PriceTypeDetailRepository priceTypeDetailRepository;

  @Autowired(required = false) private PriceTypeDetailItemService priceTypeDetailItemService;

  @Autowired(required = false) private PriceDimensionContainerService priceDimensionContainerService;

  @Override
  @Transactional
  public void saveBatch(List<PriceTypeDetail> list, String typeCode) {
    Validate.notBlank(typeCode, "价格类型编码不能为空");
    this.priceTypeDetailRepository.deleteByTypeCode(typeCode);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    Set<String> set = Sets.newHashSet();
    for (PriceTypeDetail item : list) {
      item.setTypeCode(typeCode);
      item.setTenantCode(TenantUtils.getTenantCode());
      Validate.notBlank(item.getTypeDetailCode(), "定价维度编码不能为空");
      Validate.notBlank(item.getTypeDetailName(), "定价维度名称不能为空");
      Validate.notNull(item.getSort(), "排序不能为空");
      Validate.isTrue(CollectionUtils.isNotEmpty(item.getItemList()), "价格维度配置明细不能为空");
      Validate.isTrue(set.add(item.getTypeDetailCode()), item.getTypeDetailName() + "定价维度编码存在重复");
    }
    this.priceTypeDetailRepository.saveBatch(list);
    List<PriceTypeDetailItem> itemList = Lists.newLinkedList();
    for (PriceTypeDetail item : list) {
      for (PriceTypeDetailItem sub : item.getItemList()) {
        sub.setTypeCode(typeCode);
        sub.setTypeDetailCode(item.getTypeDetailCode());
        sub.setTenantCode(TenantUtils.getTenantCode());
        Validate.notBlank(sub.getDimensionCode(), item.getTypeDetailName() + "对应的明细中维度编码不能为空");
        itemList.add(sub);
      }
    }
    if (CollectionUtils.isNotEmpty(itemList)) {
      this.priceTypeDetailItemService.saveBatch(itemList, typeCode);
    }
  }

  @Override
  public List<PriceTypeDetail> findByTypeCodeAndTypeDetailNameLike(
      String typeCode, String typeDetailName) {
    List<PriceTypeDetail> list =
        this.priceTypeDetailRepository.findByTypeCodeAndTypeDetailNameLike(
            typeCode, typeDetailName);
    return this.initDetails(list, Lists.newArrayList(typeCode));
  }

  @Override
  public List<PriceTypeDetail> findByTypeCodes(List<String> typeCodeList) {
    if (CollectionUtils.isEmpty(typeCodeList)) {
      return Lists.newLinkedList();
    }
    List<PriceTypeDetail> list = this.priceTypeDetailRepository.findByTypeCodes(typeCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return this.initDetails(list, typeCodeList);
  }

  /**
   * 期初明细的配置明细
   *
   * @param list
   * @param typeCodeList
   * @return
   */
  private List<PriceTypeDetail> initDetails(List<PriceTypeDetail> list, List<String> typeCodeList) {
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<PriceDimensionDictVo> dictVoList =
        this.priceDimensionContainerService.findDimensionSelect();
    List<PriceTypeDetailItem> itemList =
        this.priceTypeDetailItemService.findByTypeCodes(typeCodeList);
    // k-typeDetailCode
    Map<String, List<PriceTypeDetailItem>> mapItem =
        itemList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTypeDetailCode()))
            .collect(Collectors.groupingBy(PriceTypeDetailItem::getTypeDetailCode));
    for (PriceTypeDetail item : list) {
      List<PriceTypeDetailItem> subList = mapItem.get(item.getTypeDetailCode());
      if (CollectionUtils.isEmpty(subList)) {
        continue;
      }
      Map<String, PriceTypeDetailItem> curMap =
          subList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getDimensionCode()))
              .collect(
                  Collectors.toMap(
                      PriceTypeDetailItem::getDimensionCode, Function.identity(), (a, b) -> a));
      if (CollectionUtils.isNotEmpty(dictVoList)) {
        subList = Lists.newLinkedList();
        for (PriceDimensionDictVo vo : dictVoList) {
          PriceTypeDetailItem priceTypeDetailItem = curMap.get(vo.getCode());
          if (Objects.nonNull(priceTypeDetailItem)) {
            priceTypeDetailItem.setDimensionName(vo.getName());
            subList.add(priceTypeDetailItem);
          }
        }
      }
      item.setItemList(subList);
    }
    return list;
  }
}
