package com.biz.crm.mdm.business.product.level.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.product.level.local.service.ProductLevelVoService;
import com.biz.crm.mdm.business.product.level.sdk.dto.ProductLevelCreateDto;
import com.biz.crm.mdm.business.product.level.sdk.dto.ProductLevelPaginationDto;
import com.biz.crm.mdm.business.product.level.sdk.dto.ProductLevelUpdateDto;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.google.common.collect.Sets;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 迁移原/baseTreeController/productLevelTree接口
 * 产品层级控制器
 *
 * @author bao hongbin
 * @date 2021-09-27 14:13:56
 */

@Api(tags = "产品层级：ProductLevelVo：与产品层级相关的内容")
@Slf4j
@RestController
@RequestMapping("/v1/productLevel/productLevel")
public class ProductLevelVoController {

  @Autowired(required = false)
  private ProductLevelVoService productLevelVoService;

  /**
   * 多条件分页查询
   *
   * @return
   */
  @ApiOperation(value = "多条件分页查询(可适用于列表分页查询和公用分页弹框)", notes = "分页参数为page和size，page从0开始，size默认50;")
  @GetMapping("findByConditions")
  public Result<Page<ProductLevelVo>> findByConditions(@PageableDefault(50) Pageable pageable,
                                                       @ApiParam(name = "productLevelPaginationDto", value = "分页Dto") ProductLevelPaginationDto productLevelPaginationDto) {
    try {
      Page<ProductLevelVo> result = this.productLevelVoService.findByConditions(pageable, productLevelPaginationDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 按照ProductLevel实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   *
   * @param id 主键
   */
  @ApiOperation(value = "按照ProductLevel实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @GetMapping("/findDetailsById")
  public Result<ProductLevelVo> findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      ProductLevelVo result = this.productLevelVoService.findDetailsById(id);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 相关的创建过程，http接口。请注意该创建过程除了可以创建模型中的基本信息以外，还可以对模型中属于OneToMany关联的明细信息一同进行创建注意：
   * 基于模型的创建操作传入的JSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   */
  @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建模型中的基本信息以外，" +
      "还可以对模型中属于OneToMany关联的明细信息一同进行创建注意：基于模型的创建操作传入的JSON对象，" +
      "其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PostMapping(value = "")
  public Result<ProductLevelVo> create(
      @RequestBody @ApiParam(name = "productLevelCreateDto",
          value = "相关的创建过程，http接口。请注意该创建过程除了可以创建模型中的基本信息以外，" +
              "还可以对模型中属于OneToMany关联的明细信息一同进行创建注意：基于模型的创建操作传入的JSON对象，" +
              "其主键信息不能有值，服务端将会自动为其赋予相关值。" +
              "另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") ProductLevelCreateDto productLevelCreateDto) {
    try {
      ProductLevelVo current = this.productLevelVoService.create(productLevelCreateDto);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，
   * 包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，
   * 虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（的修改操作传入的SON对象，
   * 其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   */
  @ApiOperation(value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，" +
      "包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，" +
      "但需要开发人员自行在Service层完善其更新过程注意：基于模型的修改操作传入的JSON对象，其主键信息必须有值，" +
      "服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PatchMapping(value = "")
  public Result<ProductLevelVo> update(
      @RequestBody @ApiParam(name = "productLevelUpdateDto",
          value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，" +
              "包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，" +
              "虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型的修改操作传入的JSON对象，" +
              "其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，" +
              "系统将返回该对象的基本信息（不包括任何关联信息）") ProductLevelUpdateDto productLevelUpdateDto) {
    try {
      ProductLevelVo current = this.productLevelVoService.update(productLevelUpdateDto);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 启用
   */
  @ApiOperation(value = "启用")
  @PatchMapping("/enable")
  public Result<?> enable(@RequestBody List<String> ids) {
    try {
      this.productLevelVoService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 禁用
   */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody List<String> ids) {
    try {
      this.productLevelVoService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 删除
   */
  @ApiOperation(value = "删除")
  @DeleteMapping("/delete")
  public Result<?> delete(@RequestParam("ids") List<String> ids) {
    try {
      this.productLevelVoService.deleteBatch(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation("重置降维编码")
  @PatchMapping("/updateRuleCode")
  public Result<?> updateRuleCode() {
    try {
      this.productLevelVoService.updateRuleCode();
      return Result.ok("重置降维编码成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据产品层级编码查询该产品层级及全部下级层级的编码
   *
   * @param productLevelCode 产品层级编码
   */
  @ApiOperation(value = "根据产品层级编码查询该产品层级及全部下级层级的编码")
  @GetMapping("/findCurAndChildrenCodesByCode")
  public Result<List<String>> findCurAndChildrenCodesByCode(
      @RequestParam("productLevelCode") @ApiParam("产品层级编码") String productLevelCode) {
    try {
      List<String> result = this.productLevelVoService.findCurAndChildrenCodesByCodes(Sets.newHashSet(productLevelCode));
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
  
  /**
   * 获取所有的产品层级
   */
  @ApiOperation(value = "获取所有的产品层级")
  @GetMapping("/findAll")
  public Result<List<ProductLevelVo>> findAll() {
    try {
      List<ProductLevelVo> result = this.productLevelVoService.findAll();
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

}
