package com.biz.crm.mdm.business.product.level.local.repository;
/**
 * Created by Bao Hongbin on 2021-10-09 10:14.
 */

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.dto.TreeDto;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.vo.LazyTreeVo;
import com.biz.crm.mdm.business.product.level.local.entity.ProductLevel;
import com.biz.crm.mdm.business.product.level.local.mapper.ProductLevelMapper;
import com.biz.crm.mdm.business.product.level.sdk.dto.ProductLevelPaginationDto;
import com.biz.crm.mdm.business.product.level.sdk.enums.ProductLevelEnum;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import org.springframework.util.CollectionUtils;

/**
 * @program: crm
 * @description: 产品层级的数据库访问类
 * @author: Bao Hongbin
 * @create: 2021-10-09 10:14
 **/
@Component
public class ProductLevelRepository extends ServiceImpl<ProductLevelMapper, ProductLevel> {
  /**
   * 分页条件查询
   *
   * @param pageable
   * @param productLevelPaginationDto
   * @return
   */
  public Page<ProductLevelVo> findByConditions(Pageable pageable, ProductLevelPaginationDto productLevelPaginationDto) {
    Page<ProductLevelVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, productLevelPaginationDto, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 通过id查询详情
   *
   * @param id
   * @param tenantCode
   * @return
   */
  public ProductLevel findDetailsById(String id, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .eq(ProductLevel::getId, id).one();
  }

  /**
   * 通过code查询详情
   *
   * @param code
   * @param tenantCode
   * @return
   */
  public ProductLevel findDetailsByCode(String code, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .eq(ProductLevel::getProductLevelCode, code).one();
  }

  /**
   * 通过code集合查询详情
   *
   * @param codeSet
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findDetailsByCodes(Set<String> codeSet, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .in(ProductLevel::getProductLevelCode, codeSet).list();
  }

  /**
   * 通过code查询list
   *
   * @param code
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findListByCode(String code, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .eq(ProductLevel::getProductLevelCode, code).list();
  }

  /**
   * 根据父级Code查询子集
   *
   * @param parentCode
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findChildrenListByParentCode(String parentCode, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getTenantCode, tenantCode)
        .isNull(!StringUtils.isNotEmpty(parentCode), ProductLevel::getParentCode)
        .eq(StringUtils.isNotEmpty(parentCode), ProductLevel::getParentCode, parentCode).list();
  }

  /**
   * 通过id集合查询产品层级数据
   *
   * @param ids
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findListByIds(List<String> ids, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .in(ProductLevel::getId, ids).list();
  }

  /**
   * 通过RuleCode集合查询产品层级数据
   *
   * @param ruleCodes
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findListByRuleCodes(Set<String> ruleCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .in(ProductLevel::getRuleCode, ruleCodes).list();
  }

  /**
   * 通过启用状态和RuleCodes模糊查询自身及子集
   *
   * @param ruleCodes
   * @param enableStatus
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findCurAndChildrenByRuleCodeList(List<String> ruleCodes, String enableStatus, String tenantCode) {
    return this.baseMapper.findCurAndChildrenByRuleCodeList(ruleCodes, enableStatus, tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 根据父级Codes查询所有子集
   *
   * @param parentCodes
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findChildrenListByParentCodes(List<String> parentCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .in(ProductLevel::getParentCode, parentCodes).list();
  }

  /**
   * 查找parentCode不为空但找不到对应上级的数据,设置parentCode为null
   *
   * @return
   */
  public void updateOrphanParentCodeNull(String tenantCode) {
    this.baseMapper.updateOrphanParentCodeNull(tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 查询所有没有父级编码的子集
   *
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findListWithoutParentCode(String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .isNull(ProductLevel::getParentCode)
        .list();
  }

  /**
   * 通过code集合查询产品层级数据
   *
   * @param codes
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findListByCodes(List<String> codes, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .in(ProductLevel::getProductLevelCode, codes).list();
  }

  /**
   * 根据父级Code查询所有子集
   *
   * @param ruleCode
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findCurAndChildrenByRuleCode(String ruleCode, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .likeRight(ProductLevel::getRuleCode, ruleCode)
        .list();
  }

  /**
   * 查询所有产品层级
   *
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findAll(String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .list();
  }

  /**
   * 通过名称模糊查询产品层级list
   *
   * @param enableStatus
   * @param name
   * @param tenantCode
   * @return
   */
  public List<ProductLevel> findListLikeName(String enableStatus, String name, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .eq(StringUtils.isNotEmpty(enableStatus), ProductLevel::getEnableStatus, enableStatus)
        .like(StringUtils.isNotEmpty(name), ProductLevel::getProductLevelName, name)
        .list();
  }

  public List<ProductLevel> findListLikeName( TreeDto dto){
    dto = Optional.ofNullable(dto).orElse(new TreeDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    return this.baseMapper.findListLikeName(dto);

  }

  /**
   * 查询产品层级懒加载数据
   *
   * @param enableStatus    启用状态
   * @param topOnly         传true只查第一层
   * @param parentCode      只查询该编码下一级
   * @param codeList        只查询这些编码
   * @param ruleCodeList    只查询这些降维编码
   * @param excludeRuleCode 排除这个降维编码的下级
   * @param tenantCode
   * @return
   */
  public List<LazyTreeVo> findLazyTreeList(String enableStatus,
                                           Boolean topOnly,
                                           String parentCode,
                                           List<String> codeList,
                                           List<String> ruleCodeList,
                                           String excludeRuleCode,
                                           String tenantCode) {
    return this.baseMapper.findLazyTreeList(enableStatus, topOnly, parentCode, codeList, ruleCodeList, excludeRuleCode, tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 手动设置父级编码为空
   *
   * @param id
   */
  public void setParentCodeNull(String id) {
//    this.lambdaUpdate().set(ProductLevel::getParentCode, null)
//        .eq(ProductLevel::getId, id)
//        .update();
    this.baseMapper.setParentCodeNull(id);
  }

  /**
   * 手动设置父级编码，规则编码为空
   *
   * @param id
   */
  public void setParentCodeAndRuleCodeNull(String id) {
    this.lambdaUpdate()
        .set(ProductLevel::getParentCode, null)
        .set(ProductLevel::getRuleCode, "")
        .eq(ProductLevel::getId, id)
        .update();
  }

  /**
   * 根据产品层级类型获取产品层级信息
   *
   * @param productLevelType 产品层级类型
   * @param tenantCode 租户编码
   * @return 产品层级信息
   */
  public List<ProductLevel> findByProductLevelType(String productLevelType, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProductLevel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ProductLevel::getTenantCode, tenantCode)
        .eq(ProductLevel::getProductLevelType, productLevelType)
        .list();
  }

  /**
   * 根据编码规则获取当前及子级的产品层级编码集合
   *
   * @param ruleCodeSet
   * @param tenantCode
   * @return
   */
  public List<String> findCodeByCurAndChildrenByRuleCodes(Set<String> ruleCodeSet, String tenantCode) {
    if(CollectionUtils.isEmpty(ruleCodeSet)) {
      return Lists.newLinkedList();
    }
    return this.baseMapper.findCodeByCurAndChildrenByRuleCodes(ruleCodeSet, tenantCode, DelFlagStatusEnum.NORMAL.getCode());
  }
}
