package com.biz.crm.mdm.business.product.level.local.service.helper;
/**
 * Created by Bao Hongbin on 2021-10-12 15:32.
 */

import com.biz.crm.mdm.business.product.level.local.entity.ProductLevel;
import com.biz.crm.mdm.business.product.level.local.repository.ProductLevelRepository;
import com.biz.crm.mdm.business.product.level.sdk.dto.ProductLevelCreateDto;
import com.biz.crm.mdm.business.product.level.sdk.dto.ProductLevelUpdateDto;
import com.biz.crm.mdm.business.product.level.sdk.event.ProductLevelEventListener;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;

/**
 * @program: crm
 * @description: 产品层级服务助手
 * @author: Bao Hongbin
 * @create: 2021-10-12 15:32
 **/
@Component
@Slf4j
public class ProductLevelServiceHelper {
  @Autowired(required = false)
  @Lazy
  private List<ProductLevelEventListener> productLevelEventListeners;
  @Autowired(required = false)
  private ProductLevelRepository productLevelRepository;

  /**
   * 验证创建请求参数
   *
   * @param productLevelCreateDto
   */
  public void createValidation(ProductLevelCreateDto productLevelCreateDto) {
    Validate.notNull(productLevelCreateDto, "进行当前操作时，信息对象必须传入!");
    log.info("新增产品层级请求dto：{}", JsonUtils.obj2JsonString(productLevelCreateDto));
    if (StringUtils.isNotEmpty(productLevelCreateDto.getProductLevelCode())) {
      List<ProductLevel> productLevels =
          productLevelRepository.findListByCode(productLevelCreateDto.getProductLevelCode());
      Validate.isTrue(CollectionUtils.isEmpty(productLevels),
          "编码[" + productLevelCreateDto.getProductLevelCode() + "]已存在");
    }
    if (StringUtils.isNotEmpty(productLevelCreateDto.getParentCode())) {
      ProductLevel parent = productLevelRepository.findDetailsByCode(productLevelCreateDto.getParentCode());
      Validate.notNull(parent, "上级编码无效");
    }
    Validate.notBlank(productLevelCreateDto.getProductLevelName(), "新增信息时，产品层级名称不能为空！");
    Validate.notNull(productLevelCreateDto.getProductLevelType(), "新增信息时，产品层级类型不能为空！");
    Validate.isTrue(productLevelCreateDto.getProductLevelName().length() < 64,
        "产品层级名称信息，在进行新增时填入值超过了限定长度(64)，请检查!");
  }

  /**
   * 推送产品层级创建事件
   *
   * @param productLevelVos
   */
  public void sendCreateEvent(List<ProductLevelVo> productLevelVos) {
    if (Objects.nonNull(this.productLevelEventListeners)) {
      for (ProductLevelEventListener listener : this.productLevelEventListeners) {
        listener.onBatchCreate(productLevelVos);
      }
    }
  }

  /**
   * 验证更新请求参数
   *
   * @param productLevelUpdateDto
   */
  public void updateValidation(ProductLevelUpdateDto productLevelUpdateDto) {
    Validate.notNull(productLevelUpdateDto, "进行当前操作时，信息对象必须传入!");
    log.info("编辑产品层级请求dto：{}", JsonUtils.obj2JsonString(productLevelUpdateDto));
    Validate.notBlank(productLevelUpdateDto.getId(), "修改信息时，产品层级ID不能为空！");
    ProductLevel productLevel = productLevelRepository.findDetailsById(productLevelUpdateDto.getId());
    Validate.notNull(productLevel, "无效的业务技术编号信息");
    Validate.isTrue(
        productLevel.getProductLevelCode().equals(productLevelUpdateDto.getProductLevelCode()),
        "产品层级编码不能修改");
    if (StringUtils.isNotEmpty(productLevelUpdateDto.getParentCode())) {
      ProductLevel parent = productLevelRepository.findDetailsByCode(productLevelUpdateDto.getParentCode());
      Validate.notNull(parent, "上级产品层级不存在");
      Validate.isTrue(!parent.getId().equals(productLevel.getId()), "上级产品层级不能是自己");
      Validate.isTrue(!parent.getRuleCode().startsWith(productLevel.getRuleCode()),
          "上级产品层级不能是当前层级的下级");
    }
    Validate.notBlank(productLevelUpdateDto.getProductLevelName(), "修改信息时，产品层级名称不能为空！");
    Validate.notNull(productLevelUpdateDto.getProductLevelType(), "修改信息时，产品层级类型不能为空！");
    Validate.isTrue(productLevelUpdateDto.getProductLevelName().length() < 64,
        "产品层级名称信息，在进行修改时填入值超过了限定长度(64)，请检查!");
  }

  /**
   * 推送产品层级创建事件
   *
   * @param productLevelVos
   */
  public void sendUpdateEvent(List<ProductLevelVo> productLevelVos) {
    if (Objects.nonNull(this.productLevelEventListeners)) {
      for (ProductLevelEventListener listener : this.productLevelEventListeners) {
        listener.onBatchUpdate(productLevelVos);
      }
    }
  }

  /**
   * 推送产品层级删除事件
   *
   * @param productLevelVos
   */
  public void sendDeleteEvent(List<ProductLevelVo> productLevelVos) {
    if (Objects.nonNull(this.productLevelEventListeners)) {
      for (ProductLevelEventListener listener : this.productLevelEventListeners) {
        listener.onBatchDelete(productLevelVos);
      }
    }
  }

  /**
   * 推送产品层级启用事件
   *
   * @param productLevelVos
   */
  public void sendEnableEvent(List<ProductLevelVo> productLevelVos) {
    if (Objects.nonNull(this.productLevelEventListeners)) {
      for (ProductLevelEventListener listener : this.productLevelEventListeners) {
        listener.onBatchEnable(productLevelVos);
      }
    }
  }


  /**
   * 推送产品层级禁用事件
   *
   * @param productLevelVos
   */
  public void sendDisableEvent(List<ProductLevelVo> productLevelVos) {
    if (Objects.nonNull(this.productLevelEventListeners)) {
      for (ProductLevelEventListener listener : this.productLevelEventListeners) {
        listener.onBatchDisable(productLevelVos);
      }
    }
  }
}
