package com.biz.crm.mdm.business.product.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.product.local.entity.Product;
import com.biz.crm.mdm.business.product.local.mapper.ProductMapper;
import com.biz.crm.mdm.business.product.sdk.dto.ContractProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductPaginationDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.List;
import java.util.Set;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

/**
 * mdm_product 商品主数据(repository)
 *
 * @author sunx
 * @since 2021-10-13 15:38:57
 */
@Component
public class ProductRepository extends ServiceImpl<ProductMapper, Product> {

  private static final Integer SIZE = 1000;

  /**
   * 分页
   *
   * @param page
   * @param dto
   * @return
   */
  public Page<Product> findByConditions(Page<Product> page, ProductPaginationDto dto) {
    dto.setTenantCode(TenantUtils.getTenantCode());
    return baseMapper.findByConditions(page, dto);
  }

  public Product findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 根据编码集合获取商品集合
   *
   * @param productCodes
   * @return
   */
  public List<Product> findByProductCodes(List<String> productCodes) {
    return this.lambdaQuery()
        .eq(Product::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Product::getTenantCode, TenantUtils.getTenantCode())
        .in(Product::getProductCode, productCodes)
        .list();
  }

  /**
   * 根据id或编码集合获取商品集合
   *
   * @param ids
   * @param productCodes
   * @return
   */
  public List<Product> findByIdsOrProductCodes(List<String> ids, List<String> productCodes) {
    return this.lambdaQuery()
        .eq(Product::getTenantCode, TenantUtils.getTenantCode())
        .and(
            i ->
                i.in(CollectionUtils.isNotEmpty(ids), Product::getId, ids)
                    .or()
                    .in(
                        CollectionUtils.isNotEmpty(productCodes),
                        Product::getProductCode,
                        productCodes))
        .list();
  }

  /**
   * 根据id集合 更新组织启用/禁用状态
   *
   * @param ids
   * @param enable
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    this.lambdaUpdate()
        .in(Product::getId, ids)
        .set(Product::getEnableStatus, enable.getCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(Product::getId, ids)
        .set(Product::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 更新上下加状态
   *
   * @param ids
   * @param code
   */
  public void updateIsShelfByIds(List<String> ids, String code) {
    this.lambdaUpdate().in(Product::getId, ids).set(Product::getIsShelf, code).update();
  }

  /**
   * 根据商品编码更新上下加状态
   *
   * @param productCodeList
   * @param code
   */
  public void updateIsShelfByByProductCodes(List<String> productCodeList, String code) {
    this.lambdaUpdate()
        .in(Product::getProductCode, productCodeList)
        .set(Product::getIsShelf, code)
        .update();
  }

  /**
   * 根据商品层级编码或删除标获取匹配商品数量
   *
   * @param productLevelCodes
   * @param delFlag
   * @return
   */
  public Integer countByProductLevelCodesAndDelFlag(
      List<String> productLevelCodes, String delFlag) {
    return lambdaQuery()
        .eq(StringUtils.isNotBlank(delFlag), Product::getDelFlag, delFlag)
        .eq(Product::getTenantCode, TenantUtils.getTenantCode())
        .in(
            CollectionUtils.isNotEmpty(productLevelCodes),
            Product::getProductLevelCode,
            productLevelCodes)
        .count();
  }

  public List<ProductVo> findByProductLevelCodes(List<String> productLevelCodeList) {
    return this.baseMapper.findByProductLevelCodes(
        productLevelCodeList, TenantUtils.getTenantCode(),DelFlagStatusEnum.NORMAL.getCode());
  }

  /**
   * 获取商品对应的商品层级编码集合
   *
   * @param productCodeSet
   * @return
   */
  public Set<String> findProductLevelCodeSetByProductCodes(Set<String> productCodeSet) {
    return this.baseMapper.findProductLevelCodeSetByProductCodes(
        productCodeSet, TenantUtils.getTenantCode());
  }

  /**
   * 通过商品编码获取
   *
   * @param code
   * @param delFlag
   * @return
   */
  public Product findByProductCode(String code, String delFlag) {
    return this.lambdaQuery()
        .eq(Product::getProductCode, code)
        .eq(Product::getDelFlag, delFlag)
        .one();
  }

  /**
   * 根据ProductQueryDto获取商品主信息
   *
   * @param dto
   * @return
   */
  public List<Product> findByProductQueryDto(ProductQueryDto dto) {
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<Product> list = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getProductCodeList())) {
      for (List<String> item :
          Lists.partition(Lists.newArrayList(dto.getProductCodeList()), SIZE)) {
        dto.setProductCodeList(item);
        List<Product> cur = this.baseMapper.findByProductQueryDto(dto);
        if (CollectionUtils.isNotEmpty(cur)) {
          list.addAll(cur);
        }
      }
    } else {
      list = this.baseMapper.findByProductQueryDto(dto);
    }
    return list;
  }

  /**
   * 合同商品信息获取
   *
   * @param dto
   * @return
   */
  public List<Product> findByContractProductQueryDto(ContractProductQueryDto dto) {
    String tenantCode = TenantUtils.getTenantCode();
    String delFlag = DelFlagStatusEnum.NORMAL.getCode();
    String enableStatus = EnableStatusEnum.ENABLE.getCode();
    String shelfFlag = IsShelfEnum.UP.getCode();
    return this.baseMapper.findByContractProductQueryDto(
        dto, tenantCode, delFlag, enableStatus, shelfFlag);
  }
}
