package com.biz.crm.mdm.business.product.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import com.biz.crm.mdm.business.product.level.sdk.dto.RelateProductLevelCodeQueryDto;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.local.constant.ProductConstant;
import com.biz.crm.mdm.business.product.local.entity.Product;
import com.biz.crm.mdm.business.product.local.entity.ProductIntroduction;
import com.biz.crm.mdm.business.product.local.entity.ProductMaterial;
import com.biz.crm.mdm.business.product.local.entity.ProductMedia;
import com.biz.crm.mdm.business.product.local.repository.ProductRepository;
import com.biz.crm.mdm.business.product.local.service.ProductIntroductionService;
import com.biz.crm.mdm.business.product.local.service.ProductMaterialService;
import com.biz.crm.mdm.business.product.local.service.ProductMediaService;
import com.biz.crm.mdm.business.product.local.service.ProductService;
import com.biz.crm.mdm.business.product.sdk.dto.ProductDetailQueryDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductEventDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductMaterialDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductMediaDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductPaginationDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.sdk.enums.MediaTypeEnum;
import com.biz.crm.mdm.business.product.sdk.enums.ProductExtInfoTypeEnum;
import com.biz.crm.mdm.business.product.sdk.event.ProductEventListener;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.MaterialProductVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductIntroductionVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductMaterialVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductMediaVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 商品sdk实现类
 *
 * @author sunx
 * @date 2021/10/13
 */
@Service("productVoService")
public class ProductVoServiceImpl implements ProductVoService {

  @Autowired(required = false)
  private ProductService productService;

  @Autowired(required = false)
  private ProductMaterialService productMaterialService;

  @Autowired(required = false)
  private ProductMediaService productMediaService;

  @Autowired(required = false)
  private ProductIntroductionService productIntroductionService;

  @Autowired(required = false)
  private ProductLevelVoSdkService productLevelVoSdkService;

  @Autowired(required = false)
  private MaterialVoService materialVoService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private ProductRepository productRepository;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Override
  @Transactional
  public ProductVo create(ProductDto dto) {
    this.createValidation(dto);
    Product product = this.buildProductByDto(dto);
    dto.setProductCode(product.getProductCode());
    this.bindProductExtInfo(dto);
    this.productService.create(product);
    ProductEventDto productEventDto = new ProductEventDto();
    productEventDto.setOriginal(null);
    ProductVo productVo =
        this.nebulaToolkitService.copyObjectByBlankList(
            product, ProductVo.class, HashSet.class, LinkedList.class);
    productEventDto.setNewest(productVo);
    // 商品创建事件
    SerializableBiConsumer<ProductEventListener, ProductEventDto> onCreate =
        ProductEventListener::onCreate;
    this.nebulaNetEventClient.publish(productEventDto, ProductEventListener.class, onCreate);
    return this.buildByDtoAndProduct(dto, product);
  }

  @Override
  @Transactional
  public ProductVo update(ProductDto dto) {
    this.updateValidation(dto);
    List<ProductVo> current =
        this.findDetailsByIdsOrProductCodes(null, Collections.singletonList(dto.getProductCode()));
    ProductVo current2 = current.stream().findFirst().orElse(null);
    Validate.notNull(current2, "未获取到修改的商品信息");
    ProductVo productVo =
        this.nebulaToolkitService.copyObjectByBlankList(
            current2, ProductVo.class, HashSet.class, LinkedList.class);
    Product product = this.buildProductByDto(dto);
    this.bindProductExtInfo(dto);
    this.productService.update(product);
    ProductEventDto productEventDto = new ProductEventDto();
    productEventDto.setOriginal(productVo);
    ProductVo productVo2 =
        this.nebulaToolkitService.copyObjectByBlankList(
            dto, ProductVo.class, HashSet.class, LinkedList.class);
    productEventDto.setNewest(productVo2);
    // 商品编辑事件
    SerializableBiConsumer<ProductEventListener, ProductEventDto> onUpdate =
        ProductEventListener::onUpdate;
    this.nebulaNetEventClient.publish(productEventDto, ProductEventListener.class, onUpdate);
    return this.buildByDtoAndProduct(dto, product);
  }

  @Override
  public List<ProductVo> findSelectByKeyword(String keyword) {
    Pageable pageable = PageRequest.of(0, 20);
    ProductPaginationDto dto = new ProductPaginationDto();
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    dto.setKeyword(keyword);
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<Product> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Product> pageResult = this.productRepository.findByConditions(page, dto);
    if (Objects.isNull(pageResult) || CollectionUtils.isEmpty(pageResult.getRecords())) {
      return Lists.newLinkedList();
    }
    List<ProductVo> list = Lists.newArrayList();
    for (Product item : pageResult.getRecords()) {
      ProductVo cur = new ProductVo();
      cur.setProductCode(item.getProductCode());
      cur.setProductName(item.getProductName());
      list.add(cur);
    }
    return list;
  }

  @Override
  public MaterialProductVo findMaterialProductVoByMaterialCode(String materialCode) {
    if (StringUtils.isBlank(materialCode)) {
      return null;
    }
    Set<String> productCodeSet =
        this.productMaterialService.findProductCodeByMaterialCode(materialCode);
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return null;
    }
    MaterialProductVo re = new MaterialProductVo();
    re.setProductCodeSet(productCodeSet);
    Set<String> productParentLevelCodeSet =
        this.findParentLevelCodeSetByProductCodes(productCodeSet);
    re.setProductLevelCodeSet(productParentLevelCodeSet);
    return re;
  }

  @Override
  public List<ProductVo> findByProductLevelCodes(List<String> productLevelCodeList) {
    if (CollectionUtils.isEmpty(productLevelCodeList)) {
      return Lists.newLinkedList();
    }
    return this.productRepository.findByProductLevelCodes(productLevelCodeList);
  }

  @Override
  public Set<String> findParentLevelCodeSetByProductCodes(Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return Sets.newHashSet();
    }
    Set<String> productLevelCodeSet =
        this.productService.findProductLevelCodeSetByProductCodes(productCodeSet);
    if (CollectionUtils.isEmpty(productLevelCodeSet)) {
      return Sets.newHashSet();
    }
    Map<String, List<ProductLevelVo>> map =
        this.productLevelVoSdkService.findCurAndParentByCodes(
            Lists.newArrayList(productLevelCodeSet));
    if (map.isEmpty()) {
      return Sets.newHashSet();
    }
    Set<String> re = Sets.newHashSet();
    for (Entry<String, List<ProductLevelVo>> item : map.entrySet()) {
      for (ProductLevelVo sub : item.getValue()) {
        re.add(sub.getProductLevelCode());
      }
    }
    return re;
  }

  @Override
  public List<ProductVo> findByProductQueryDto(ProductQueryDto dto) {
    Boolean f =
        Objects.isNull(dto)
            || (StringUtils.isAllBlank(
                    dto.getProductCode(),
                    dto.getProductName(),
                    dto.getProductName(),
                    dto.getIsShelf(),
                    dto.getEnableStatus())
                && CollectionUtils.isEmpty(dto.getProductLevelCodeList())
                && CollectionUtils.isEmpty(dto.getProductCodeList()));
    if (Boolean.TRUE.equals(f)) {
      return Lists.newLinkedList();
    }
    List<Product> list = this.productRepository.findByProductQueryDto(dto);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<ProductVo>)
        this.nebulaToolkitService.copyCollectionByBlankList(
            list, Product.class, ProductVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<ProductVo> findDetailsByIdsOrProductCodes(
      List<String> ids, List<String> productCodes) {
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(productCodes)) {
      return Lists.newLinkedList();
    }
    final ProductDetailQueryDto dto = new ProductDetailQueryDto();
    dto.setExInfoCodes(ProductExtInfoTypeEnum.findAllKey());
    if (CollectionUtils.isNotEmpty(ids)) {
      dto.setCodeQueryFlag(false);
      dto.setCodes(ids);
    } else if (CollectionUtils.isNotEmpty(productCodes)) {
      dto.setCodeQueryFlag(true);
      dto.setCodes(productCodes);
    }
    return this.findDetailsByProductDetailQueryDto(dto);
  }

  @Override
  public List<ProductVo> findMainDetailsByProductCodes(List<String> productCodeList) {
    if (CollectionUtils.isEmpty(productCodeList)) {
      return Lists.newLinkedList();
    }
    final ProductDetailQueryDto dto = new ProductDetailQueryDto();
    dto.setExInfoCodes(Sets.newHashSet(ProductExtInfoTypeEnum.PRODUCT_LEVEL.getDictCode()));
    dto.setCodeQueryFlag(true);
    dto.setCodes(productCodeList);
    return this.findDetailsByProductDetailQueryDto(dto);
  }

  @Override
  public List<ProductVo> findDetailsByProductDetailQueryDto(ProductDetailQueryDto dto) {
    if (Objects.isNull(dto) || CollectionUtils.isEmpty(dto.getCodes())) {
      return Lists.newLinkedList();
    }
    List<String> ids = Lists.newLinkedList();
    List<String> productCodes = Lists.newLinkedList();
    if (Boolean.TRUE.equals(dto.getCodeQueryFlag())) {
      productCodes = dto.getCodes();
    } else {
      ids = dto.getCodes();
    }
    List<Product> productList = productService.findDetailsByIdsOrProductCodes(ids, productCodes);
    if (CollectionUtils.isEmpty(productList)) {
      return Lists.newLinkedList();
    }
    List<ProductVo> list =
        (List<ProductVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                productList, Product.class, ProductVo.class, HashSet.class, ArrayList.class);
    Set<String> productCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
            .map(ProductVo::getProductCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(dto.getExInfoCodes()) || CollectionUtils.isEmpty(productCodeSet)) {
      return list;
    }
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.PRODUCT_LEVEL.getDictCode())) {
      this.findProductLevelInfo(list);
    }
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.MATERIAL.getDictCode())) {
      this.findProductMaterialInfo(list, productCodeSet);
    }
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.PICTURE.getDictCode())
        || dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.VIDEO.getDictCode())) {
      this.findProductMediaInfo(list, productCodeSet);
    }
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.INTRO.getDictCode())) {
      this.findProductIntroInfo(list, productCodeSet);
    }
    return list;
  }

  @Override
  public Map<String, String> findAllowSaleProductByProductLevelCodes(
      Set<String> productLevelCodes) {
    if (CollectionUtils.isEmpty(productLevelCodes)) {
      return Maps.newHashMap();
    }
    final RelateProductLevelCodeQueryDto queryDto = new RelateProductLevelCodeQueryDto();
    queryDto.setProductLevelCodeSet(productLevelCodes);
    queryDto.setSearchType(-1);
    final Map<String, String> productLevelRuleMap =
        this.productLevelVoSdkService.findByRelateProductLevelCodeQueryDto(queryDto);
    if (productLevelRuleMap.isEmpty()) {
      return Maps.newHashMap();
    }
    final ProductQueryDto productQueryDto = new ProductQueryDto();
    productQueryDto.setProductLevelCodeList(Lists.newArrayList(productLevelCodes));
    productQueryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    productQueryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    productQueryDto.setIsShelf(IsShelfEnum.UP.getCode());
    productQueryDto.setTenantCode(TenantUtils.getTenantCode());
    List<ProductVo> list = this.findByProductQueryDto(productQueryDto);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, String> map =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(a.getProductCode(), a.getProductLevelCode())
                        && productLevelRuleMap.keySet().contains(a.getProductLevelCode()))
            .collect(
                Collectors.toMap(
                    ProductVo::getProductCode, ProductVo::getProductLevelCode, (a, b) -> a));
    Map<String, String> re = Maps.newHashMap();
    for (Entry<String, String> item : map.entrySet()) {
      final String s = productLevelRuleMap.get(item.getValue());
      if (StringUtils.isBlank(s)) {
        continue;
      }
      re.put(item.getKey(), s);
    }
    return re;
  }

  @Override
  public Page<ProductVo> findByConditions(Pageable pageable, ProductPaginationDto dto) {
    Page<Product> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Product> pageResult = this.productRepository.findByConditions(page, dto);
    Page<ProductVo> productVoPage = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    if (CollectionUtils.isEmpty(pageResult.getRecords())) {
      return productVoPage;
    }
    List<ProductVo> productVos = (List<ProductVo>) this.nebulaToolkitService.copyCollectionByBlankList(pageResult.getRecords(), Product.class, ProductVo.class, HashSet.class, ArrayList.class);
    productVoPage.setRecords(productVos);
    return productVoPage;
  }

  /**
   * 保存商品关联的物料图片视频以及介绍信息
   *
   * @param dto
   */
  private void bindProductExtInfo(ProductDto dto) {
    /*
     * 保存商品关联的物料图片视频以及介绍信息：
     * 1、保存关联物料
     * 2、保存图片视频信息
     * 3、保存商品介绍信息
     * */
    Validate.notNull(dto, "商品信息缺失");
    List<ProductMaterial> productMaterialList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getMaterialList())) {
      productMaterialList =
          (List<ProductMaterial>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getMaterialList(),
                  ProductMaterialDto.class,
                  ProductMaterial.class,
                  HashSet.class,
                  ArrayList.class);
      productMaterialList.forEach(a -> a.setProductCode(dto.getProductCode()));
    }
    productMaterialService.saveBatch(productMaterialList, dto.getProductCode());

    List<ProductMedia> productMediaList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getPictureMediaList())) {
      productMediaList.addAll(
          this.nebulaToolkitService.copyCollectionByWhiteList(
              dto.getPictureMediaList(),
              ProductMediaDto.class,
              ProductMedia.class,
              HashSet.class,
              ArrayList.class));
      productMediaList.forEach(a -> a.setProductCode(dto.getProductCode()));
    }

    if (CollectionUtils.isNotEmpty(dto.getVideoMediaList())) {
      productMediaList.addAll(
          this.nebulaToolkitService.copyCollectionByWhiteList(
              dto.getVideoMediaList(),
              ProductMediaDto.class,
              ProductMedia.class,
              HashSet.class,
              ArrayList.class));
      productMediaList.forEach(a -> a.setProductCode(dto.getProductCode()));
    }
    productMediaService.saveBatch(productMediaList, dto.getProductCode());

    ProductIntroduction productIntroduction = null;
    if (Objects.nonNull(dto.getIntroduction())) {
      productIntroduction =
          this.nebulaToolkitService.copyObjectByWhiteList(
              dto.getIntroduction(), ProductIntroduction.class, HashSet.class, ArrayList.class);
      productIntroduction.setProductCode(dto.getProductCode());
    }
    productIntroductionService.saveIntroduction(productIntroduction, dto.getProductCode());
  }

  /**
   * 构建需要落库的商品信息
   *
   * @param dto
   * @return
   */
  private Product buildProductByDto(ProductDto dto) {
    Product product =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, Product.class, HashSet.class, ArrayList.class);
    product.setMaintenancePicture(CollectionUtils.isNotEmpty(dto.getPictureMediaList()));
    product.setMaintenanceIntroduction(Objects.nonNull(dto.getIntroduction()));
    if (StringUtils.isBlank(product.getProductCode())) {
      String productCode =
          this.generateCodeService
              .generateCode(ProductConstant.PRODUCT_CODE, ProductConstant.PRODUCT_CODE_LENGTH)
              .get(0);
      product.setProductCode(productCode);
    }
    if (CollectionUtils.isEmpty(dto.getPictureMediaList())) {
      return product;
    }
    Optional<ProductMediaDto> first = dto.getPictureMediaList().stream().findFirst();
    if (first.isPresent()) {
      product.setPrimaryPictureUrl(first.get().getUrlAddress());
    }
    return product;
  }

  /**
   * dto转vo
   *
   * @param dto
   * @param product
   * @return
   */
  private ProductVo buildByDtoAndProduct(ProductDto dto, Product product) {
    ProductVo vo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            product, ProductVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(dto.getMaterialList())) {
      vo.setMaterialList(
          (List<ProductMaterialVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getMaterialList(),
                  ProductMaterialDto.class,
                  ProductMaterialVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getPictureMediaList())) {
      vo.setPictureMediaList(
          (List<ProductMediaVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getVideoMediaList(),
                  ProductMediaDto.class,
                  ProductMediaVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getVideoMediaList())) {
      vo.setVideoMediaList(
          (List<ProductMediaVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getVideoMediaList(),
                  ProductMediaDto.class,
                  ProductMediaVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (Objects.nonNull(dto.getIntroduction())) {
      vo.setIntroduction(
          this.nebulaToolkitService.copyObjectByWhiteList(
              dto.getIntroduction(), ProductIntroductionVo.class, HashSet.class, ArrayList.class));
    }
    return vo;
  }

  /**
   * 获取商品的层级信息
   *
   * @param list
   */
  private void findProductLevelInfo(List<ProductVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    Set<String> productLevelCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getProductLevelCode()))
            .map(ProductVo::getProductLevelCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(productLevelCodeSet)) {
      return;
    }
    List<ProductLevelVo> productLevelVoList =
        this.productLevelVoSdkService.findListByCodes(Lists.newArrayList(productLevelCodeSet));
    // k-productLevelCode,v-productLevelName
    Map<String, String> mapLevel =
        productLevelVoList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getProductLevelCode(), a.getProductLevelName()))
            .collect(
                Collectors.toMap(
                    ProductLevelVo::getProductLevelCode,
                    ProductLevelVo::getProductLevelName,
                    (a, b) -> a));
    for (ProductVo item : list) {
      item.setProductLevelName(mapLevel.get(item.getProductLevelCode()));
    }
  }

  /**
   * 获取商品物料信息
   *
   * @param list
   * @param productCodeSet
   */
  private void findProductMaterialInfo(List<ProductVo> list, Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(productCodeSet)) {
      return;
    }
    List<ProductMaterial> materialList =
        productMaterialService.findByProductCodes(Lists.newArrayList(productCodeSet));
    if (CollectionUtils.isEmpty(materialList)) {
      return;
    }
    Set<String> materialCodes =
        materialList.stream().map(ProductMaterial::getMaterialCode).collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(materialCodes)) {
      return;
    }
    // 获取物料表信息
    List<MaterialVo> materialVoList = materialVoService.findDetailByMaterialCodes(materialCodes);
    if (CollectionUtils.isEmpty(materialVoList)) {
      return;
    }
    Map<String, MaterialVo> map =
        materialVoList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getMaterialCode()))
            .collect(
                Collectors.toMap(MaterialVo::getMaterialCode, Function.identity(), (a, b) -> a));
    List<ProductMaterialVo> voList =
        (List<ProductMaterialVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                materialList,
                ProductMaterial.class,
                ProductMaterialVo.class,
                HashSet.class,
                ArrayList.class);
    for (ProductMaterialVo v : voList) {
      if (!map.containsKey(v.getMaterialCode())) {
        continue;
      }
      MaterialVo materialVo = map.get(v.getMaterialCode());
      v.setProductLevelCode(materialVo.getProductLevelCode());
      v.setProductLevelName(materialVo.getProductLevelName());
      v.setMaterialType(materialVo.getMaterialType());
      v.setMaterialTypeName(materialVo.getMaterialTypeName());
      v.setUnitTypeCode(materialVo.getUnitTypeCode());
    }
    Map<String, List<ProductMaterialVo>> mapProductMaterial =
        voList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
            .collect(Collectors.groupingBy(ProductMaterialVo::getProductCode));
    for (ProductVo item : list) {
      item.setMaterialList(mapProductMaterial.get(item.getProductCode()));
    }
  }

  /**
   * 获取商品图片视频信息
   *
   * @param list
   * @param productCodeSet
   */
  private void findProductMediaInfo(List<ProductVo> list, Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(productCodeSet)) {
      return;
    }
    List<ProductMedia> mediaList =
        productMediaService.findByProductCodes(Lists.newArrayList(productCodeSet));

    if (CollectionUtils.isEmpty(mediaList)) {
      return;
    }
    List<ProductMediaVo> voList =
        (List<ProductMediaVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                mediaList,
                ProductMedia.class,
                ProductMediaVo.class,
                HashSet.class,
                ArrayList.class);
    Map<String, List<ProductMediaVo>> map =
        voList.stream().collect(Collectors.groupingBy(ProductMediaVo::getType));
    Map<String, List<ProductMediaVo>> mapPictureMedia =
        map.getOrDefault(MediaTypeEnum.PICTURE.getCode(), Lists.newArrayList()).stream()
            .collect(Collectors.groupingBy(ProductMediaVo::getProductCode));
    Map<String, List<ProductMediaVo>> mapVideoMedia =
        map.getOrDefault(MediaTypeEnum.VIDEO.getCode(), Lists.newArrayList()).stream()
            .collect(Collectors.groupingBy(ProductMediaVo::getProductCode));
    for (ProductVo item : list) {
      item.setPictureMediaList(mapPictureMedia.get(item.getProductCode()));
      item.setVideoMediaList(mapVideoMedia.get(item.getProductCode()));
    }
  }

  /**
   * 获取商品详情信息
   *
   * @param list
   * @param productCodeSet
   */
  private void findProductIntroInfo(List<ProductVo> list, Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(productCodeSet)) {
      return;
    }
    List<ProductIntroduction> introductionList =
        productIntroductionService.findByProductCodes(Lists.newArrayList(productCodeSet));
    if (CollectionUtils.isEmpty(introductionList)) {
      return;
    }
    List<ProductIntroductionVo> voList =
        (List<ProductIntroductionVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                introductionList,
                ProductIntroduction.class,
                ProductIntroductionVo.class,
                HashSet.class,
                ArrayList.class);
    Map<String, ProductIntroductionVo> map =
        voList.stream()
            .collect(
                Collectors.toMap(
                    ProductIntroductionVo::getProductCode, Function.identity(), (a, b) -> a));
    for (ProductVo item : list) {
      item.setIntroduction(map.get(item.getProductCode()));
    }
  }

  private void createValidation(ProductDto dto) {
    this.validation(dto);
  }

  private void updateValidation(ProductDto dto) {
    Validate.notNull(dto, "商品信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(dto.getId()), "商品id不能为空");
    this.validation(dto);
  }

  private void validation(ProductDto dto) {
    Validate.notNull(dto, "商品信息缺失");
    // 商品编码前端如果不填，后台自动生成
    if (StringUtils.isNotBlank(dto.getProductCode())) {
      Product byProductCode = productService.findByProductCode(dto.getProductCode());
      Validate.isTrue(
          Objects.isNull(byProductCode) ||
              (StringUtils.isNotBlank(dto.getId())? dto.getId().equals(byProductCode.getId()):Objects.isNull(byProductCode)), "商品编码重复");
    }
    Validate.notBlank(dto.getProductType(), "商品类型不能为空");
    Validate.notBlank(dto.getProductName(), "商品名称不能为空");
    Validate.notBlank(dto.getIsShelf(), "上下架不能为空");
    Validate.notNull(dto.getBeginDateTime(), "开始时间不能为空");
    Validate.notNull(dto.getEndDateTime(), "结束时间不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getMaterialList()), "物料信息不能为空");
  }
}
