package com.biz.crm.mdm.business.product.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.product.local.entity.Product;
import com.biz.crm.mdm.business.product.local.mapper.ProductMapper;
import com.biz.crm.mdm.business.product.sdk.dto.ContractProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.dto.DisplayProductDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductPaginationDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collector;
import java.util.stream.Collectors;

/**
 * mdm_product 商品主数据(repository)
 *
 * @author sunx
 * @since 2021-10-13 15:38:57
 */
@Component
public class ProductRepository extends ServiceImpl<ProductMapper, Product> {

    private static final Integer SIZE = 1000;

    /**
     * 分页
     *
     * @param page
     * @param dto
     * @return
     */
    public Page<Product> findByConditions(Page<Product> page, ProductPaginationDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        return baseMapper.findByConditions(page, dto);
    }

    /**
     * 分页
     *
     * @param page
     * @param dto
     * @return
     */
    public Page<ProductVo> findVoByConditions(Page<ProductVo> page, ProductPaginationDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        return baseMapper.findVoByConditions(page, dto);
    }

    public Product findById(String id) {
        // return this.baseMapper.selectById(id);
        return this.lambdaQuery().eq(Product::getId, id).eq(Product::getTenantCode, TenantUtils.getTenantCode()).one();
    }

    /**
     * 根据编码集合获取商品集合
     *
     * @param productCodes
     * @return
     */
    public List<Product> findByProductCodes(List<String> productCodes) {
        return this.lambdaQuery().eq(Product::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(Product::getTenantCode, TenantUtils.getTenantCode()).in(Product::getProductCode, productCodes).list();
    }

    /**
     * 根据id或编码集合获取商品集合
     *
     * @param ids
     * @param productCodes
     * @return
     */
    public List<Product> findByIdsOrProductCodes(List<String> ids, List<String> productCodes) {
        return this.lambdaQuery().eq(Product::getTenantCode, TenantUtils.getTenantCode())
                .and(i -> i.in(CollectionUtils.isNotEmpty(ids), Product::getId, ids).or()
                        .in(CollectionUtils.isNotEmpty(productCodes), Product::getProductCode, productCodes))
                .list();
    }

    /**
     * 根据id集合 更新组织启用/禁用状态
     *
     * @param ids
     * @param enable
     */
    public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
        this.lambdaUpdate().in(Product::getId, ids).eq(Product::getTenantCode, TenantUtils.getTenantCode())
                .set(Product::getEnableStatus, enable.getCode()).update();
    }

    /**
     * 逻辑删除
     *
     * @param ids
     */
    public void updateDelFlagByIds(List<String> ids) {
        this.lambdaUpdate().in(Product::getId, ids).eq(Product::getTenantCode, TenantUtils.getTenantCode())
                .set(Product::getDelFlag, DelFlagStatusEnum.DELETE.getCode()).update();
    }

    /**
     * 更新上下加状态
     *
     * @param ids
     * @param code
     */
    public void updateIsShelfByIds(List<String> ids, String code) {
        this.lambdaUpdate().in(Product::getId, ids).eq(Product::getTenantCode, TenantUtils.getTenantCode())
                .set(Product::getIsShelf, code).update();
    }

    /**
     * 根据商品编码更新上下加状态
     *
     * @param productCodeList
     * @param code
     */
    public void updateIsShelfByByProductCodes(List<String> productCodeList, String code) {
        this.lambdaUpdate().in(Product::getProductCode, productCodeList)
                .eq(Product::getTenantCode, TenantUtils.getTenantCode()).set(Product::getIsShelf, code).update();
    }

    /**
     * 根据商品层级编码或删除标获取匹配商品数量
     *
     * @param productLevelCodes
     * @param delFlag
     * @return
     */
    public Integer countByProductLevelCodesAndDelFlag(List<String> productLevelCodes, String delFlag) {
        return lambdaQuery().eq(StringUtils.isNotBlank(delFlag), Product::getDelFlag, delFlag)
                .eq(Product::getTenantCode, TenantUtils.getTenantCode())
                .in(CollectionUtils.isNotEmpty(productLevelCodes), Product::getProductLevelCode, productLevelCodes).count();
    }

    public List<ProductVo> findByProductLevelCodes(List<String> productLevelCodeList) {
        return this.baseMapper.findByProductLevelCodes(productLevelCodeList, TenantUtils.getTenantCode(),
                DelFlagStatusEnum.NORMAL.getCode());
    }

    /**
     * 获取商品对应的商品层级编码集合
     *
     * @param productCodeSet
     * @return
     */
    public Set<String> findProductLevelCodeSetByProductCodes(Set<String> productCodeSet) {
        return this.baseMapper.findProductLevelCodeSetByProductCodes(productCodeSet, TenantUtils.getTenantCode());
    }

    /**
     * 通过商品编码获取
     *
     * @param code
     * @param delFlag
     * @return
     */
    public Product findByProductCode(String code, String delFlag) {
        return this.lambdaQuery().eq(Product::getProductCode, code)
                .eq(Product::getTenantCode, TenantUtils.getTenantCode()).eq(Product::getDelFlag, delFlag).one();
    }

    /**
     * 根据ProductQueryDto获取商品主信息
     *
     * @param dto
     * @return
     */
    public List<Product> findByProductQueryDto(ProductQueryDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        List<Product> list = Lists.newLinkedList();
        if (CollectionUtils.isNotEmpty(dto.getProductCodeList())) {
            for (List<String> item : Lists.partition(Lists.newArrayList(dto.getProductCodeList()), SIZE)) {
                dto.setProductCodeList(item);
                List<Product> cur = this.baseMapper.findByProductQueryDto(dto);
                if (CollectionUtils.isNotEmpty(cur)) {
                    list.addAll(cur);
                }
            }
        } else {
            list = this.baseMapper.findByProductQueryDto(dto);
        }
        return list;
    }

    /**
     * 合同商品信息获取
     *
     * @param dto
     * @return
     */
    public List<Product> findByContractProductQueryDto(ContractProductQueryDto dto) {
        String tenantCode = TenantUtils.getTenantCode();
        String delFlag = DelFlagStatusEnum.NORMAL.getCode();
        String enableStatus = EnableStatusEnum.ENABLE.getCode();
        String shelfFlag = IsShelfEnum.UP.getCode();
        return this.baseMapper.findByContractProductQueryDto(dto, tenantCode, delFlag, enableStatus, shelfFlag);
    }

    /**
     * 条件查询
     *
     * @param dto
     * @return java.util.List<com.biz.crm.mdm.business.product.local.entity.Product>
     * @author huojia
     * @date 2022/12/2 17:19
     **/
    public List<Product> queryCondition(ProductDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            return null;
        }
        if (StringUtils.isEmpty(dto.getProductBrandCode()) && StringUtils.isEmpty(dto.getProductCode())
                && StringUtils.isEmpty(dto.getProductLevelCode()) && StringUtils.isEmpty(dto.getProductCategoryCode())
                && CollectionUtils.isEmpty(dto.getExcludeMainCategoryList()) && CollectionUtils.isEmpty(dto.getIncludeMainCategoryList())) {
            return null;
        }
        return this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(dto.getProductBrandCode()), Product::getProductBrandCode,
                        dto.getProductBrandCode())
                .eq(StringUtils.isNotEmpty(dto.getProductCode()), Product::getProductCode, dto.getProductCode())
                .eq(StringUtils.isNotEmpty(dto.getProductCategoryCode()), Product::getProductCategoryCode,
                        dto.getProductCategoryCode())
                .eq(StringUtils.isNotEmpty(dto.getProductLevelCode()), Product::getProductLevelCode,
                        dto.getProductLevelCode())
                .notIn(!CollectionUtils.isEmpty(dto.getExcludeMainCategoryList()), Product::getMainCategory,
                        dto.getExcludeMainCategoryList())
                .in(!CollectionUtils.isEmpty(dto.getIncludeMainCategoryList()), Product::getMainCategory,
                        dto.getIncludeMainCategoryList())
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
    }

    /**
     * 根据商品编码集合获取对应的商品数据信息
     */
    public Product findByProductPlanningDetails(String productCode) {
        return this.lambdaQuery()
                .eq(Product::getProductCode, productCode)
                .eq(Product::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
    }

    /**
     * 根据物料批量查询
     *
     * @param productCodeList
     * @return java.util.List<com.biz.crm.mdm.business.position.local.entity.Product>
     * @author huojia
     * @date 2022/12/10 15:38
     **/
    public List<Product> findAllListByCodeList(List<String> productCodeList) {
        return this.lambdaQuery()
                .eq(Product::getTenantCode, TenantUtils.getTenantCode())
                .in(Product::getProductCode, productCodeList)
                .list();
    }


    /**
     * 根据商品编码查询商品信息
     *
     * @param unitTypeCode
     * @return java.util.List<com.biz.crm.mdm.business.position.local.entity.Product>
     * @author huojia
     * @date 2022/12/10 15:38
     **/
    public Product finByUnitTypeCode(String unitTypeCode) {
        return this.lambdaQuery().eq(Product::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(Product::getProductCode, unitTypeCode)
                .one();
    }

    /**
     * 根据商品编码集合获取商品集合
     *
     * @return List<String>
     */
    public List<String> findAllProductCode(String tenantCode) {
        return this.baseMapper.findAllProductCode(tenantCode);
    }

    /**
     * 根据商品编码集合获取商品集合
     *
     * @param codes      产品编码
     * @param tenantCode 租户编码
     * @return List<CommonSelectVo>
     */
    public List<CommonSelectVo> getProductByCodes(List<List<String>> codes, String tenantCode) {
        return this.baseMapper.getProductByCodes(codes, tenantCode);
    }

    /**
     * 批量查询
     *
     * @param codes
     * @return java.util.List<com.biz.crm.mdm.business.product.local.entity.Product>
     * @author huojia
     * @date 2023/1/20 19:18
     **/
    public List<Product> findByCodes(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)) {
            return null;
        }
        return this.lambdaQuery().eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .in(Product::getProductCode, codes)
                .list();
    }

    /**
     * 根据商品编码查询商品名称
     * @param ids
     * @param productCodes
     * @return {@link Map}<{@link String}, {@link String}>
     */
    public Map<String, String> findNameByIdsOrCodes(List<String> ids, List<String> productCodes) {
        Assert.isTrue(!org.springframework.util.CollectionUtils.isEmpty(ids)
            || !org.springframework.util.CollectionUtils.isEmpty(productCodes), "参数Id或者编码至少有一个");
        Map<String, String> resultMap = Maps.newHashMap();
        String tenantCode = TenantUtils.getTenantCode();
        List<ProductVo> productVos = this.baseMapper.findNameByIdsOrCodes(ids, productCodes, tenantCode);
        if(org.springframework.util.CollectionUtils.isEmpty(productVos)){
            return resultMap;
        }
        productVos.forEach(o -> {
            if(!org.springframework.util.CollectionUtils.isEmpty(ids)){
                resultMap.put(o.getId(), o.getProductName());
            }else if(!org.springframework.util.CollectionUtils.isEmpty(productCodes)){
                resultMap.put(o.getProductCode(), o.getProductName());
            }
        });
        return resultMap;
    }
    /**
     * 根据业务type查询物料包或者品牌组或者品类名称
     *
     * @return
     */
    public List<Product> findPackageDescOrBrandOrgOrCategory(DisplayProductDto displayProductDto) {
        String brandOrg = displayProductDto.getBrandOrg();
        String productCategoryCode = displayProductDto.getProductCategoryCode();
        String packageDesc = displayProductDto.getPackageDesc();
        return this.lambdaQuery()
            .eq(Product::getTenantCode,TenantUtils.getTenantCode())
            .eq(Product::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
            .eq(Product::getEnableStatus,EnableStatusEnum.ENABLE.getCode())
            .eq(StringUtils.isNotEmpty(brandOrg),Product::getBrandOrg,brandOrg)
            .eq(StringUtils.isNotEmpty(productCategoryCode),Product::getProductCategoryCode,productCategoryCode)
            .eq(StringUtils.isNotEmpty(packageDesc),Product::getPackageDesc,packageDesc)
            .select(Product::getBrandOrg,Product::getProductCategoryCode,Product::getProductCategoryName,Product::getPackageDesc)
            .list();

    }

    /**
     * 单个编码查询
     * @param productCode
     * @param enableStatusFlag
     * @return {@link ProductVo}
     */
    public ProductVo findByCode(String productCode, Boolean enableStatusFlag) {
        ProductDto dto = new ProductDto();
        dto.setProductCode(productCode);
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        if(enableStatusFlag) {
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findByProductDto(dto);
    }
}
