package com.biz.crm.mdm.business.product.local.service.internal;

import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import com.biz.crm.mdm.business.product.local.entity.ProductMaterial;
import com.biz.crm.mdm.business.product.local.repository.ProductMaterialRepository;
import com.biz.crm.mdm.business.product.local.service.ProductMaterialService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author sunx
 * @date 2021/10/14
 */
@Service("productMaterialService")
public class ProductMaterialServiceImpl implements ProductMaterialService {

  @Autowired(required = false) private ProductMaterialRepository productMaterialRepository;

  @Autowired(required = false) private MaterialVoService materialVoService;

  @Override
  public List<ProductMaterial> findByProductCodes(List<String> productCodeList) {
    if (CollectionUtils.isEmpty(productCodeList)) {
      return Lists.newLinkedList();
    }
    List<ProductMaterial> list = productMaterialRepository.findByProductCodes(productCodeList);
    // 设置物料名称
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<String> materialCodeList =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getMaterialCode()))
            .map(ProductMaterial::getMaterialCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(materialCodeList)) {
      return list;
    }
    List<MaterialVo> materialVoList =
        materialVoService.findDetailByMaterialCodes(Sets.newHashSet(materialCodeList));
    if (CollectionUtils.isEmpty(materialVoList)) {
      return list;
    }
    Map<String, MaterialVo> map =
        materialVoList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getMaterialCode()))
            .collect(
                Collectors.toMap(MaterialVo::getMaterialCode, Function.identity(), (a, b) -> b));
    list.forEach(
        a -> {
          MaterialVo vo = map.get(a.getMaterialCode());
          if (Objects.nonNull(vo)) {
            a.setMaterialName(vo.getMaterialName());
          }
        });
    return list;
  }

  @Override
  @Transactional
  public void saveBatch(List<ProductMaterial> productMaterialList, String productCode) {
    Validate.notBlank(productCode, "商品编码不能为空");
    productMaterialRepository.deleteByProductCodes(Lists.newArrayList(productCode));
    if (CollectionUtils.isEmpty(productMaterialList)) {
      return;
    }
    Optional<ProductMaterial> first =
        productMaterialList.stream()
            .filter(
                a ->
                    StringUtils.isBlank(a.getProductCode())
                        || StringUtils.isBlank(a.getMaterialCode())
                        || !productCode.equals(a.getProductCode()))
            .findFirst();
    Validate.isTrue(!first.isPresent(), "商品编码或物料编码不能为空,且必须属于同一商品");
    Map<String, List<ProductMaterial>> map =
        productMaterialList.stream()
            .collect(Collectors.groupingBy(ProductMaterial::getMaterialCode));
    map.entrySet().forEach(a -> Validate.isTrue(a.getValue().size() <= 1, "存在重复的物料信息"));

    for (ProductMaterial productMaterial : productMaterialList) {
      productMaterial.setId(null);
      productMaterial.setTenantCode(TenantUtils.getTenantCode());
    }
    productMaterialRepository.saveBatch(productMaterialList);
  }

  /**
   * 根据物料编码获取匹配的数量
   *
   * @param materialCodes
   * @return
   */
  @Override
  public Integer countByMaterialCodes(List<String> materialCodes) {
    if (CollectionUtils.isEmpty(materialCodes)) {
      return 0;
    }
    return productMaterialRepository.countByMaterialCodes(materialCodes);
  }

  @Override
  public Set<String> findProductCodeByMaterialCode(String materialCode) {
    if (StringUtils.isBlank(materialCode)) {
      return Sets.newHashSet();
    }
    return this.productMaterialRepository.findProductCodeByMaterialCode(materialCode);
  }

  @Override
  @Transactional
  public void create(ProductMaterial productMaterial) {
    Validate.notNull(productMaterial, "物料信息不能为空");
    //新增租户编号
    productMaterial.setTenantCode(TenantUtils.getTenantCode());
    this.productMaterialRepository.save(productMaterial);
  }
}
