package com.biz.crm.mdm.business.product.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.db.PageResult;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.material.unit.service.MaterialUnitVoService;
import com.biz.crm.mdm.business.price.sdk.service.PriceVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceFeeVo;
import com.biz.crm.mdm.business.product.brand.sdk.service.ProductBrandBusinessFormatService;
import com.biz.crm.mdm.business.product.brand.sdk.service.ProductBrandService;
import com.biz.crm.mdm.business.product.brand.sdk.vo.ProductBrandVo;
import com.biz.crm.mdm.business.product.level.sdk.dto.RelateProductLevelCodeQueryDto;
import com.biz.crm.mdm.business.product.level.sdk.enums.ProductLevelEnum;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelService;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.MdgProductLevelVo;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.local.constant.ProductConstant;
import com.biz.crm.mdm.business.product.local.entity.Product;
import com.biz.crm.mdm.business.product.local.entity.ProductIntroduction;
import com.biz.crm.mdm.business.product.local.entity.ProductUnit;
import com.biz.crm.mdm.business.product.local.repository.ProductRepository;
import com.biz.crm.mdm.business.product.local.service.ProductIntroductionService;
import com.biz.crm.mdm.business.product.local.service.ProductService;
import com.biz.crm.mdm.business.product.local.service.ProductUnitService;
import com.biz.crm.mdm.business.product.sdk.constant.ProductUnitConstant;
import com.biz.crm.mdm.business.product.sdk.dto.ContractProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductEventDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductPaginationDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductSingleEventDto;
import com.biz.crm.mdm.business.product.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.sdk.event.ProductEventListener;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductPlanningDetailsVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalRelaOrgVo;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgMaterialVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * mdm_product 商品主数据(Product)表服务实现类
 *
 * @author sunx
 * @since 2021-10-13 16:30:34
 */
@Service("productService")
@Slf4j
public class ProductServiceImpl implements ProductService {

    /**
     * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
     */
    private static volatile Cache<String, List<Product>> cache = null;
    /**
     * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
     */
    @Autowired(required = false)
    private ProductRepository productRepository;
    @Autowired(required = false)
    private ProductVoService productVoService;
    @Autowired(required = false)
    private ProductLevelVoSdkService productLevelVoSdkService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private ProductIntroductionService productIntroductionService;
    @Autowired(required = false)
    private PriceVoService priceVoService;

    @Autowired(required = false)
    private ProductLevelService productLevelService;

    @Autowired(required = false)
    private ProductBrandService productBrandService;
    @Autowired(required = false)
    @Lazy
    private List<ProductEventListener> listeners;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;
    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;

  @Autowired(required = false)
  private ProductUnitService productUnitService;

    @Autowired(required = false)
    private MaterialUnitVoService materialUnitVoService;

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private ProductBrandBusinessFormatService productBrandBusinessFormatService;

    public ProductServiceImpl() {
        if (cache == null) {
            synchronized (ProductServiceImpl.class) {
                while (cache == null) {
                    cache = CacheBuilder.newBuilder().initialCapacity(10000)
                            .expireAfterWrite(500, TimeUnit.MILLISECONDS).maximumSize(100000).build();
                }
            }
        }
    }

    /**
     * 设置产品和品牌和产品类型管理
     *
     * @param tenantCode                  租户编码
     * @param productBrandCodeSet         品牌编码集合
     * @param productLevelCodeSet         品项编码集合
     * @param productBrandList            品牌集合
     * @param productLevelList            品项集合
     * @param masterDataMdgSupplierBankVo MDG产品信息
     */
    private Product setProductData(String tenantCode, Set<String> productLevelCodeSet, Set<String> productBrandCodeSet,
                                   List<MdgProductLevelVo> productLevelList, List<ProductBrandVo> productBrandList,
                                   MasterDataMdgMaterialVo masterDataMdgSupplierBankVo,
                                   Map<String, Set<String>> businessFormatBrandMap) {
        Product product = new Product();
        // 商品编码 --> SAP物料编码
        product.setProductCode(masterDataMdgSupplierBankVo.getMatnr());
        // 商品名称 --> 物料描述
        product.setProductName(masterDataMdgSupplierBankVo.getMaktx());
        // 基本单位描述
        product.setBaseUnitDes(masterDataMdgSupplierBankVo.getMeinsdesc());
        // 商品类型
        product.setProductType(masterDataMdgSupplierBankVo.getMtart());
        // 商品组
        product.setProductOrg(masterDataMdgSupplierBankVo.getMatkl());
        // 行业领域
        product.setBizField(masterDataMdgSupplierBankVo.getMbrsh());
        // 物料组描述
        product.setProductOrgDesc(masterDataMdgSupplierBankVo.getMbrsh());
        // 产品入数
        product.setProductOrgDesc(masterDataMdgSupplierBankVo.getMbrsh());
        // 产品入数
        product.setProductIncoming(masterDataMdgSupplierBankVo.getZzmat010());
        // 重量单位
        product.setUnitWeight(masterDataMdgSupplierBankVo.getGewei());
        // 体积单位
        product.setVolumeWeight(masterDataMdgSupplierBankVo.getVoleh());
        // 小包装容量
        product.setPackageQuantityMin(masterDataMdgSupplierBankVo.getZfxbz());
        // 小包装容量单位
        product.setPackageQuantityMinUnite(masterDataMdgSupplierBankVo.getZzmat008());
        // 净重
        product.setNetWeight(new BigDecimal(masterDataMdgSupplierBankVo.getNtgew().trim()));
        // 物料组:包装物料
        product.setPackageProduct(masterDataMdgSupplierBankVo.getMagrv());
        // 包装描述
        product.setPackageDesc(masterDataMdgSupplierBankVo.getBezei());
        // 财务管理报告品牌
        product.setPackageReport(masterDataMdgSupplierBankVo.getZzmat901());
        // 财务管理报告品牌描述
        product.setPackageReportDescDesc(masterDataMdgSupplierBankVo.getZzmat901dng());
        // 产品真实属性
        product.setPackageTrueProps(masterDataMdgSupplierBankVo.getZzmat011());
        // 口味
        product.setTaste(masterDataMdgSupplierBankVo.getZzmat025());
        // 纸箱条码(外包)
        product.setCartonBarCode(masterDataMdgSupplierBankVo.getZzmat903());
        // 长
        product.setForte(masterDataMdgSupplierBankVo.getLaeng());
        // 宽
        product.setWidth(masterDataMdgSupplierBankVo.getBreit());
        // 高
        product.setHigh(masterDataMdgSupplierBankVo.getHoehe());
        // 包装系数
        product.setPackageScale(masterDataMdgSupplierBankVo.getZzmat004());
        // 液态产品密度
        product.setLiquidStateProductDensity(masterDataMdgSupplierBankVo.getZzmat007());
        // 业务量
        product.setBusinessAmount(masterDataMdgSupplierBankVo.getVolum());
        // 液态产品密度
        product.setLiquidStateProductDensity(masterDataMdgSupplierBankVo.getWgbez());
        // 条形码
        product.setBarCode(masterDataMdgSupplierBankVo.getZzmat902());
        // 产品品类编码
        product.setProductCategoryCode(masterDataMdgSupplierBankVo.getZzmat030());
        // 产品品类名称
        product.setProductCategoryName(masterDataMdgSupplierBankVo.getZzmat030desc());
        // 产品品相编码
        product.setProductLevelCode(masterDataMdgSupplierBankVo.getZzmat034());
        // 产品品相名称
        product.setProductLevelName(masterDataMdgSupplierBankVo.getBackup2());
        // 产品品牌编码
        product.setProductBrandCode(masterDataMdgSupplierBankVo.getZzmat031());
        // 产品品牌名称
        product.setProductBrandName(masterDataMdgSupplierBankVo.getBackup1());
        // 业态
        product.setBusinessFormatCode(masterDataMdgSupplierBankVo.getSpart());
        // 基本单位
        product.setBaseUnit(masterDataMdgSupplierBankVo.getMeins());
        //把ST转换成PC
        if(ProductUnitConstant.ST.equals(product.getBaseUnit())){
            product.setBaseUnit(ProductUnitConstant.PC);
        }
        // 规格
        product.setSpec(masterDataMdgSupplierBankVo.getZzmat039());
        // 品牌组
        product.setBrandOrg(masterDataMdgSupplierBankVo.getZzmat033desc());
        // 保质期
        Integer expirationDate = new Integer(masterDataMdgSupplierBankVo.getMhdhb().trim());
        product.setExpirationDate(expirationDate);
        // 税率
        if (StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getZzmat036())) {
            BigDecimal bigDecimal = new BigDecimal(masterDataMdgSupplierBankVo.getZzmat036().trim());
            product.setRate(bigDecimal);
        }

        // 数据业务状态（启用状态） mstae字段只要有数据就是禁用
        // 上下架状态
        if (StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getMstae())) {
            product.setIsShelf(BooleanEnum.FALSE.getCapital());
            product.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
        } else {
            product.setIsShelf(BooleanEnum.TRUE.getCapital());
            product.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        }
        // 租户编号
        product.setTenantCode(tenantCode);
        // 数据状态（删除状态）
        product.setDelFlag(EnableStatusEnum.ENABLE.getCode());

        //品牌与业态关系
        if(org.springframework.util.StringUtils.hasText(product.getProductBrandCode())
            && org.springframework.util.StringUtils.hasText(product.getBusinessFormatCode())){
            //品牌业态关系
            Set<String> businessFormatSet = businessFormatBrandMap.get(product.getProductBrandCode());
            if(org.springframework.util.CollectionUtils.isEmpty(businessFormatSet)){
                businessFormatSet = Sets.newHashSet();
            }
            businessFormatSet.add(product.getBusinessFormatCode());
            businessFormatBrandMap.put(product.getProductBrandCode(), businessFormatSet);
        }

        if (StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getZzmat031())
                && StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getBackup1())
                && !productBrandCodeSet.contains(masterDataMdgSupplierBankVo.getZzmat031())) {
            // 添加品牌对象
            ProductBrandVo productBrand = new ProductBrandVo();
            productBrandCodeSet.add(masterDataMdgSupplierBankVo.getZzmat031());
            productBrand.setProductBrandCode(masterDataMdgSupplierBankVo.getZzmat031());
            // 产品品牌名称
            productBrand.setProductBrandName(masterDataMdgSupplierBankVo.getBackup1());
            // 租户编号
            productBrand.setTenantCode(tenantCode);
            // 数据业务状态（启用状态）
            productBrand.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            // 数据业务状态（启用状态）
            productBrand.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            productBrandList.add(productBrand);
        }
        if (StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getZzmat034())
                || StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getZzmat030())) {
            if (StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getZzmat034())
                    && StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getBackup2())
                    && !productLevelCodeSet.contains(masterDataMdgSupplierBankVo.getZzmat034())) {
                productLevelCodeSet.add(masterDataMdgSupplierBankVo.getZzmat034());
                MdgProductLevelVo productLevel = new MdgProductLevelVo();
                productLevel.setProductLevelType(ProductLevelEnum.items);
                // 品相编码
                productLevel.setProductLevelCode(masterDataMdgSupplierBankVo.getZzmat034());
                // 品相名称
                productLevel.setProductLevelName(masterDataMdgSupplierBankVo.getBackup2());
                // 所属品类编码
                productLevel.setParentCode(masterDataMdgSupplierBankVo.getZzmat030());
                // 租户编号
                productLevel.setTenantCode(tenantCode);
                // 数据业务状态（启用状态）
                productLevel.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                // 数据状态（删除状态）
                productLevel.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                productLevelList.add(productLevel);
            } else if (StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getZzmat030())
                    && StringUtils.isNotBlank(masterDataMdgSupplierBankVo.getZzmat030desc())
                    && !productLevelCodeSet.contains(masterDataMdgSupplierBankVo.getZzmat030())) {
                productLevelCodeSet.add(masterDataMdgSupplierBankVo.getZzmat030());
                MdgProductLevelVo productLevel = new MdgProductLevelVo();
                productLevel.setProductLevelType(ProductLevelEnum.category);
                // 品类编码
                productLevel.setProductLevelCode(masterDataMdgSupplierBankVo.getZzmat030());
                // 品类名称
                productLevel.setProductLevelName(masterDataMdgSupplierBankVo.getZzmat030desc());
                // 租户编号
                productLevel.setTenantCode(tenantCode);
                // 数据业务状态（启用状态）
                productLevel.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                // 数据状态（删除状态）
                productLevel.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                productLevelList.add(productLevel);
            }

        }
        return product;
    }

    @Override
    public Page<Product> findByConditions(Pageable pageable, ProductPaginationDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new ProductPaginationDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<Product> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<Product> pageResult = this.productRepository.findByConditions(page, dto);
        if(CollectionUtils.isEmpty(pageResult.getRecords())){
         return pageResult;
        }
        // 绑定商品层级名称
        List<String> productLevelCodes = Lists.newLinkedList();
        Map<String, String> productLevelMap = Maps.newHashMap();
        List<ProductLevelVo> productLevelVos = Lists.newLinkedList();
        Boolean flag = pageResult != null && CollectionUtils.isNotEmpty(pageResult.getRecords());
        if (Boolean.TRUE.equals(flag)) {
            productLevelCodes =
                    pageResult.getRecords().stream().filter(a -> StringUtils.isNotBlank(a.getProductLevelCode()))
                            .map(Product::getProductLevelCode).collect(Collectors.toList());
        }
        if (CollectionUtils.isNotEmpty(productLevelCodes)) {
            productLevelVos = this.productLevelVoSdkService.findListByCodes(productLevelCodes);
        }
        if (CollectionUtils.isNotEmpty(productLevelVos)) {
            productLevelMap = productLevelVos.stream().filter(
                    a -> StringUtils.isNotBlank(a.getProductLevelCode()) && StringUtils.isNotBlank(a.getProductLevelName()))
                    .collect(Collectors.toMap(ProductLevelVo::getProductLevelCode, ProductLevelVo::getProductLevelName,
                            (a, b) -> a));
        }
      List<ProductUnit> productUnits = productUnitService.findByProductCodes((ArrayList<String>) pageResult.getRecords().stream()
              .map(Product::getProductCode).collect(Collectors.toList()));
      if(CollectionUtils.isNotEmpty(productUnits)){
        Map<String, List<ProductUnit>> collect
                = productUnits.stream().collect(Collectors.groupingBy(ProductUnit::getProductCode));
        pageResult.getRecords().forEach(s->s.setUnitList(collect.get(s.getProductCode())));
      }
      if (Boolean.TRUE.equals(flag)) {
            for (Product item : pageResult.getRecords()) {
                item.setProductLevelName(productLevelMap.getOrDefault(item.getProductLevelCode(), StringUtils.EMPTY));
                item.setNetWeight(item.getNetWeight() == null ? BigDecimal.ZERO : item.getNetWeight());
            }
        }
        return pageResult;
    }

    @Override
    public List<Product> findDetailsByIdsOrProductCodes(List<String> ids, List<String> productCodes) {
        if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(productCodes)) {
            return Lists.newLinkedList();
        }
        StringBuilder sb = new StringBuilder();
        if (CollectionUtils.isNotEmpty(ids)) {
            sb.append(StringUtils.join(ids));
        }
        if (CollectionUtils.isNotEmpty(productCodes)) {
            sb.append(StringUtils.join(productCodes));
        }
        String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), sb.toString());
        List<Product> graph = cache.getIfPresent(cacheKey);
        if (graph == null) {
            graph = this.productRepository.findByIdsOrProductCodes(ids, productCodes);
            cache.put(cacheKey, graph);
        }
        return graph;
    }

    @Override
    @Transactional
    public Product create(Product product) {
        this.createValidation(product);
        product.setId(null);
        product.setTenantCode(TenantUtils.getTenantCode());
        product.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        product.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        this.productRepository.saveOrUpdate(product);
        return product;
    }

    @Override
    @Transactional
    public Product update(Product product) {
        this.updateValidation(product);
        String currentId = product.getId();
        Product current = this.productRepository.findById(currentId);
        Validate.notNull(current, "修改信息不存在");
        Validate.isTrue(product.getProductCode().equals(current.getProductCode()), "商品编码不能修改");
        // 新增租户编号
        product.setTenantCode(TenantUtils.getTenantCode());
        this.productRepository.saveOrUpdate(product);
        return product;
    }

    @Override
    @Transactional
    public void updateDelFlagByIds(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        List<ProductVo> list = this.productVoService.findDetailsByIdsOrProductCodes(ids, null);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        this.productRepository.updateDelFlagByIds(ids);
        ProductEventDto productEventDto = new ProductEventDto();
        for (ProductVo productVo : list) {
            productEventDto.setOriginal(productVo);
            productEventDto.setNewest(null);
            SerializableBiConsumer<ProductEventListener, ProductEventDto> onDelete = ProductEventListener::onDelete;
            this.nebulaNetEventClient.publish(productEventDto, ProductEventListener.class, onDelete);
        }
    }

    @Override
    @Transactional
    public void upShelf(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.productRepository.updateIsShelfByIds(ids, IsShelfEnum.UP.getCode());
        List<ProductVo> list = this.productVoService.findDetailsByIdsOrProductCodes(ids, null);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (ProductVo productVo : list) {
            ProductSingleEventDto productSingleEventDto = new ProductSingleEventDto();
            productSingleEventDto.setId(productVo.getId());
            final JSONObject object1 = new JSONObject();
            object1.put("isShelf", IsShelfEnum.DOWN.getCode());
            object1.put("enableStatus", productVo.getEnableStatus());
            object1.put("productCode", productVo.getProductCode());
            object1.put("productLevelCode", productVo.getProductLevelCode());
            final JSONObject object2 = new JSONObject();
            object2.put("isShelf", IsShelfEnum.UP.getCode());
            object2.put("enableStatus", productVo.getEnableStatus());
            object2.put("productCode", productVo.getProductCode());
            object2.put("productLevelCode", productVo.getProductLevelCode());
            productSingleEventDto.setOriginal(object1);
            productSingleEventDto.setNewest(object2);
            SerializableBiConsumer<ProductEventListener, ProductSingleEventDto> onUpShelf =
                    ProductEventListener::onUpShelf;
            this.nebulaNetEventClient.publish(productSingleEventDto, ProductEventListener.class, onUpShelf);
        }
    }

    @Override
    @Transactional
    public void downShelf(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.productRepository.updateIsShelfByIds(ids, IsShelfEnum.DOWN.getCode());
        List<ProductVo> list = this.productVoService.findDetailsByIdsOrProductCodes(ids, null);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (ProductVo productVo : list) {
            ProductSingleEventDto productSingleEventDto = new ProductSingleEventDto();
            productSingleEventDto.setId(productVo.getId());
            final JSONObject object1 = new JSONObject();
            object1.put("isShelf", IsShelfEnum.UP.getCode());
            object1.put("enableStatus", productVo.getEnableStatus());
            object1.put("productCode", productVo.getProductCode());
            object1.put("productLevelCode", productVo.getProductLevelCode());
            final JSONObject object2 = new JSONObject();
            object2.put("isShelf", IsShelfEnum.DOWN.getCode());
            object2.put("enableStatus", productVo.getEnableStatus());
            object2.put("productCode", productVo.getProductCode());
            object2.put("productLevelCode", productVo.getProductLevelCode());
            productSingleEventDto.setOriginal(object1);
            productSingleEventDto.setNewest(object2);
            SerializableBiConsumer<ProductEventListener, ProductSingleEventDto> onDownShelf =
                    ProductEventListener::onDownShelf;
            this.nebulaNetEventClient.publish(productSingleEventDto, ProductEventListener.class, onDownShelf);
        }
    }

    @Override
    @Transactional
    public void upShelfByProductCodes(List<String> productCodeList) {
        Validate.isTrue(CollectionUtils.isNotEmpty(productCodeList), "productCode集合不能为空");
        this.productRepository.updateIsShelfByByProductCodes(productCodeList, IsShelfEnum.UP.getCode());

        List<ProductVo> list = this.productVoService.findDetailsByIdsOrProductCodes(null, productCodeList);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (ProductVo productVo : list) {
            ProductSingleEventDto productSingleEventDto = new ProductSingleEventDto();
            productSingleEventDto.setId(productVo.getId());
            final JSONObject object1 = new JSONObject();
            object1.put("isShelf", IsShelfEnum.DOWN.getCode());
            object1.put("enableStatus", productVo.getEnableStatus());
            object1.put("productCode", productVo.getProductCode());
            object1.put("productLevelCode", productVo.getProductLevelCode());
            final JSONObject object2 = new JSONObject();
            object2.put("isShelf", IsShelfEnum.UP.getCode());
            object2.put("enableStatus", productVo.getEnableStatus());
            object2.put("productCode", productVo.getProductCode());
            object2.put("productLevelCode", productVo.getProductLevelCode());
            productSingleEventDto.setOriginal(object1);
            productSingleEventDto.setNewest(object2);
            SerializableBiConsumer<ProductEventListener, ProductSingleEventDto> onUpShelf =
                    ProductEventListener::onUpShelf;
            this.nebulaNetEventClient.publish(productSingleEventDto, ProductEventListener.class, onUpShelf);
        }
    }

    @Override
    @Transactional
    public void downShelfByProductCodes(List<String> productCodeList) {
        Validate.isTrue(CollectionUtils.isNotEmpty(productCodeList), "productCode集合不能为空");
        this.productRepository.updateIsShelfByByProductCodes(productCodeList, IsShelfEnum.DOWN.getCode());
        List<ProductVo> list = this.productVoService.findDetailsByIdsOrProductCodes(null, productCodeList);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (ProductVo productVo : list) {
            ProductSingleEventDto productSingleEventDto = new ProductSingleEventDto();
            productSingleEventDto.setId(productVo.getId());
            final JSONObject object1 = new JSONObject();
            object1.put("isShelf", IsShelfEnum.UP.getCode());
            object1.put("enableStatus", productVo.getEnableStatus());
            object1.put("productCode", productVo.getProductCode());
            object1.put("productLevelCode", productVo.getProductLevelCode());
            final JSONObject object2 = new JSONObject();
            object2.put("isShelf", IsShelfEnum.DOWN.getCode());
            object2.put("enableStatus", productVo.getEnableStatus());
            object2.put("productCode", productVo.getProductCode());
            object2.put("productLevelCode", productVo.getProductLevelCode());
            productSingleEventDto.setOriginal(object1);
            productSingleEventDto.setNewest(object2);
            SerializableBiConsumer<ProductEventListener, ProductSingleEventDto> onDownShelf =
                    ProductEventListener::onDownShelf;
            this.nebulaNetEventClient.publish(productSingleEventDto, ProductEventListener.class, onDownShelf);
        }
    }

    @Override
    @Transactional
    public void enableBatch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.productRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
        List<ProductVo> list = this.productVoService.findDetailsByIdsOrProductCodes(ids, null);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (ProductVo productVo : list) {
            ProductSingleEventDto productSingleEventDto = new ProductSingleEventDto();
            productSingleEventDto.setId(productVo.getId());
            final JSONObject object1 = new JSONObject();
            object1.put("isShelf", productVo.getIsShelf());
            object1.put("enableStatus", EnableStatusEnum.DISABLE.getCode());
            object1.put("productCode", productVo.getProductCode());
            object1.put("productLevelCode", productVo.getProductLevelCode());
            final JSONObject object2 = new JSONObject();
            object2.put("isShelf", productVo.getIsShelf());
            object2.put("enableStatus", EnableStatusEnum.ENABLE.getCode());
            object2.put("productCode", productVo.getProductCode());
            object2.put("productLevelCode", productVo.getProductLevelCode());
            productSingleEventDto.setOriginal(object1);
            productSingleEventDto.setNewest(object2);
            SerializableBiConsumer<ProductEventListener, ProductSingleEventDto> onEnable =
                    ProductEventListener::onEnable;
            this.nebulaNetEventClient.publish(productSingleEventDto, ProductEventListener.class, onEnable);
        }
    }

    @Override
    @Transactional
    public void disableBatch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.productRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
        List<ProductVo> list = this.productVoService.findDetailsByIdsOrProductCodes(ids, null);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (ProductVo productVo : list) {
            ProductSingleEventDto productSingleEventDto = new ProductSingleEventDto();
            productSingleEventDto.setId(productVo.getId());
            final JSONObject object1 = new JSONObject();
            object1.put("isShelf", productVo.getIsShelf());
            object1.put("enableStatus", EnableStatusEnum.ENABLE.getCode());
            object1.put("productCode", productVo.getProductCode());
            object1.put("productLevelCode", productVo.getProductLevelCode());
            final JSONObject object2 = new JSONObject();
            object2.put("isShelf", productVo.getIsShelf());
            object2.put("enableStatus", EnableStatusEnum.DISABLE.getCode());
            object2.put("productCode", productVo.getProductCode());
            object2.put("productLevelCode", productVo.getProductLevelCode());
            productSingleEventDto.setOriginal(object1);
            productSingleEventDto.setNewest(object2);
            //暂时注释商品禁用推送
//            SerializableBiConsumer<ProductEventListener, ProductSingleEventDto> onDisable =
//                    ProductEventListener::onDisable;
//            this.nebulaNetEventClient.publish(productSingleEventDto, ProductEventListener.class, onDisable);
        }
    }

    @Override
    public Integer countByProductLevelCodesAndDelFlag(List<String> productLevelCodes, String delFlag) {
        if (CollectionUtils.isEmpty(productLevelCodes) || StringUtils.isBlank(delFlag)) {
            return 0;
        }
        return this.productRepository.countByProductLevelCodesAndDelFlag(productLevelCodes, delFlag);
    }

    @Override
    public Set<String> findProductLevelCodeSetByProductCodes(Set<String> productCodeSet) {
        if (CollectionUtils.isEmpty(productCodeSet)) {
            return Sets.newHashSet();
        }
        return this.productRepository.findProductLevelCodeSetByProductCodes(productCodeSet);
    }

    @Override
    public Product findByProductCode(String code) {
        if (StringUtils.isBlank(code)) {
            return null;
        }
        return this.productRepository.findByProductCode(code, DelFlagStatusEnum.NORMAL.getCode());
    }

    @Override
    public List<Product> findByContractProductQueryDto(ContractProductQueryDto dto) {
        if (Objects.isNull(dto) || (CollectionUtils.isEmpty(dto.getProductCodeSet())
                && CollectionUtils.isEmpty(dto.getProductLevelCodeSet()))) {
            return Lists.newLinkedList();
        }
        Set<String> levelCodeSet = Sets.newHashSet();
        // 获取层级及下级
        if (!CollectionUtils.isEmpty(dto.getProductLevelCodeSet())) {
            final RelateProductLevelCodeQueryDto queryDto = new RelateProductLevelCodeQueryDto();
            queryDto.setSearchType(-1);
            queryDto.setProductLevelCodeSet(dto.getProductLevelCodeSet());
            // k-层级编码
            final Map<String, String> mapLevel =
                    this.productLevelVoSdkService.findByRelateProductLevelCodeQueryDto(queryDto);
            if (!mapLevel.isEmpty()) {
                levelCodeSet.addAll(mapLevel.keySet());
            }
        }
        dto.setProductLevelCodeSet(levelCodeSet);
        dto.setUnionType(Boolean.FALSE);
        if (!CollectionUtils.isEmpty(dto.getProductCodeSet())
                && !CollectionUtils.isEmpty(dto.getProductLevelCodeSet())) {
            dto.setUnionType(Boolean.TRUE);
        }
        return this.productRepository.findByContractProductQueryDto(dto);
    }

    /**
     * 根据商品编码集合获取对应的商品数据信息
     *
     * @param productCode 商品编码信息
     */
    @Override
    public ProductPlanningDetailsVo findByProductPlanningDetails(String productCode, String customerCode,
                                                                 String businessModel, String startDate) {
        if (StringUtils.isEmpty(productCode)) {
            return new ProductPlanningDetailsVo();
        }
        Product planningDetails = this.productRepository.findByProductPlanningDetails(productCode);
        if (ObjectUtils.isEmpty(planningDetails)) {
            throw new RuntimeException("价格管理查询数据不不存在！");
        }
        ProductPlanningDetailsVo vo = new ProductPlanningDetailsVo(planningDetails.getProductCode(),
                planningDetails.getProductName(), planningDetails.getProductBrandName(),
                planningDetails.getProductBrandCode(), planningDetails.getRate());
        // 通过产品编码查询对应价格
        PriceFeeVo feeVo =
                priceVoService.findByGoodsCode(planningDetails.getProductCode(), customerCode, businessModel, startDate);

        BeanUtils.copyProperties(feeVo, vo);
        return vo;
    }

    /**
     * 批量拉取 MDG 物料数据
     *
     * @param dto
     * @author huojia
     * @date 2022/12/5 21:37
     **/
    @Override
    public void pullMaterialList(MasterDataMdgBaseDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        int pageNum = 1;
        if (StringUtils.isNotBlank(dto.getPageNum())) {
            pageNum = Integer.parseInt(dto.getPageNum());
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }
        boolean lock = true;
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        if (!StringUtils.isEmpty(dto.getDs())) {
            lockKey = dto.getDs();
        }
        boolean loopFlag = true;
        int loopMAX = 40;
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            //  编码去 重
            Set<String> codeSet = new HashSet<>(4096);
            Set<String> productBrandCodeSet = new HashSet<>(4096);
            Set<String> productLevelCodeSet = new HashSet<>(4096);
            //品牌与业态关系(key 品牌，value 业态)
            Map<String, Set<String>> businessFormatBrandMap = Maps.newHashMap();
            while (loopFlag && pageNum <= loopMAX) {
                dto.setPageNum(String.valueOf(pageNum));
                log.info("拉取MDG商品信息,页码[{}],每页大小[{}]", dto.getPageNum(), dto.getPageSize());
                List<MasterDataMdgMaterialVo> masterDataMdgMaterialVos = masterDataMdgService.pullMaterialList(dto);
                pageNum++;
                if (CollectionUtils.isEmpty(masterDataMdgMaterialVos)) {
                    loopFlag = false;
                    continue;
                }
                List<MdgProductLevelVo> productLevelList = new ArrayList<>();
                List<ProductBrandVo> productBrandList = new ArrayList<>();
                // 数据校验
                List<Product> pullList = this.materialValidate(masterDataMdgMaterialVos, productLevelList,
                    productBrandList, codeSet, productBrandCodeSet, productLevelCodeSet, businessFormatBrandMap);
                List<Product> saveList = new ArrayList<>();
                List<Product> updateList = new ArrayList<>();
                List<MdgProductLevelVo> saveProductLevelList = new ArrayList<>();
                List<MdgProductLevelVo> updateProductLevelList = new ArrayList<>();
                List<ProductBrandVo> saveProductBrandList = new ArrayList<>();
                List<ProductBrandVo> updateProductBrandList = new ArrayList<>();
                // 区分产品更新、新增
                this.filterProduct(pullList, saveList, updateList);
                // 区分产品品类品项更新、新增
                this.filterProductLevel(productLevelList, saveProductLevelList, updateProductLevelList);
                // 区分产品品牌更新、新增
                this.filterProductBrand(productBrandList, saveProductBrandList, updateProductBrandList);
                this.saveOrUpdateBatch(saveList, updateList, saveProductLevelList, updateProductLevelList,
                        saveProductBrandList, updateProductBrandList);
            }
            //保存品牌业态关系
            this.productBrandBusinessFormatService.deleteAll(businessFormatBrandMap);
            this.productBrandBusinessFormatService.saveBusinessFormatBrand(businessFormatBrandMap);
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }

    /**
     * 过滤富文本信息
     *
     * @param productIntroductionList
     * @param saveProductIntroductionList
     * @param updateProductIntroductionList
     */
    private void filterProductIntroductionBrand(List<ProductIntroduction> productIntroductionList,
                                                List<ProductIntroduction> saveProductIntroductionList,
                                                List<ProductIntroduction> updateProductIntroductionList) {
        if (CollectionUtil.isEmpty(productIntroductionList)) {
            return;
        }
        List<String> codeList = productIntroductionList.stream().filter(v -> StringUtils.isNotBlank(v.getProductCode()))
                .map(ProductIntroduction::getProductCode).distinct().collect(Collectors.toList());
        List<ProductIntroduction> productIntroductions = productIntroductionService.findAllListByCodeList(codeList);
        if (CollectionUtil.isEmpty(productIntroductions)) {
            saveProductIntroductionList.addAll(productIntroductionList);
            return;
        } else {
            updateProductIntroductionList.addAll(productIntroductionList);
        }
    }

    /**
     * 过滤产品信息
     *
     * @param list
     * @param saveList
     * @param updateList
     */
    private void filterProduct(List<Product> list, List<Product> saveList, List<Product> updateList) {
        if (CollectionUtil.isEmpty(list)) {
            return;
        }
        List<String> productCodeList = list.stream().filter(v -> StringUtils.isNotBlank(v.getProductCode()))
                .map(Product::getProductCode).distinct().collect(Collectors.toList());
        // 区分更新、新增操作
        List<Product> byProductCodes = productRepository.findAllListByCodeList(productCodeList);
        if (CollectionUtils.isEmpty(byProductCodes)) {
            saveList.addAll(list);
            return;
        }
        Map<String, Product> codeMap = byProductCodes.stream()
                .collect(Collectors.toMap(Product::getProductCode, v -> v, (oldValue, newValue) -> newValue));
        list.forEach(pull -> {
            Product oldEntity = codeMap.get(pull.getProductCode());
            if (Objects.nonNull(oldEntity)) {
                pull.setId(oldEntity.getId());
                pull.setCreateAccount(oldEntity.getCreateAccount());
                pull.setCreateName(oldEntity.getCreateName());
                pull.setCreateTime(oldEntity.getCreateTime());
                updateList.add(pull);
            } else {
                saveList.add(pull);
            }
        });
    }

    /**
     * 过滤产品品类品项信息
     *
     * @param list
     * @param saveList
     * @param updateList
     */
    private void filterProductLevel(List<MdgProductLevelVo> list, List<MdgProductLevelVo> saveList,
                                    List<MdgProductLevelVo> updateList) {
        if (CollectionUtil.isEmpty(list)) {
            return;
        }
        List<String> codeList = list.stream().filter(v -> StringUtils.isNotBlank(v.getProductLevelCode()))
                .map(MdgProductLevelVo::getProductLevelCode).distinct().collect(Collectors.toList());
        Map<String, String> codeMap = productLevelService.findAllListByCodeList(codeList);
        if (CollectionUtil.isEmpty(codeMap)) {
            saveList.addAll(list);
            return;
        }
        list.forEach(pull -> {
            String id = codeMap.get(pull.getProductLevelCode());
            if (StringUtils.isNotBlank(id)) {
                pull.setId(id);
                /*pull.setCreateAccount(oldEntity.getCreateAccount());
                pull.setCreateName(oldEntity.getCreateName());
                pull.setCreateTime(oldEntity.getCreateTime());*/
                updateList.add(pull);
            } else {
                saveList.add(pull);
            }
        });
    }

    /**
     * 过滤产品品牌信息
     *
     * @param list
     * @param saveList
     * @param updateList
     */
    private void filterProductBrand(List<ProductBrandVo> list, List<ProductBrandVo> saveList,
                                    List<ProductBrandVo> updateList) {
        if (CollectionUtil.isEmpty(list)) {
            return;
        }
        List<String> codeList = list.stream().filter(v -> StringUtils.isNotBlank(v.getProductBrandCode()))
                .map(ProductBrandVo::getProductBrandCode).distinct().collect(Collectors.toList());
        Map<String, String> codeMap = productBrandService.findAllListByCodeList(codeList);
        if (CollectionUtil.isEmpty(codeMap)) {
            saveList.addAll(list);
            return;
        }
        list.forEach(pull -> {
            String id = codeMap.get(pull.getProductBrandCode());
            if (StringUtils.isNotBlank(id)) {
                pull.setId(id);
                updateList.add(pull);
            } else {
                saveList.add(pull);
            }
        });
    }

    /**
     * 批量新增数据（MDG用）
     *
     * @param saveList
     * @param updateList
     * @author dutaotao
     * @date 2023/1/3 17:38
     */
    @Transactional(rollbackFor = Exception.class)
    public void saveOrUpdateBatch(List<Product> saveList, List<Product> updateList,
                                  List<MdgProductLevelVo> saveProductLevelList, List<MdgProductLevelVo> updateProductLevelList,
                                  List<ProductBrandVo> saveProductBrandList, List<ProductBrandVo> updateProductBrandList) {
        if (CollectionUtils.isNotEmpty(saveList)) {
            this.productRepository.saveBatch(saveList);
        }
        if (CollectionUtils.isNotEmpty(updateList)) {
            this.productRepository.updateBatchById(updateList);
        }
        this.productLevelService.saveOrUpdateBatch(saveProductLevelList, updateProductLevelList);
        this.productBrandService.saveOrUpdateBatch(saveProductBrandList, updateProductBrandList);
    }

    /**
     * @param yearMonthDay
     * @author huojia
     * @date 2022/12/10 15:32
     **/
    private void unLock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取物料解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(ProductConstant.MATERIAL_LOCK + yearMonthDay);
    }

    /**
     * 数据校验
     *
     * @param masterDataMdgMaterialVos
     * @return
     * @author huojia
     * @date 2022/12/6 21:37
     **/
    private List<Product> materialValidate(List<MasterDataMdgMaterialVo> masterDataMdgMaterialVos,
                                           List<MdgProductLevelVo> productLevelList,
                                           List<ProductBrandVo> productBrandList, Set<String> codeSet,
                                           Set<String> productBrandCodeSet,
                                           Set<String> productLevelCodeSet,
                                           Map<String, Set<String>> businessFormatBrandMap) {
        List<Product> pullList = new ArrayList<>();
        String tenantCode = TenantUtils.getTenantCode();
        for (MasterDataMdgMaterialVo masterDataMdgSupplierBankVo : masterDataMdgMaterialVos) {
            if (StringUtils.isEmpty(masterDataMdgSupplierBankVo.getMatnr())) {
                log.error("本次拉取数据物料编码为空" + masterDataMdgMaterialVos);
                throw new RuntimeException("物料编码不能为空，请检查！");
            }
            if (codeSet.contains(masterDataMdgSupplierBankVo.getMatnr())) {
                log.error("本次拉取数据物料编码重复" + masterDataMdgMaterialVos);
                throw new RuntimeException("编码" + masterDataMdgSupplierBankVo.getMatnr() + "重复拉取，请检查！");
            }
            codeSet.add(masterDataMdgSupplierBankVo.getMatnr());
            // 设置产品和品牌和产品类型管理
            Product product = setProductData(tenantCode, productBrandCodeSet, productLevelCodeSet, productLevelList,
                    productBrandList, masterDataMdgSupplierBankVo, businessFormatBrandMap);
            pullList.add(product);
        }
        return pullList;
    }

    /**
     * 按年月日加锁
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private boolean lock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取物料加锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(ProductConstant.MATERIAL_LOCK + yearMonthDay, TimeUnit.HOURS, 12);
    }

    private void createValidation(Product product) {
        this.validation(product);
    }

    private void updateValidation(Product product) {
        Validate.notNull(product, "商品信息缺失");
        Validate.isTrue(StringUtils.isNotBlank(product.getId()), "商品id不能为空");
        this.validation(product);
    }

    private void validation(Product product) {
        Validate.notNull(product, "商品信息缺失");
        Validate.isTrue(StringUtils.isNotBlank(product.getProductCode()), "商品编码不能为空");
//        Validate.isTrue(StringUtils.isNotBlank(product.getProductType()), "商品类型不能为空");
        Validate.isTrue(StringUtils.isNotBlank(product.getProductName()), "商品名称不能为空");
//        Validate.isTrue(StringUtils.isNotBlank(product.getIsShelf()), "上下架不能为空");
//        Validate.notNull(product.getBeginDateTime(), "开始时间不能为空");
//        Validate.notNull(product.getEndDateTime(), "结束时间不能为空");
    }
}
