package com.biz.crm.mdm.business.product.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import com.biz.crm.mdm.business.product.level.sdk.dto.RelateProductLevelCodeQueryDto;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.local.constant.ProductConstant;
import com.biz.crm.mdm.business.product.local.entity.*;
import com.biz.crm.mdm.business.product.local.repository.ProductRepository;
import com.biz.crm.mdm.business.product.local.service.*;
import com.biz.crm.mdm.business.product.local.strategy.DisplayProductStrategy;
import com.biz.crm.mdm.business.product.sdk.contanst.DisplayProductConstant;
import com.biz.crm.mdm.business.product.sdk.dto.*;
import com.biz.crm.mdm.business.product.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.sdk.enums.MediaTypeEnum;
import com.biz.crm.mdm.business.product.sdk.enums.ProductExtInfoTypeEnum;
import com.biz.crm.mdm.business.product.sdk.event.ProductEventListener;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.*;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.tools.DocumentationTool;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 商品sdk实现类
 *
 * @author sunx
 * @date 2021/10/13
 */
@Slf4j
@Service("productVoService")
public class ProductVoServiceImpl implements ProductVoService {

  @Autowired(required = false)
  private ProductService productService;

  @Autowired(required = false)
  private ProductMaterialService productMaterialService;

  @Autowired(required = false)
  private ProductMediaService productMediaService;

  @Autowired(required = false)
  private ProductIntroductionService productIntroductionService;

  @Autowired(required = false)
  private ProductLevelVoSdkService productLevelVoSdkService;

  @Autowired(required = false)
  private MaterialVoService materialVoService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private ProductRepository productRepository;

  @Autowired(required = false)
  private ProductUnitService productUnitService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  private DisplayProductStrategy displayProductStrategy;

  @Override
  @Transactional
  public ProductVo create(ProductDto dto) {
    this.createValidation(dto);
    Product product = this.buildProductByDto(dto);
    dto.setProductCode(product.getProductCode());
    //新增租户编号
    product.setTenantCode(TenantUtils.getTenantCode());
    this.bindProductExtInfo(dto);
    this.productService.create(product);
    ProductEventDto productEventDto = new ProductEventDto();
    productEventDto.setOriginal(null);
    ProductVo productVo =
            this.nebulaToolkitService.copyObjectByBlankList(
                    product, ProductVo.class, HashSet.class, LinkedList.class);
    productEventDto.setNewest(productVo);
    // 商品创建事件
    SerializableBiConsumer<ProductEventListener, ProductEventDto> onCreate =
            ProductEventListener::onCreate;
    this.nebulaNetEventClient.publish(productEventDto, ProductEventListener.class, onCreate);
    return this.buildByDtoAndProduct(dto, product);
  }

  @Override
  @Transactional
  public ProductVo update(ProductDto dto) {
    this.updateValidation(dto);
    List<ProductVo> current =
            this.findDetailsByIdsOrProductCodes(Collections.singletonList(dto.getId()), null);
    ProductVo current2 = current.stream().findFirst().orElse(null);
    Validate.notNull(current2, "未获取到修改的商品信息");
    ProductVo productVo =
            this.nebulaToolkitService.copyObjectByBlankList(
                    current2, ProductVo.class, HashSet.class, LinkedList.class);
    Product product = this.buildProductByDto(dto);
    this.bindProductExtInfo(dto);
    //新增租户编号
    product.setTenantCode(TenantUtils.getTenantCode());
    this.productService.update(product);
    ProductEventDto productEventDto = new ProductEventDto();
    productEventDto.setOriginal(productVo);
    ProductVo productVo2 =
            this.nebulaToolkitService.copyObjectByBlankList(
                    dto, ProductVo.class, HashSet.class, LinkedList.class);
    productEventDto.setNewest(productVo2);
    // 商品编辑事件
    //暂时屏蔽
//    SerializableBiConsumer<ProductEventListener, ProductEventDto> onUpdate =
//            ProductEventListener::onUpdate;
//    this.nebulaNetEventClient.publish(productEventDto, ProductEventListener.class, onUpdate);
    return this.buildByDtoAndProduct(dto, product);
  }

  @Override
  public List<ProductVo> findSelectByKeyword(String keyword) {
    Pageable pageable = PageRequest.of(0, 20);
    ProductPaginationDto dto = new ProductPaginationDto();
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    dto.setKeyword(keyword);
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<Product> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Product> pageResult = this.productRepository.findByConditions(page, dto);
    if (Objects.isNull(pageResult) || CollectionUtils.isEmpty(pageResult.getRecords())) {
      return Lists.newLinkedList();
    }
    List<ProductVo> list = Lists.newArrayList();
    for (Product item : pageResult.getRecords()) {
      ProductVo cur = new ProductVo();
      cur.setProductCode(item.getProductCode());
      cur.setProductName(item.getProductName());
      list.add(cur);
    }
    return list;
  }

  @Override
  public MaterialProductVo findMaterialProductVoByMaterialCode(String materialCode) {
    if (StringUtils.isBlank(materialCode)) {
      return null;
    }
    Set<String> productCodeSet =
            this.productMaterialService.findProductCodeByMaterialCode(materialCode);
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return null;
    }
    MaterialProductVo re = new MaterialProductVo();
    re.setProductCodeSet(productCodeSet);
    Set<String> productParentLevelCodeSet =
            this.findParentLevelCodeSetByProductCodes(productCodeSet);
    re.setProductLevelCodeSet(productParentLevelCodeSet);
    return re;
  }

  @Override
  public List<ProductVo> findByProductLevelCodes(List<String> productLevelCodeList) {
    if (CollectionUtils.isEmpty(productLevelCodeList)) {
      return Lists.newLinkedList();
    }
    List<ProductVo> productVos = this.productRepository.findByProductLevelCodes(productLevelCodeList);
    if (CollectionUtils.isEmpty(productVos)) {
      return Lists.newLinkedList();
    }

    return findDetailsByIdsOrProductCodes(null,
            productVos.stream().map(ProductVo::getProductCode).collect(Collectors.toList()));
  }

  @Override
  public Set<String> findParentLevelCodeSetByProductCodes(Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return Sets.newHashSet();
    }
    Set<String> productLevelCodeSet =
            this.productService.findProductLevelCodeSetByProductCodes(productCodeSet);
    if (CollectionUtils.isEmpty(productLevelCodeSet)) {
      return Sets.newHashSet();
    }
    Map<String, List<ProductLevelVo>> map =
            this.productLevelVoSdkService.findCurAndParentByCodes(
                    Lists.newArrayList(productLevelCodeSet));
    if (map.isEmpty()) {
      return Sets.newHashSet();
    }
    Set<String> re = Sets.newHashSet();
    for (Entry<String, List<ProductLevelVo>> item : map.entrySet()) {
      for (ProductLevelVo sub : item.getValue()) {
        re.add(sub.getProductLevelCode());
      }
    }
    return re;
  }

  @Override
  public List<ProductVo> findByProductQueryDto(ProductQueryDto dto) {
    Boolean f =
            Objects.isNull(dto)
                    || (StringUtils.isAllBlank(
                    dto.getProductCode(),
                    dto.getProductName(),
                    dto.getProductName(),
                    dto.getIsShelf(),
                    dto.getEnableStatus())
                    && CollectionUtils.isEmpty(dto.getProductLevelCodeList())
                    && CollectionUtils.isEmpty(dto.getProductCodeList()));
    if (Boolean.TRUE.equals(f)) {
      return Lists.newLinkedList();
    }
    List<Product> list = this.productRepository.findByProductQueryDto(dto);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<ProductVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                    list, Product.class, ProductVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<ProductVo> findByQueryDto(ProductQueryDto dto) {
    return ProductVoService.super.findByQueryDto(dto);
  }

  @Override
  public List<ProductVo> findDetailsByIdsOrProductCodes(
          List<String> ids, List<String> productCodes) {
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(productCodes)) {
      return Lists.newLinkedList();
    }
    final ProductDetailQueryDto dto = new ProductDetailQueryDto();
    dto.setExInfoCodes(ProductExtInfoTypeEnum.findAllKey());
    if (CollectionUtils.isNotEmpty(ids)) {
      dto.setCodeQueryFlag(false);
      dto.setCodes(ids);
    } else if (CollectionUtils.isNotEmpty(productCodes)) {
      dto.setCodeQueryFlag(true);
      dto.setCodes(productCodes);
    }
    return this.findDetailsByProductDetailQueryDto(dto);
  }

  @Override
  public List<ProductVo> findDetailsByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Lists.newLinkedList();
    }
    final ProductDetailQueryDto dto = new ProductDetailQueryDto();
    dto.setExInfoCodes(ProductExtInfoTypeEnum.findAllKey());
    dto.setCodeQueryFlag(false);
    dto.setCodes(ids);
    return this.findDetailsByProductDetailQueryDto(dto);
  }

  @Override
  public List<ProductVo> findDetailsByProductCodes(List<String> productCodes) {
    if (CollectionUtils.isEmpty(productCodes)) {
      return Lists.newLinkedList();
    }
    final ProductDetailQueryDto dto = new ProductDetailQueryDto();
    dto.setExInfoCodes(ProductExtInfoTypeEnum.findAllKey());
    dto.setCodeQueryFlag(true);
    dto.setCodes(productCodes);
    return this.findDetailsByProductDetailQueryDto(dto);
  }

  @Override
  public List<ProductVo> findByCodes(List<String> codes) {
    if (CollectionUtils.isEmpty(codes)) {
      return Lists.newArrayList();
    }
    List<Product> productList = productRepository.findByCodes(codes);
    if (CollectionUtils.isEmpty(productList)) {
      return Lists.newArrayList();
    }
    return (List<ProductVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            productList, Product.class, ProductVo.class, LinkedHashSet.class, ArrayList.class
    );
  }

  @Override
  public List<ProductVo> findByProductCodes(List<String> productCodes) {
    if (CollectionUtils.isEmpty(productCodes)) {
      return Lists.newLinkedList();
    }
    final ProductDetailQueryDto dto = new ProductDetailQueryDto();
    dto.setExInfoCodes(ProductExtInfoTypeEnum.findAllKey());
    dto.setCodeQueryFlag(true);
    dto.setCodes(productCodes);
    return this.findDetailsByProductDetailQueryDto(dto);
  }

  @Override
  public List<ProductVo> findMainDetailsByProductCodes(List<String> productCodeList) {
    if (CollectionUtils.isEmpty(productCodeList)) {
      return Lists.newLinkedList();
    }
    final ProductDetailQueryDto dto = new ProductDetailQueryDto();
    dto.setExInfoCodes(Sets.newHashSet(ProductExtInfoTypeEnum.PRODUCT_LEVEL.getDictCode()));
    dto.setCodeQueryFlag(true);
    dto.setCodes(productCodeList);
    return this.findDetailsByProductDetailQueryDto(dto);
  }

  @Override
  public List<ProductVo> findDetailsByProductDetailQueryDto(ProductDetailQueryDto dto) {
    if (Objects.isNull(dto) || CollectionUtils.isEmpty(dto.getCodes())) {
      return Lists.newLinkedList();
    }
    List<String> ids = Lists.newLinkedList();
    List<String> productCodes = Lists.newLinkedList();
    if (Boolean.TRUE.equals(dto.getCodeQueryFlag())) {
      productCodes = dto.getCodes();
    } else {
      ids = dto.getCodes();
    }
    List<Product> productList = productService.findDetailsByIdsOrProductCodes(ids, productCodes);
    if (CollectionUtils.isEmpty(productList)) {
      return Lists.newLinkedList();
    }
    List<ProductVo> list =
            (List<ProductVo>)
                    this.nebulaToolkitService.copyCollectionByBlankList(
                            productList, Product.class, ProductVo.class, HashSet.class, ArrayList.class);
    Set<String> productCodeSet =
            list.stream()
                    .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
                    .map(ProductVo::getProductCode)
                    .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(dto.getExInfoCodes()) || CollectionUtils.isEmpty(productCodeSet)) {
      return list;
    }
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.PRODUCT_LEVEL.getDictCode())) {
      this.findProductLevelInfo(list);
    }
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.MATERIAL.getDictCode())) {
      this.findProductMaterialInfo(list, productCodeSet);
    }
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.PICTURE.getDictCode())
            || dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.VIDEO.getDictCode())) {
      this.findProductMediaInfo(list, productCodeSet);
    }
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.INTRO.getDictCode())) {
      this.findProductIntroInfo(list, productCodeSet);
    }
    log.info("查询条件==========" + String.join(",", dto.getExInfoCodes()));
    if (dto.getExInfoCodes().contains(ProductExtInfoTypeEnum.UNIT.getDictCode())) {
      this.findProductUnitInfo(list, productCodeSet);
    }
    return list;
  }

  private void findProductUnitInfo(List<ProductVo> list, Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(productCodeSet)) {
      return;
    }
    List<ProductUnit> unitList =
            productUnitService.findByProductCodes(Lists.newArrayList(productCodeSet));
    if (CollectionUtils.isEmpty(unitList)) {
      return;
    }
    List<ProductUnitVo> voList =
            (List<ProductUnitVo>)
                    this.nebulaToolkitService.copyCollectionByWhiteList(
                            unitList,
                            ProductUnit.class,
                            ProductUnitVo.class,
                            HashSet.class,
                            ArrayList.class);
    Map<String, List<ProductUnitVo>> listMap =
            voList.stream()
                    .collect(Collectors.groupingBy(ProductUnitVo::getProductCode));
    for (ProductVo item : list) {
      item.setUnitList(listMap.get(item.getProductCode()));
    }
  }

  @Override
  public Map<String, String> findAllowSaleProductByProductLevelCodes(
          Set<String> productLevelCodes) {
    if (CollectionUtils.isEmpty(productLevelCodes)) {
      return Maps.newHashMap();
    }
    final RelateProductLevelCodeQueryDto queryDto = new RelateProductLevelCodeQueryDto();
    queryDto.setProductLevelCodeSet(productLevelCodes);
    queryDto.setSearchType(-1);
    final Map<String, String> productLevelRuleMap =
            this.productLevelVoSdkService.findByRelateProductLevelCodeQueryDto(queryDto);
    if (productLevelRuleMap.isEmpty()) {
      return Maps.newHashMap();
    }
    final ProductQueryDto productQueryDto = new ProductQueryDto();
    productQueryDto.setProductLevelCodeList(Lists.newArrayList(productLevelCodes));
    productQueryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    productQueryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    productQueryDto.setIsShelf(IsShelfEnum.UP.getCode());
    productQueryDto.setTenantCode(TenantUtils.getTenantCode());
    List<ProductVo> list = this.findByProductQueryDto(productQueryDto);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, String> map =
            list.stream()
                    .filter(
                            a ->
                                    StringUtils.isNoneBlank(a.getProductCode(), a.getProductLevelCode())
                                            && productLevelRuleMap.keySet().contains(a.getProductLevelCode()))
                    .collect(
                            Collectors.toMap(
                                    ProductVo::getProductCode, ProductVo::getProductLevelCode, (a, b) -> a));
    Map<String, String> re = Maps.newHashMap();
    for (Entry<String, String> item : map.entrySet()) {
      final String s = productLevelRuleMap.get(item.getValue());
      if (StringUtils.isBlank(s)) {
        continue;
      }
      re.put(item.getKey(), s);
    }
    return re;
  }

  @Override
  public Page<ProductVo> findByConditions(Pageable pageable, ProductPaginationDto dto) {
//    Page<Product> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
//    Page<Product> pageResult = this.productRepository.findByConditions(page, dto);
//    Page<ProductVo> productVoPage = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
//    if (CollectionUtils.isEmpty(pageResult.getRecords())) {
//      return productVoPage;
//    }
//    List<ProductVo> productVos = (List<ProductVo>) this.nebulaToolkitService.copyCollectionByBlankList(pageResult.getRecords(), Product.class, ProductVo.class, HashSet.class, ArrayList.class);
//    productVoPage.setRecords(productVos);
//    return productVoPage;

    Page<ProductVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.productRepository.findVoByConditions(page, dto);
  }

  /**
   * 条件查询
   *
   * @param dto
   * @return java.util.List<com.biz.crm.mdm.business.product.sdk.vo.ProductVo>
   * @author huojia
   * @date 2022/12/2 17:17
   **/
  @Override
  public List<ProductVo> queryCondition(ProductDto dto) {
    if (Objects.isNull(dto)) {
      return Lists.newArrayList();
    }
    List<Product> productList = this.productRepository.queryCondition(dto);
    if (CollectionUtils.isEmpty(productList)) {
      return Lists.newArrayList();
    }
    return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(
            productList, Product.class, ProductVo.class, HashSet.class, ArrayList.class)
    );
  }

  /**
   * 条件查询
   *
   * @param unitTypeCode
   * @return
   */
  @Override
  public ProductVo finByUnitTypeCode(String unitTypeCode) {
    if (Objects.isNull(unitTypeCode)) {
      return new ProductVo();
    }
    //根据商品编码查询商品信息
    Product product = this.productRepository.finByUnitTypeCode(unitTypeCode);
    if (Objects.isNull(product)) {
      return new ProductVo();
    }
    ProductVo productResult = this.nebulaToolkitService.copyObjectByWhiteList(product,
            ProductVo.class,
            HashSet.class,
            ArrayList.class);
    return productResult;
  }

  /**
   * 根据商品编码集合获取对应的商品数据信息
   *
   * @param productCode   商品编码
   * @param customerCode
   * @param businessModel
   * @param startDate
   */
  @Override
  public ProductPlanningDetailsVo findByProductPlanningDetails(String productCode, String customerCode, String businessModel, String startDate) {
    return productService.findByProductPlanningDetails(productCode, customerCode, businessModel, startDate);
  }

  /**
   * 根据商品编码集合获取商品集合
   *
   * @return List<String>
   */
  @Override
  public List<String> findAllProductCode() {
    return this.productRepository.findAllProductCode(TenantUtils.getTenantCode());
  }

  /**
   * 根据商品编码集合获取商品集合
   *
   * @param codes 产品编码
   * @return Map<String, String>
   */
  @Override
  public Map<String, String> getProductByCodes(List<List<String>> codes) {
    Map<String, String> map = new HashMap<>();
    if (CollectionUtils.isEmpty(codes)) {
      return map;
    }
    List<CommonSelectVo> list = this.productRepository.getProductByCodes(codes, TenantUtils.getTenantCode());
    if (!CollectionUtils.isEmpty(list)) {
      map = list.stream().collect(Collectors.toMap(CommonSelectVo::getCode, CommonSelectVo::getValue));
    }
    return map;
  }

  /**
   * 保存商品关联的物料图片视频以及介绍信息
   *
   * @param dto
   */
  private void bindProductExtInfo(ProductSynchronousUpdateDto dto, String productCode) {
    /*
     * 保存商品关联的物料图片视频以及介绍信息：
     * 1、保存关联物料
     * 2、保存图片视频信息
     * 3、保存商品介绍信息
     * */
    Validate.notNull(dto, "商品信息缺失");


    List<ProductMedia> productMediaList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getPictureMediaList())) {
      productMediaList.addAll(
              this.nebulaToolkitService.copyCollectionByWhiteList(
                      dto.getPictureMediaList(),
                      ProductMediaDto.class,
                      ProductMedia.class,
                      HashSet.class,
                      ArrayList.class));
      productMediaList.forEach(a -> a.setProductCode(productCode));
      //新增租户编号
      productMediaList.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
    }

    if (CollectionUtils.isNotEmpty(dto.getVideoMediaList())) {
      productMediaList.addAll(
              this.nebulaToolkitService.copyCollectionByWhiteList(
                      dto.getVideoMediaList(),
                      ProductMediaDto.class,
                      ProductMedia.class,
                      HashSet.class,
                      ArrayList.class));
      productMediaList.forEach(a -> a.setProductCode(productCode));
      //新增租户编号
      productMediaList.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
    }
    productMediaService.saveBatch(productMediaList, productCode);

    ProductIntroduction productIntroduction = null;
    if (Objects.nonNull(dto.getIntroduction())) {
      productIntroduction =
              this.nebulaToolkitService.copyObjectByWhiteList(
                      dto.getIntroduction(), ProductIntroduction.class, HashSet.class, ArrayList.class);
      productIntroduction.setProductCode(productCode);
      //新增租户编号
      productIntroduction.setTenantCode(TenantUtils.getTenantCode());
    }
    productIntroductionService.saveIntroduction(productIntroduction, productCode);

    List<ProductUnit> unitLinkedList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getUnitList())) {
      unitLinkedList.addAll(
              this.nebulaToolkitService.copyCollectionByWhiteList(
                      dto.getUnitList(),
                      ProductUnitDto.class,
                      ProductUnit.class,
                      HashSet.class,
                      ArrayList.class));
      unitLinkedList.forEach(a -> {
        a.setProductCode(productCode);
        a.setTenantCode(TenantUtils.getTenantCode());
      });
      productUnitService.saveBatch(unitLinkedList);
    }
  }

  /**
   * 保存商品关联的物料图片视频以及介绍信息
   *
   * @param dto
   */
  private void bindProductExtInfo(ProductDto dto) {
    /*
     * 保存商品关联的物料图片视频以及介绍信息：
     * 1、保存关联物料
     * 2、保存图片视频信息
     * 3、保存商品介绍信息
     * */
    Validate.notNull(dto, "商品信息缺失");
    List<ProductMaterial> productMaterialList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getMaterialList())) {
      productMaterialList =
              (List<ProductMaterial>)
                      this.nebulaToolkitService.copyCollectionByWhiteList(
                              dto.getMaterialList(),
                              ProductMaterialDto.class,
                              ProductMaterial.class,
                              HashSet.class,
                              ArrayList.class);
      productMaterialList.forEach(a -> a.setProductCode(dto.getProductCode()));
      //新增租户编号
      productMaterialList.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
    }
    productMaterialService.saveBatch(productMaterialList, dto.getProductCode());

    List<ProductMedia> productMediaList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getPictureMediaList())) {
      Collection<ProductMedia> productMedia = this.nebulaToolkitService.copyCollectionByWhiteList(
              dto.getPictureMediaList(),
              ProductMediaDto.class,
              ProductMedia.class,
              HashSet.class,
              ArrayList.class);
      productMedia.forEach(a -> a.setProductCode(dto.getProductCode()));
      //新增租户编号
      productMedia.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
      productMedia.forEach(a -> a.setType(MediaTypeEnum.PICTURE.getCode()));
      productMediaList.addAll(productMedia);

    }

    if (CollectionUtils.isNotEmpty(dto.getMajorPictureMedia())) {
      Collection<ProductMedia> productMedia = this.nebulaToolkitService.copyCollectionByWhiteList(
              dto.getMajorPictureMedia(),
              ProductMediaDto.class,
              ProductMedia.class,
              HashSet.class,
              ArrayList.class);
      productMedia.forEach(a -> a.setProductCode(dto.getProductCode()));
      //新增租户编号
      productMedia.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
      productMedia.forEach(a -> a.setType(MediaTypeEnum.MAJOR_PICTURE.getCode()));
      productMediaList.addAll(productMedia);
    }

    if (CollectionUtils.isNotEmpty(dto.getVideoMediaList())) {
      Collection<ProductMedia> productMedia = this.nebulaToolkitService.copyCollectionByWhiteList(
              dto.getVideoMediaList(),
              ProductMediaDto.class,
              ProductMedia.class,
              HashSet.class,
              ArrayList.class);
      productMedia.forEach(a -> a.setProductCode(dto.getProductCode()));
      //新增租户编号
      productMedia.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
      productMedia.forEach(a -> a.setType(MediaTypeEnum.VIDEO.getCode()));
      productMediaList.addAll(productMedia);
    }
    productMediaService.saveBatch(productMediaList, dto.getProductCode());

    ProductIntroduction productIntroduction = null;
    if (Objects.nonNull(dto.getIntroduction())) {
      productIntroduction =
              this.nebulaToolkitService.copyObjectByWhiteList(
                      dto.getIntroduction(), ProductIntroduction.class, HashSet.class, ArrayList.class);
      productIntroduction.setProductCode(dto.getProductCode());
      //新增租户编号
      productIntroduction.setTenantCode(TenantUtils.getTenantCode());
    }
    productIntroductionService.saveIntroduction(productIntroduction, dto.getProductCode());

    List<ProductUnit> unitLinkedList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getUnitList())) {
      unitLinkedList.addAll(
              this.nebulaToolkitService.copyCollectionByWhiteList(
                      dto.getUnitList(),
                      ProductUnitDto.class,
                      ProductUnit.class,
                      HashSet.class,
                      ArrayList.class));
      unitLinkedList.forEach(a -> {
        a.setId(null);
        a.setProductCode(dto.getProductCode());
        a.setTenantCode(TenantUtils.getTenantCode());
      });
      productUnitService.saveBatch(unitLinkedList);
    }
  }

  /**
   * 构建需要落库的商品信息
   *
   * @param dto
   * @return
   */
  private Product buildProductByDto(ProductDto dto) {
    Product product =
            this.nebulaToolkitService.copyObjectByWhiteList(
                    dto, Product.class, HashSet.class, ArrayList.class);
    product.setMaintenancePicture(CollectionUtils.isNotEmpty(dto.getPictureMediaList()));
    product.setMaintenanceIntroduction(Objects.nonNull(dto.getIntroduction()));
    if (StringUtils.isBlank(product.getProductCode())) {
      String ruleCode = StringUtils.join(ProductConstant.PRODUCT_CODE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
      String productCode = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS).get(0);
      product.setProductCode(productCode);
    }
    if (CollectionUtils.isEmpty(dto.getPictureMediaList())) {
      return product;
    }
    Optional<ProductMediaDto> first = dto.getPictureMediaList().stream().findFirst();
    if (first.isPresent()) {
      product.setPrimaryPictureUrl(first.get().getId());
    }
    return product;
  }

  /**
   * dto转vo
   *
   * @param dto
   * @param product
   * @return
   */
  private ProductVo buildByDtoAndProduct(ProductDto dto, Product product) {
    ProductVo vo =
            this.nebulaToolkitService.copyObjectByWhiteList(
                    product, ProductVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(dto.getMaterialList())) {
      vo.setMaterialList(
              (List<ProductMaterialVo>)
                      this.nebulaToolkitService.copyCollectionByWhiteList(
                              dto.getMaterialList(),
                              ProductMaterialDto.class,
                              ProductMaterialVo.class,
                              HashSet.class,
                              ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getMajorPictureMedia())) {
      vo.setMajorPictureMedia(
              (List<ProductMediaVo>)
                      this.nebulaToolkitService.copyCollectionByWhiteList(
                              dto.getMajorPictureMedia(),
                              ProductMediaDto.class,
                              ProductMediaVo.class,
                              HashSet.class,
                              ArrayList.class));
    }

    if (CollectionUtils.isNotEmpty(dto.getPictureMediaList())) {
      vo.setPictureMediaList(
              (List<ProductMediaVo>)
                      this.nebulaToolkitService.copyCollectionByWhiteList(
                              dto.getPictureMediaList(),
                              ProductMediaDto.class,
                              ProductMediaVo.class,
                              HashSet.class,
                              ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getVideoMediaList())) {
      vo.setVideoMediaList(
              (List<ProductMediaVo>)
                      this.nebulaToolkitService.copyCollectionByWhiteList(
                              dto.getVideoMediaList(),
                              ProductMediaDto.class,
                              ProductMediaVo.class,
                              HashSet.class,
                              ArrayList.class));
    }
    if (Objects.nonNull(dto.getIntroduction())) {
      vo.setIntroduction(
              this.nebulaToolkitService.copyObjectByWhiteList(
                      dto.getIntroduction(), ProductIntroductionVo.class, HashSet.class, ArrayList.class));
    }
    return vo;
  }

  /**
   * 获取商品的层级信息
   *
   * @param list
   */
  private void findProductLevelInfo(List<ProductVo> list) {
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    Set<String> productLevelCodeSet =
            list.stream()
                    .filter(a -> StringUtils.isNotBlank(a.getProductLevelCode()))
                    .map(ProductVo::getProductLevelCode)
                    .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(productLevelCodeSet)) {
      return;
    }
    List<ProductLevelVo> productLevelVoList =
            this.productLevelVoSdkService.findListByCodes(Lists.newArrayList(productLevelCodeSet));
    // k-productLevelCode,v-productLevelName
    Map<String, String> mapLevel =
            productLevelVoList.stream()
                    .filter(a -> StringUtils.isNoneBlank(a.getProductLevelCode(), a.getProductLevelName()))
                    .collect(
                            Collectors.toMap(
                                    ProductLevelVo::getProductLevelCode,
                                    ProductLevelVo::getProductLevelName,
                                    (a, b) -> a));
    for (ProductVo item : list) {
      item.setProductLevelName(mapLevel.get(item.getProductLevelCode()));
    }
  }

  /**
   * 获取商品物料信息
   *
   * @param list
   * @param productCodeSet
   */
  private void findProductMaterialInfo(List<ProductVo> list, Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(productCodeSet)) {
      return;
    }
    List<ProductMaterial> materialList =
            productMaterialService.findByProductCodes(Lists.newArrayList(productCodeSet));
    if (CollectionUtils.isEmpty(materialList)) {
      return;
    }
    Set<String> materialCodes =
            materialList.stream().map(ProductMaterial::getMaterialCode).collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(materialCodes)) {
      return;
    }
    // 获取物料表信息
    List<MaterialVo> materialVoList = materialVoService.findDetailByMaterialCodes(materialCodes);
    if (CollectionUtils.isEmpty(materialVoList)) {
      return;
    }
    Map<String, MaterialVo> map =
            materialVoList.stream()
                    .filter(a -> StringUtils.isNotBlank(a.getMaterialCode()))
                    .collect(
                            Collectors.toMap(MaterialVo::getMaterialCode, Function.identity(), (a, b) -> a));
    List<ProductMaterialVo> voList =
            (List<ProductMaterialVo>)
                    this.nebulaToolkitService.copyCollectionByBlankList(
                            materialList,
                            ProductMaterial.class,
                            ProductMaterialVo.class,
                            HashSet.class,
                            ArrayList.class);
    for (ProductMaterialVo v : voList) {
      if (!map.containsKey(v.getMaterialCode())) {
        continue;
      }
      MaterialVo materialVo = map.get(v.getMaterialCode());
      v.setProductLevelCode(materialVo.getProductLevelCode());
      v.setProductLevelName(materialVo.getProductLevelName());
      v.setMaterialType(materialVo.getMaterialType());
      v.setMaterialTypeName(materialVo.getMaterialTypeName());
      v.setUnitTypeCode(materialVo.getUnitTypeCode());
    }
    Map<String, List<ProductMaterialVo>> mapProductMaterial =
            voList.stream()
                    .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
                    .collect(Collectors.groupingBy(ProductMaterialVo::getProductCode));
    for (ProductVo item : list) {
      item.setMaterialList(mapProductMaterial.get(item.getProductCode()));
    }
  }

  /**
   * 获取商品图片视频信息
   *
   * @param list
   * @param productCodeSet
   */
  private void findProductMediaInfo(List<ProductVo> list, Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(productCodeSet)) {
      return;
    }
    List<ProductMedia> mediaList =
            productMediaService.findByProductCodes(Lists.newArrayList(productCodeSet));

    if (CollectionUtils.isEmpty(mediaList)) {
      return;
    }
    List<ProductMediaVo> voList =
            (List<ProductMediaVo>)
                    this.nebulaToolkitService.copyCollectionByWhiteList(
                            mediaList,
                            ProductMedia.class,
                            ProductMediaVo.class,
                            HashSet.class,
                            ArrayList.class);
    Map<String, List<ProductMediaVo>> map =
            voList.stream().collect(Collectors.groupingBy(ProductMediaVo::getType));
    Map<String, List<ProductMediaVo>> mapPictureMedia =
            map.getOrDefault(MediaTypeEnum.PICTURE.getCode(), Lists.newArrayList()).stream()
                    .collect(Collectors.groupingBy(ProductMediaVo::getProductCode));
    Map<String, List<ProductMediaVo>> mapVideoMedia =
            map.getOrDefault(MediaTypeEnum.VIDEO.getCode(), Lists.newArrayList()).stream()
                    .collect(Collectors.groupingBy(ProductMediaVo::getProductCode));
    Map<String, List<ProductMediaVo>> mapmajotMedia =
            map.getOrDefault(MediaTypeEnum.MAJOR_PICTURE.getCode(), Lists.newArrayList()).stream()
                    .collect(Collectors.groupingBy(ProductMediaVo::getProductCode));
    for (ProductVo item : list) {
      item.setPictureMediaList(mapPictureMedia.get(item.getProductCode()));
      item.setVideoMediaList(mapVideoMedia.get(item.getProductCode()));
      item.setMajorPictureMedia(mapmajotMedia.get(item.getProductCode()));
    }
  }

  /**
   * 获取商品详情信息
   *
   * @param list
   * @param productCodeSet
   */
  private void findProductIntroInfo(List<ProductVo> list, Set<String> productCodeSet) {
    if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(productCodeSet)) {
      return;
    }
    List<ProductIntroduction> introductionList =
            productIntroductionService.findByProductCodes(Lists.newArrayList(productCodeSet));
    if (CollectionUtils.isEmpty(introductionList)) {
      return;
    }
    List<ProductIntroductionVo> voList =
            (List<ProductIntroductionVo>)
                    this.nebulaToolkitService.copyCollectionByWhiteList(
                            introductionList,
                            ProductIntroduction.class,
                            ProductIntroductionVo.class,
                            HashSet.class,
                            ArrayList.class);
    Map<String, ProductIntroductionVo> map =
            voList.stream()
                    .collect(
                            Collectors.toMap(
                                    ProductIntroductionVo::getProductCode, Function.identity(), (a, b) -> a));
    for (ProductVo item : list) {
      item.setIntroduction(map.get(item.getProductCode()));
    }
  }

  private void createValidation(ProductDto dto) {
    this.validation(dto);
  }

  private void updateValidation(ProductDto dto) {
    Validate.notNull(dto, "商品信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(dto.getId()), "商品id不能为空");
    this.validation(dto);
  }

  private void updateValidation(ProductSynchronousUpdateDto dto) {
    Validate.notNull(dto, "商品信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(dto.getId()), "商品id不能为空");
  }

  private void validation(ProductDto dto) {
    Validate.notNull(dto, "商品信息缺失");
    // 商品编码前端如果不填，后台自动生成
    if (StringUtils.isNotBlank(dto.getProductCode())) {
      Product byProductCode = productService.findByProductCode(dto.getProductCode());
      Validate.isTrue(
              Objects.isNull(byProductCode) ||
                      (StringUtils.isNotBlank(dto.getId()) ? dto.getId().equals(byProductCode.getId()) : Objects.isNull(byProductCode)), "商品编码重复");
    }
//    Validate.notBlank(dto.getProductType(), "商品类型不能为空");
    Validate.notBlank(dto.getProductName(), "商品名称不能为空");
//    Validate.notBlank(dto.getIsShelf(), "上下架不能为空");
//    Validate.notNull(dto.getBeginDateTime(), "开始时间不能为空");
//    Validate.notNull(dto.getEndDateTime(), "结束时间不能为空");
//    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getMaterialList()), "物料信息不能为空");
  }


  /**
   * 根据商品编码查询商品名称
   *
   * @param ids
   * @param productCodes
   * @return {@link Map}<{@link String}, {@link String}>
   */
  @Override
  public Map<String, String> findNameByIdsOrCodes(List<String> ids, List<String> productCodes) {
    return this.productRepository.findNameByIdsOrCodes(ids, productCodes);
  }

  @Override
  public ProductVo updateSynchronous(ProductSynchronousUpdateDto dto) {
    updateValidation(dto);
    Product product = productRepository.findById(dto.getId());
    Validate.notNull(product, "未获取到修改的商品信息");
    List<ProductVo> current =
            this.findDetailsByIdsOrProductCodes(
                    null, Collections.singletonList(product.getProductCode()));
    ProductVo current2 = current.stream().findFirst().orElse(null);
    Validate.notNull(current2, "未获取到修改的商品信息");
    ProductVo productVo =
            this.nebulaToolkitService.copyObjectByBlankList(
                    current2, ProductVo.class, HashSet.class, LinkedList.class);
    this.bindProductExtInfo(dto, productVo.getProductCode());
    BeanUtils.copyProperties(dto, product);
    this.productService.update(product);
    ProductEventDto productEventDto = new ProductEventDto();
    productEventDto.setOriginal(productVo);
    ProductVo productVo2 =
            this.nebulaToolkitService.copyObjectByBlankList(
                    dto, ProductVo.class, HashSet.class, LinkedList.class);
    productEventDto.setNewest(productVo2);
    // 商品编辑事件
    SerializableBiConsumer<ProductEventListener, ProductEventDto> onUpdate =
            ProductEventListener::onUpdate;
    this.nebulaNetEventClient.publish(productEventDto, ProductEventListener.class, onUpdate);
//    return this.buildByDtoAndProduct(dto, product);
    return null;
  }

  @Override
  public List<DisplayProductVo> findPackageDescOrBrandOrgOrCategory(DisplayProductDto displayProductDto) {
    List<DisplayProductVo> displayProductVoList = Lists.newArrayList();
    // 如果数组的长度为1，则表示只携带一个type，客户是首次访问
    if (displayProductDto.getType().length==1){
      List<DisplayProductVo> productInfos = this.findProductInfosOnce(displayProductDto);
      if (CollectionUtils.isEmpty(productInfos)) {
        return null;
      }
      displayProductVoList.addAll(productInfos);
    }else if(displayProductDto.getType().length >1) {
      // 如果数组的长度大于1，则表示携带多个type，客户是第二次访问
      List<DisplayProductVo> productInfos = this.findProductInfosTwo(displayProductDto);
      if (CollectionUtils.isEmpty(productInfos)){
        return null;
      }
      displayProductVoList.addAll(productInfos);
    }else {
      // 数组长度小于1 ，则表示未携带type，客户是第三次访问
     List<DisplayProductVo> productInfos = this.findProductInfosThree(displayProductDto);
     if (CollectionUtils.isEmpty(productInfos)) {
       return null;
      }
     displayProductVoList.addAll(productInfos);
    }
    return displayProductVoList;
  }

  /**
   * 第三次随机查询产品信息
   * @param displayProductDto
   * @return
   */
  private List<DisplayProductVo> findProductInfosThree(DisplayProductDto displayProductDto) {
    String brandOrg = displayProductDto.getBrandOrg();
    String productCategoryCode = displayProductDto.getProductCategoryCode();
    String packageDesc = displayProductDto.getPackageDesc();
    List<DisplayProductVo> result = Lists.newArrayList();
    // 基于品牌组，品类编码查询物料包装物料信息
    if (StringUtils.isNotEmpty(brandOrg)
        && StringUtils.isNotEmpty(productCategoryCode)
        && StringUtils.isEmpty(packageDesc)) {
      List<DisplayProductVo> packageDescList = this.displayProductStrategy.findPackageDesc(
          displayProductDto);
      if (CollectionUtils.isEmpty(packageDescList)){
        return null;
      }
      result.addAll(packageDescList);
    }
    // 基于品牌组，物料包装物料查询品类名称信息
    if (StringUtils.isNotEmpty(brandOrg)
        && StringUtils.isNotEmpty(packageDesc)
        && StringUtils.isEmpty(productCategoryCode)) {
      List<DisplayProductVo> categoryList = this.displayProductStrategy.findCategoryName(
          displayProductDto);
      if (CollectionUtils.isEmpty(categoryList)){
        return null;
      }
      result.addAll(categoryList);
    }
    // 基于物料包装物料，品类编码查询品牌组信息
    if (StringUtils.isNotEmpty(packageDesc)
        && StringUtils.isNotEmpty(productCategoryCode)
        && StringUtils.isEmpty(brandOrg)) {
      List<DisplayProductVo> brandOrgList = this.displayProductStrategy.findBrandOrg(
          displayProductDto);
      if (CollectionUtils.isEmpty(brandOrgList)){
        return null;
      }
      result.addAll(brandOrgList);
    }
    return result;
  }

  /**
   * 第二次随机查询产品信息
   * @param dto
   * @return
   */
  private List<DisplayProductVo> findProductInfosTwo(DisplayProductDto dto) {
    Validate.notNull(dto.getType(), "二次查询产品信息时，type不能为空！");
    DisplayProductDto displayProductDto = new DisplayProductDto();
    displayProductDto.setType(dto.getType());
    List<DisplayProductVo> result = Lists.newArrayList();
    for (String type : displayProductDto.getType()){
      // 首次访问的品牌组 二次访问的是品类名称
      if (type.equals(DisplayProductConstant.BRAND_ORG+DisplayProductConstant.PRODUCT_CATEGORY)){
        displayProductDto.setBrandOrg(dto.getBrandOrg());
        Validate.notNull(displayProductDto.getBrandOrg(), "选择品类名称时品牌组不能为空！");
        List<DisplayProductVo> categories = this.displayProductStrategy.findCategoryName(
            displayProductDto);
        if (CollectionUtils.isEmpty(categories)) {
          return null;
        }
        result.addAll(categories);
      }
      // 首次访问的品牌组 二次访问的是物料包装物料
      if (type.equals(DisplayProductConstant.BRAND_ORG+DisplayProductConstant.PACKAGE_DESC)){
        displayProductDto.setBrandOrg(dto.getBrandOrg());
        Validate.notNull(displayProductDto.getBrandOrg(), "选择物料包装物料时品牌组不能为空！");
        List<DisplayProductVo> packageDescList = this.displayProductStrategy.findPackageDesc(
            displayProductDto);
        if (CollectionUtils.isEmpty(packageDescList)) {
          return null;
        }
        result.addAll(packageDescList);
      }
      // 首次访问的品类名称 二次访问的是品牌组
      if (type.equals(DisplayProductConstant.PRODUCT_CATEGORY+DisplayProductConstant.BRAND_ORG)) {
        displayProductDto.setProductCategoryCode(dto.getProductCategoryCode());
        Validate.notNull(displayProductDto.getProductCategoryCode(), "选择品牌组时品类编码不能为空！");
        List<DisplayProductVo> brandOrgList = this.displayProductStrategy.findBrandOrg(
            displayProductDto);
        if (CollectionUtils.isEmpty(brandOrgList)){
          return null;
        }
        result.addAll(brandOrgList);
      }
      // 首次访问的品类名称 二次访问的是物料包装物料
      if (type.equals(DisplayProductConstant.PRODUCT_CATEGORY+DisplayProductConstant.PACKAGE_DESC)){
        displayProductDto.setProductCategoryCode(dto.getProductCategoryCode());
        Validate.notNull(displayProductDto.getProductCategoryCode(), "选择物料包装物料时品类编码不能为空！");
        List<DisplayProductVo> packageDescList = this.displayProductStrategy.findPackageDesc(
            displayProductDto);
        if (CollectionUtils.isEmpty(packageDescList)){
          return null;
        }
        result.addAll(packageDescList);
      }
      // 首次访问的物料包装物料 二次访问的是品牌组
      if (type.equals(DisplayProductConstant.PACKAGE_DESC+DisplayProductConstant.BRAND_ORG)){
        displayProductDto.setPackageDesc(dto.getPackageDesc());
        Validate.notNull(displayProductDto.getPackageDesc(), "选择品牌组时物料包装物料不能为空！");
        List<DisplayProductVo> brandOrgList = this.displayProductStrategy.findBrandOrg(
            displayProductDto);
        if (CollectionUtils.isEmpty(brandOrgList)) {
          return null;
        }
        result.addAll(brandOrgList);
      }
      // 首次访问的物料包装物料 二次访问的是品类名称
      if (type.equals(DisplayProductConstant.PACKAGE_DESC+DisplayProductConstant.PRODUCT_CATEGORY)){
        if (StringUtils.isEmpty(displayProductDto.getPackageDesc())) {
          displayProductDto.setPackageDesc(dto.getPackageDesc());
          Validate.notNull(displayProductDto.getPackageDesc(), "选择品类名称时物料包装物料不能为空！");
        }
        List<DisplayProductVo> categoryNameList = this.displayProductStrategy.findCategoryName(displayProductDto);
        if (CollectionUtils.isEmpty(categoryNameList)) {
          return null;
        }
        result.addAll(categoryNameList);
      }
    }
    return result;
  }

  /**
   * 第一次随机查询产品信息
   * @param dto
   * @return
   */
  private List<DisplayProductVo> findProductInfosOnce(DisplayProductDto dto) {
    Validate.notNull(dto.getType(), "首次查询产品信息时，type不能为空！");
    DisplayProductDto displayProductDto = new DisplayProductDto();
    displayProductDto.setType(dto.getType());
    List<DisplayProductVo> result = Lists.newArrayList();
    for (String type : displayProductDto.getType()){
      if (type.equals(DisplayProductConstant.BRAND_ORG)) {
        //客户第一次访问的是品牌组选择 查询所有品牌组信息
        List<DisplayProductVo> brandOrgList = this.displayProductStrategy.findBrandOrg(
            displayProductDto);
        if (CollectionUtils.isEmpty(brandOrgList)){
          return null;
        }
        result.addAll(brandOrgList);
      } else if (type.equals(DisplayProductConstant.PRODUCT_CATEGORY)) {
        //客户第一次访问的是品类名称选择 查询所有品类名称信息
        List<DisplayProductVo> categoryList = this.displayProductStrategy.findCategoryName(
            displayProductDto);
        if (CollectionUtils.isEmpty(categoryList)) {
          return null;
        }
        result.addAll(categoryList);
      } else if (type.equals(DisplayProductConstant.PACKAGE_DESC)) {
        //客户第一次访问的是物料包装物料选择 查询所有物料包装物料信息
        List<DisplayProductVo> packageDescList = this.displayProductStrategy.findPackageDesc(
            displayProductDto);
        if (CollectionUtils.isEmpty(packageDescList)) {
          return null;
        }
        result.addAll(packageDescList);
      }
    }
      return result;
    }

  /**
   * 单个编码查询
   * @param productCode
   * @param enableStatusFlag
   * @return {@link ProductVo}
   */
  @Override
  public ProductVo findByCode(String productCode, Boolean enableStatusFlag) {
    return this.productRepository.findByCode(productCode, enableStatusFlag);
  }

  @Override
  public Boolean validProductGroupUnion(List<ProductCodeUnitGroup> productCodeUnitGroups,
                                        List<ProductLevelUnitsGroup> productLevelUnitsGroups) {
    return null;
  }
}
