package com.biz.crm.mdm.business.product.local.strategy;

import com.biz.crm.mdm.business.product.local.entity.Product;
import com.biz.crm.mdm.business.product.local.repository.ProductRepository;
import com.biz.crm.mdm.business.product.sdk.dto.DisplayProductDto;
import com.biz.crm.mdm.business.product.sdk.vo.DisplayProductVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * @author Kevin
 * @title DisplayProductStrategyImpl
 * @date 2023/3/13 20:10
 * @description 产品关联陈列活动的策略接口实现类
 */
@Component
public class DisplayProductStrategyImpl implements DisplayProductStrategy {

  @Autowired(required = false)
  private ProductRepository productRepository;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;


  @Override
  public List<DisplayProductVo> findCategoryName(DisplayProductDto displayProductDto) {
    List<DisplayProductVo> categoryList = this.findCategoryByPackageDescOrBrandOrges(
        displayProductDto);
    if (CollectionUtils.isEmpty(categoryList)) {
      return null;
    }
    return categoryList;
  }

  @Override
  public List<DisplayProductVo> findPackageDesc(DisplayProductDto displayProductDto) {
    List<DisplayProductVo> packageDescList = this.findPackageDescByCategoryOrBrandOrg(
        displayProductDto);
    if (CollectionUtils.isEmpty(packageDescList)) {
      return null;
    }
    return packageDescList;

  }

  @Override
  public List<DisplayProductVo> findBrandOrg(DisplayProductDto displayProductDto) {
    List<DisplayProductVo> brandOrgList = this.findBrandOrgByCategoryOrPackageDesc(
        displayProductDto);
    if (CollectionUtils.isEmpty(brandOrgList)) {
      return null;
    }
    return brandOrgList;
  }

  /**
   * 基于物料包装物料或者品牌组查询品类信息
   * 两者都为空，则查询全部品类信息
   * 两者都不为空，则查询满足两者条件的品类信息
   *
   * @param displayProductDto
   * @return
   */
  private List<DisplayProductVo> findCategoryByPackageDescOrBrandOrges(
      DisplayProductDto displayProductDto) {
    /*
     * 基于物料包装物料或者品牌组查询品类信息
     * 两者都为空，则查询全部品类信息
     * 两者都不为空，则查询满足两者条件的品类信息
     */
    List<Product> productList = this.productRepository.findPackageDescOrBrandOrgOrCategory(
        displayProductDto);
    if (CollectionUtils.isEmpty(productList)) {
      return null;
    }
    productList.removeAll(Collections.singleton(null));
    // 拷贝数据
    List<DisplayProductVo> displayProducts =
        (List<DisplayProductVo>) this.nebulaToolkitService.copyCollectionByWhiteList(productList,
            Product.class, DisplayProductVo.class,
            HashSet.class, ArrayList.class);
    // 去重 (根据品类编码对象去重)
    List<DisplayProductVo> itemList = displayProducts.stream().collect(
        Collectors.collectingAndThen(
            Collectors.toCollection(() -> new TreeSet<>(
                Comparator.comparing(DisplayProductVo::getProductCategoryCode))), ArrayList::new));
    if (CollectionUtils.isEmpty(itemList)) {
      return null;
    }
    List<DisplayProductVo> displayProductVos = new ArrayList<>(itemList.size());
    itemList.stream().forEach(i->{
      DisplayProductVo displayProductVo = new DisplayProductVo();
      displayProductVo.setProductCategoryCode(i.getProductCategoryCode());
      displayProductVo.setProductCategoryName(i.getProductCategoryName());
      displayProductVos.add(displayProductVo);
    });
    return displayProductVos;
  }

  /**
   * 基于品类名称或者品牌组查询物料包装物料信息
   * 两者都不为空，则查询满足两者条件的物料包装物料信息
   * 两者都为空，则查询全部物料包装物料信息
   *
   * @param displayProductDto
   * @return
   */
  private List<DisplayProductVo> findPackageDescByCategoryOrBrandOrg(
      DisplayProductDto displayProductDto) {
    /*
     * 基于品类名称或者品牌组查询物料包装物料信息
     * 两者都不为空，则查询满足两者条件的物料包装物料信息
     * 两者都为空，则查询全部物料包装物料信息
     */
    List<Product> productList = this.productRepository.findPackageDescOrBrandOrgOrCategory(
        displayProductDto);
    if (CollectionUtils.isEmpty(productList)) {
      return null;
    }
    productList.removeAll(Collections.singleton(null));
    List<String> packageDescList = productList.stream()
        .filter(p -> p.getPackageDesc() != null)
        .map(p -> p.getPackageDesc())
        .distinct()
        .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(packageDescList)) {
      return null;
    }
    List<DisplayProductVo> displayProductVos = new ArrayList<>(packageDescList.size());
    packageDescList.stream().forEach(p -> {
      DisplayProductVo displayProductVo = new DisplayProductVo();
      displayProductVo.setPackageDesc(p);
      displayProductVos.add(displayProductVo);
    });
    return displayProductVos;
  }

  /**
   * 基于品类信息或者物料包装物料查询品牌组
   * 两者都为空，则查询全部品牌组信息
   * 两者都不空，则查询满足两者条件的品牌组信息
   *
   * @param displayProductDto
   * @return
   */
  private List<DisplayProductVo> findBrandOrgByCategoryOrPackageDesc(
      DisplayProductDto displayProductDto) {
    // 基于物料包装物料查询品牌组
    List<Product> productList = this.productRepository.findPackageDescOrBrandOrgOrCategory(
        displayProductDto);
    if (CollectionUtils.isEmpty(productList)) {
      return null;
    }
    productList.removeAll(Collections.singleton(null));
    List<String> brandOrgList = productList.stream()
        .filter(p -> StringUtils.isNotEmpty(p.getBrandOrg()))
        .map(p -> p.getBrandOrg())
        .distinct()
        .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(brandOrgList)) {
      return null;
    }
    List<DisplayProductVo> displayProductVos = new ArrayList<>(brandOrgList.size());
    brandOrgList.stream().forEach(b->{
      DisplayProductVo displayProductVo = new DisplayProductVo();
      displayProductVo.setBrandOrg(b);
      displayProductVos.add(displayProductVo);
    });
    return displayProductVos;
  }
}

