package com.biz.crm.mdm.business.product.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.product.local.entity.Product;
import com.biz.crm.mdm.business.product.local.service.ProductService;
import com.biz.crm.mdm.business.product.sdk.dto.ContractProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductPaginationDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * mdm_product 商品主数据(Product)表控制层
 *
 * @author sunx
 * @since 2021-10-13 16:48:13
 */
@Slf4j
@Api(tags = "商品管理: Product: 商品管理")
@RestController
@RequestMapping(value = {"/v1/product/product"})
public class ProductController {

  @Autowired(required = false)
  private ProductService productService;

  /**
   * 查询分页列表
   *
   * @param pageable
   * @param paginationDto
   * @return
   */
  @ApiOperation(value = "查询分页列表")
  @GetMapping(value = {"/findByConditions"})
  public Result<Page<Product>> findByConditions(
      @PageableDefault(50) Pageable pageable, ProductPaginationDto paginationDto) {
    try {
      paginationDto = Optional.ofNullable(paginationDto).orElse(new ProductPaginationDto());
      paginationDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      Page<Product> result = this.productService.findByConditions(pageable, paginationDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 获取合同配置可购清单商品数据
   *
   * @param dto 查询参数
   * @return
   */
  @ApiOperation(value = "获取合同配置可购清单商品数据")
  @GetMapping(value = {"/findByContractProductQueryDto"})
  public Result<List<Product>> findByContractProductQueryDto(ContractProductQueryDto dto) {
    try {
      List<Product> result = this.productService.findByContractProductQueryDto(dto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 逻辑删除
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "逻辑删除")
  @DeleteMapping("/delete")
  public Result<?> delete(@RequestBody List<String> ids) {
    try {
      this.productService.updateDelFlagByIds(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 上架
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "上架")
  @PatchMapping("/upShelf")
  public Result<?> upShelf(@RequestBody List<String> ids) {
    try {
      this.productService.upShelf(ids);
      return Result.ok("上架成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 下架
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "下架")
  @PatchMapping("/downShelf")
  public Result<?> downShelf(@RequestBody List<String> ids) {
    try {
      this.productService.downShelf(ids);
      return Result.ok("下架成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据商品编码上架
   *
   * @param productCodeList
   * @return
   */
  @ApiOperation(value = "根据商品编码上架")
  @PatchMapping("/upShelfByProductCodes")
  public Result<?> upShelfByProductCodes(@RequestBody List<String> productCodeList) {
    try {
      this.productService.upShelfByProductCodes(productCodeList);
      return Result.ok("上架成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据商品编码下架
   *
   * @param productCodeList
   * @return
   */
  @ApiOperation(value = "根据商品编码下架")
  @PatchMapping("/downShelfByProductCodes")
  public Result<?> downShelfByProductCodes(@RequestBody List<String> productCodeList) {
    try {
      this.productService.downShelfByProductCodes(productCodeList);
      return Result.ok("下架成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 启用
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "启用")
  @PatchMapping("/enable")
  public Result<?> enable(@RequestBody List<String> ids) {
    try {
      this.productService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 禁用
   *
   * @param ids
   * @return
   */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody List<String> ids) {
    try {
      this.productService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
