package com.biz.crm.mdm.business.product.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.product.sdk.dto.ProductDetailQueryDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductPaginationDto;
import com.biz.crm.mdm.business.product.sdk.dto.ProductQueryDto;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 商品主数据(Product)表控制层
 *
 * @author sunx
 * @since 2021-10-13 16:48:13
 */
@Slf4j
@Api(tags = "商品管理: ProductVo: 商品管理")
@RestController
@RequestMapping(value = {"/v1/product/product"})
public class ProductVoController {

  @Autowired(required = false)
  private ProductVoService productVoService;

  /**
   * 详情
   *
   * @param id
   * @return
   */
  @ApiOperation(value = "详情")
  @GetMapping(value = {"/findDetailById"})
  public Result<ProductVo> findDetailById(@RequestParam("id") String id) {
    List<String> ids = Lists.newLinkedList();
    if (StringUtils.isNotBlank(id)) {
      ids.add(id);
    }
    try {
      ProductVo vo = null;
      List<ProductVo> list = productVoService.findDetailsByIdsOrProductCodes(ids, null);
      Optional<ProductVo> optional = list.stream().findFirst();
      if (optional.isPresent()) {
        vo = optional.get();
      }
      return Result.ok(vo);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据商品商品id集合或编码集合获取商品信息
   *
   * @param ids
   * @param codes
   * @return
   */
  @ApiOperation(value = "根据商品商品id集合或编码集合获取商品信息")
  @GetMapping(value = {"/findDetailsByIdsOrCodes"})
  public Result<List<ProductVo>> findDetailsByIdsOrCodes(
      @RequestParam("ids") List<String> ids, @RequestParam("codes") List<String> codes) {
    try {
      List<ProductVo> list = productVoService.findDetailsByIdsOrProductCodes(ids, codes);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 创建
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "创建")
  @PostMapping(value = "")
  public Result<ProductVo> create(@RequestBody ProductDto dto) {
    try {
      ProductVo current = this.productVoService.create(dto);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 更新
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "更新")
  @PatchMapping(value = "")
  public Result<ProductVo> update(@RequestBody ProductDto dto) {
    try {
      ProductVo current = this.productVoService.update(dto);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据商品层级获取对应的商品信息
   *
   * @param productLevelCodeList
   * @return
   */
  @ApiOperation(value = "根据商品层级获取对应的商品信息")
  @GetMapping("/findByProductLevelCodes")
  public Result<List<ProductVo>> findByProductLevelCodes(
      @RequestParam("productLevelCodeList") List<String> productLevelCodeList) {
    try {
      return Result.ok(productVoService.findByProductLevelCodes(productLevelCodeList));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据商品商品编码集合获取商品主信息+层级信息
   *
   * @param productCodeList
   * @return
   */
  @ApiOperation(value = "根据商品商品编码集合获取商品主信息+层级信息")
  @GetMapping(value = {"/findMainDetailsByProductCodes"})
  public Result<List<ProductVo>> findMainDetailsByProductCodes(
      @RequestParam("productCodeList") List<String> productCodeList) {
    try {
      List<ProductVo> list = productVoService.findMainDetailsByProductCodes(productCodeList);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据ProductQueryDto获取商品主信息
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "根据ProductQueryDto获取商品主信息")
  @GetMapping(value = {"/findByProductQueryDto"})
  public Result<List<ProductVo>> findByProductQueryDto(ProductQueryDto dto) {
    try {
      List<ProductVo> list = productVoService.findByProductQueryDto(dto);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据ProductQueryDto获取商品主信息(针对请求数据量特别大的情况使用)
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "根据ProductQueryDto获取商品主信息")
  @PostMapping(value = {"/findByQueryDto"})
  public Result<List<ProductVo>> findByQueryDto(@RequestBody ProductQueryDto dto) {
    try {
      List<ProductVo> list = productVoService.findByProductQueryDto(dto);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据ProductDetailQueryDto获取商品信息
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "根据ProductDetailQueryDto获取商品信息")
  @GetMapping(value = {"/findDetailsByProductDetailQueryDto"})
  public Result<List<ProductVo>> findDetailsByProductDetailQueryDto(ProductDetailQueryDto dto) {
    try {
      List<ProductVo> list = productVoService.findDetailsByProductDetailQueryDto(dto);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据层级获取层级及下级所有的上架且启用的产品集合
   *
   * @param productLevelCodes
   * @return
   */
  @ApiOperation(value = "根据层级获取层级及下级所有的上架且启用的产品集合")
  @GetMapping(value = {"/findAllowSaleProductByProductLevelCodes"})
  public Result<Map<String, String>> findAllowSaleProductByProductLevelCodes(
      @RequestParam("productLevelCodes") Set<String> productLevelCodes) {
    try {
      Map<String, String> map =
          productVoService.findAllowSaleProductByProductLevelCodes(productLevelCodes);
      return Result.ok(map);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 查询分页列表
   *
   * @param pageable
   * @param paginationDto
   * @return
   */
  @ApiOperation(value = "查询分页列表")
  @GetMapping(value = {"/findPageProductVoByConditions"})
  public Result<Page<ProductVo>> findByConditions(
      @PageableDefault(50) Pageable pageable, ProductPaginationDto paginationDto) {
    try {
      paginationDto = Optional.ofNullable(paginationDto).orElse(new ProductPaginationDto());
      paginationDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      Page<ProductVo> result = this.productVoService.findByConditions(pageable, paginationDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
