package com.biz.crm.mdm.business.product.spu.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpu;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuIntroduction;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuMedia;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuRelateSku;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuTag;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuTagMapping;
import com.biz.crm.mdm.business.product.spu.local.repository.ProductSpuRepository;
import com.biz.crm.mdm.business.product.spu.local.repository.ProductSpuTagMappingRepository;
import com.biz.crm.mdm.business.product.spu.local.service.ProductSpuIntroductionService;
import com.biz.crm.mdm.business.product.spu.local.service.ProductSpuMediaService;
import com.biz.crm.mdm.business.product.spu.local.service.ProductSpuRelateSkuService;
import com.biz.crm.mdm.business.product.spu.local.service.ProductSpuService;
import com.biz.crm.mdm.business.product.spu.local.service.ProductSpuTagMappingService;
import com.biz.crm.mdm.business.product.spu.sdk.constant.ProductSpuConstant;
import com.biz.crm.mdm.business.product.spu.sdk.dto.ProductSpuPaginationDto;
import com.biz.crm.mdm.business.product.spu.sdk.dto.ProductSpuTagMappingQueryDto;
import com.biz.crm.mdm.business.product.spu.sdk.enums.IsShelfEnum;
import com.biz.crm.mdm.business.product.spu.sdk.enums.MediaTypeEnum;
import com.biz.crm.mdm.business.product.spu.sdk.event.ProductSpuEventListener;
import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 商品spu信息(ProductSpu)表服务实现类
 *
 * @author sunx
 * @date 2021-12-02 16:06:49
 */
@Slf4j
@Service("productSpuService")
public class ProductSpuServiceImpl implements ProductSpuService {

  @Autowired(required = false)
  private ProductSpuRepository productSpuRepository;

  @Autowired(required = false)
  @Lazy
  private List<ProductSpuEventListener> eventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private ProductSpuTagMappingService productSpuTagMappingService;

  @Autowired(required = false)
  private ProductSpuRelateSkuService productSpuRelateSkuService;

  @Autowired(required = false)
  private ProductSpuMediaService productSpuMediaService;

  @Autowired(required = false)
  private ProductSpuIntroductionService productSpuIntroductionService;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private ProductSpuTagMappingRepository productSpuTagMappingRepository;

  @Override
  public Page<ProductSpu> findByConditions(Pageable pageable, ProductSpuPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new ProductSpuPaginationDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<ProductSpu> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<ProductSpu> pageResult = this.productSpuRepository.findByConditions(page, dto);
    if (pageResult != null && CollectionUtils.isNotEmpty(pageResult.getRecords())) {
      List<String> spuCodeList =
          pageResult.getRecords().stream()
              .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
              .map(ProductSpu::getSpuCode)
              .collect(Collectors.toList());
      List<ProductSpuRelateSku> productSpuRelateSkuList =
          productSpuRelateSkuService.findBySpuCodes(spuCodeList);
      Map<String, Long> map = Maps.newHashMap();
      if (CollectionUtils.isNotEmpty(productSpuRelateSkuList)) {
        map =
            productSpuRelateSkuList.stream()
                .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
                .collect(
                    Collectors.groupingBy(ProductSpuRelateSku::getSpuCode, Collectors.counting()));
      }
      for (ProductSpu item : pageResult.getRecords()) {
        item.setProductQuantity(map.getOrDefault(item.getSpuCode(), 0L));
      }
    }
    return pageResult;
  }

  @Override
  public List<ProductSpu> findDetailsByIds(List<String> idList) {
    if (CollectionUtils.isEmpty(idList)) {
      return Lists.newLinkedList();
    }
    List<ProductSpu> list = this.productSpuRepository.findByIds(idList);
    return this.findDetailsExtInfo(list);
  }

  @Override
  public List<ProductSpu> findDetailsBySpuCodes(List<String> spuCodeList) {
    if (CollectionUtils.isEmpty(spuCodeList)) {
      return Lists.newLinkedList();
    }
    List<ProductSpu> list = this.productSpuRepository.findBySpuCodes(spuCodeList);
    return this.findDetailsExtInfo(list);
  }

  @Override
  @Transactional
  public ProductSpu create(ProductSpu productSpu) {
    this.createValidation(productSpu);
    productSpu.setTenantCode(TenantUtils.getTenantCode());
    productSpu.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    productSpu.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    if (StringUtils.isBlank(productSpu.getSpuCode())) {
      productSpu.setSpuCode(
          generateCodeService.generateCode(ProductSpuConstant.SPU_CODE, 1).get(0));
    } else {
      Integer count = this.productSpuRepository.countByRuleCode(productSpu.getSpuCode());
      Validate.isTrue(null == count || 1 > count, productSpu.getSpuCode() + "编码已存在");
    }
    this.productSpuRepository.saveOrUpdate(productSpu);
    this.bindExtInfo(productSpu);
    // 发送通知
    if (CollectionUtils.isEmpty(eventListeners)) {
      return productSpu;
    }
    ProductSpuVo vo =
        this.nebulaToolkitService.copyObjectByBlankList(
            productSpu, ProductSpuVo.class, HashSet.class, ArrayList.class);
    for (ProductSpuEventListener eventListener : eventListeners) {
      eventListener.onCreate(vo);
    }
    return productSpu;
  }

  @Override
  @Transactional
  public ProductSpu update(ProductSpu productSpu) {
    this.updateValidation(productSpu);
    String currentId = productSpu.getId();
    List<ProductSpu> list = productSpuRepository.findByIds(Lists.newArrayList(currentId));
    Validate.isTrue(CollectionUtils.isNotEmpty(list), "修改信息不存在");
    ProductSpu current = list.get(0);
    Validate.isTrue(productSpu.getSpuCode().equals(current.getSpuCode()), "商品spu编码不能修改");
    productSpuRepository.saveOrUpdate(productSpu);
    this.bindExtInfo(productSpu);
    // 发送修改通知
    if (CollectionUtils.isEmpty(eventListeners)) {
      return productSpu;
    }
    ProductSpuVo oldVo =
        this.nebulaToolkitService.copyObjectByBlankList(
            current, ProductSpuVo.class, HashSet.class, ArrayList.class);
    ProductSpuVo vo =
        this.nebulaToolkitService.copyObjectByBlankList(
            productSpu, ProductSpuVo.class, HashSet.class, ArrayList.class);
    for (ProductSpuEventListener eventListener : eventListeners) {
      eventListener.onUpdate(oldVo, vo);
    }
    return current;
  }

  /**
   * 绑定spu附件信息、sku、图片、视频、介绍
   *
   * @param productSpu
   */
  @Transactional
  public void bindExtInfo(ProductSpu productSpu) {
    Validate.notNull(productSpu, "商品spu信息缺失");
    this.productSpuTagMappingService.saveBatch(productSpu.getTagList(), productSpu.getSpuCode());
    this.productSpuRelateSkuService.saveBatch(productSpu.getProductList(), productSpu.getSpuCode());
    this.productSpuIntroductionService.save(productSpu.getIntroduction(), productSpu.getSpuCode());
    List<ProductSpuMedia> mediaList = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(productSpu.getPictureList())) {
      productSpu
          .getPictureList()
          .forEach(
              a -> {
                a.setSpuCode(productSpu.getSpuCode());
                a.setType(MediaTypeEnum.PICTURE.getCode());
              });
      mediaList.addAll(productSpu.getPictureList());
    }
    if (CollectionUtils.isNotEmpty(productSpu.getVideoList())) {
      productSpu
          .getVideoList()
          .forEach(
              a -> {
                a.setSpuCode(productSpu.getSpuCode());
                a.setType(MediaTypeEnum.VIDEO.getCode());
              });
      mediaList.addAll(productSpu.getVideoList());
    }
    this.productSpuMediaService.saveBatch(mediaList, productSpu.getSpuCode());
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<ProductSpu> list = productSpuRepository.findByIds(ids);
    List<ProductSpuVo> voList = (List<ProductSpuVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list, ProductSpu.class, ProductSpuVo.class, HashSet.class, ArrayList.class);
    this.productSpuRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //保存日志
    if (CollectionUtils.isNotEmpty(this.eventListeners)) {
      this.eventListeners.forEach(productSpuEventListener -> productSpuEventListener.onEnable(voList));
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<ProductSpu> list = productSpuRepository.findByIds(ids);
    List<ProductSpuVo> voList = (List<ProductSpuVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list, ProductSpu.class, ProductSpuVo.class, HashSet.class, ArrayList.class);
    this.productSpuRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //保存日志
    if (CollectionUtils.isNotEmpty(this.eventListeners)) {
      this.eventListeners.forEach(productSpuEventListener -> productSpuEventListener.onDisable(voList));
    }
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<ProductSpu> list = productSpuRepository.findByIds(ids);
    List<ProductSpuVo> voList = (List<ProductSpuVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list, ProductSpu.class, ProductSpuVo.class, HashSet.class, ArrayList.class);
    this.productSpuRepository.updateDelFlagByIds(ids);
    //保存日志
    if (CollectionUtils.isNotEmpty(this.eventListeners)) {
      this.eventListeners.forEach(productSpuEventListener -> productSpuEventListener.onDelete(voList));
    }
  }

  @Override
  @Transactional
  public void upShelf(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.productSpuRepository.updateIsShelfByIds(ids, IsShelfEnum.UP.getCode());
    List<ProductSpu> list = this.findDetailsByIds(ids);
    onUpOrDownShelf(list, IsShelfEnum.UP);
  }

  @Override
  @Transactional
  public void downShelf(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.productSpuRepository.updateIsShelfByIds(ids, IsShelfEnum.DOWN.getCode());
    List<ProductSpu> list = this.findDetailsByIds(ids);
    onUpOrDownShelf(list, IsShelfEnum.DOWN);
  }

  @Override
  @Transactional
  public void upShelfBySpuCodes(List<String> spuCodeList) {
    Validate.isTrue(CollectionUtils.isNotEmpty(spuCodeList), "编码集合不能为空");
    this.productSpuRepository.updateIsShelfBySpuCodes(spuCodeList, IsShelfEnum.UP.getCode());
    List<ProductSpu> list = this.findDetailsBySpuCodes(spuCodeList);
    onUpOrDownShelf(list, IsShelfEnum.UP);
  }

  @Override
  @Transactional
  public void downShelfBySpuCodes(List<String> spuCodeList) {
    Validate.isTrue(CollectionUtils.isNotEmpty(spuCodeList), "编码集合不能为空");
    this.productSpuRepository.updateIsShelfBySpuCodes(spuCodeList, IsShelfEnum.DOWN.getCode());
    List<ProductSpu> list = this.findDetailsBySpuCodes(spuCodeList);
    onUpOrDownShelf(list, IsShelfEnum.DOWN);
  }

  @Override
  @Transactional
  public void updateUpShelfByProductCodes(Set<String> productCodeSet) {
    Validate.isTrue(CollectionUtils.isNotEmpty(productCodeSet), "商品编码集合不能为空");
    List<ProductSpuRelateSku> list =
        this.productSpuRelateSkuService.findSupCodesByproductCodesAndIsShelf(
            productCodeSet, IsShelfEnum.DOWN.getCode());
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    List<String> spuCodeList =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
            .map(ProductSpuRelateSku::getSpuCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(spuCodeList)) {
      return;
    }
    this.upShelfBySpuCodes(spuCodeList);
  }

  @Override
  @Transactional
  public void updateDownShelfByProductCodes(Set<String> productCodeSet) {
    // 1. 根据productCodeSet获取关联spu-sku关联集合数据
    // 2. 获取关联的集合中不存在productCodeSet中sku编码集合
    // 3. 获取2中对应上架商品编码集合
    // 4. 获取需要下架的spu编码集合
    // 5. 下架spu
    Validate.isTrue(CollectionUtils.isNotEmpty(productCodeSet), "商品编码集合不能为空");
    List<ProductSpuRelateSku> list =
        this.productSpuRelateSkuService.findSupCodesByproductCodesAndIsShelf(
            productCodeSet, IsShelfEnum.UP.getCode());
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    Set<String> allSpuCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
            .map(ProductSpuRelateSku::getSpuCode)
            .collect(Collectors.toSet());

    // 获取关联除开productCodeSet存在的其他sku编码集合
    Set<String> skuCodeSet =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNotBlank(a.getProductCode())
                        && !productCodeSet.contains(a.getProductCode()))
            .map(ProductSpuRelateSku::getProductCode)
            .collect(Collectors.toSet());
    // 获取skuCodeSet对应商品的详情,获取上架的商品对应的编码集合
    Set<String> upShelfProductCodeSet = Sets.newHashSet();
    if (CollectionUtils.isNotEmpty(skuCodeSet)) {
      List<ProductVo> productVoList =
          this.productVoService.findDetailsByIdsOrProductCodes(
              null, Lists.newArrayList(skuCodeSet));
      if (CollectionUtils.isNotEmpty(productVoList)) {
        upShelfProductCodeSet =
            productVoList.stream()
                .filter(
                    a ->
                        StringUtils.isNotBlank(a.getProductCode())
                            && IsShelfEnum.UP.getCode().equals(a.getIsShelf()))
                .map(ProductVo::getProductCode)
                .collect(Collectors.toSet());
      }
    }

    // 获取需要下架的spu编码集合
    Set<String> spuCodeList = Sets.newHashSet();
    for (ProductSpuRelateSku item : list) {
      if (upShelfProductCodeSet.contains(item.getProductCode())) {
        spuCodeList.add(item.getSpuCode());
      }
    }

    final SetView<String> setView = Sets.difference(allSpuCodeSet, spuCodeList);

    if (CollectionUtils.isEmpty(setView)) {
      return;
    }
    this.downShelfBySpuCodes(Lists.newArrayList(setView));
  }

  private void createValidation(ProductSpu productSpu) {
    Validate.notNull(productSpu, "商品spu信息缺失");
    this.validation(productSpu);
  }

  private void updateValidation(ProductSpu productSpu) {
    Validate.notNull(productSpu, "商品spu信息缺失");
    Validate.notBlank(productSpu.getId(), "商品spu id不能为空");
    this.validation(productSpu);
  }

  private void validation(ProductSpu productSpu) {
    Validate.notBlank(productSpu.getSpuName(), "商品spu名称不能为空");
    Validate.notBlank(productSpu.getIsShelf(), "商品spu上下架状态不能为空");
    Validate.notNull(productSpu.getBeginDateTime(), "开始时间不能为空");
    Validate.notNull(productSpu.getEndDateTime(), "结束时间不能为空");
    Validate.isTrue(
        productSpu.getEndDateTime().compareTo(productSpu.getBeginDateTime()) > 0, "结束时间需大于开始时间");
    Validate.isTrue(productSpu.getEndDateTime().compareTo(new Date()) > 0, "截止时间需大于当前时间");
  }

  /**
   * 发送上下架变更通知
   *
   * @param list
   * @param shelfEnum
   */
  private void onUpOrDownShelf(List<ProductSpu> list, IsShelfEnum shelfEnum) {
    if (CollectionUtils.isEmpty(eventListeners)) {
      return;
    }
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    List<ProductSpuVo> voList =
        (List<ProductSpuVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                list, ProductSpu.class, ProductSpuVo.class, HashSet.class, ArrayList.class);
    for (ProductSpuEventListener event : eventListeners) {
      if (shelfEnum.equals(IsShelfEnum.UP)) {
        event.onUpShelf(voList);
      } else if (shelfEnum.equals(IsShelfEnum.DOWN)) {
        event.onDownShelf(voList);
      }
    }
  }

  /**
   * 获取spu扩展信息标签、sku、图片、视频、详情介绍
   *
   * @param list
   * @return
   */
  private List<ProductSpu> findDetailsExtInfo(List<ProductSpu> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<String> spuCodeList =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
            .map(ProductSpu::getSpuCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(spuCodeList)) {
      return Lists.newLinkedList();
    }
    final ProductSpuTagMappingQueryDto queryDto = new ProductSpuTagMappingQueryDto();
    queryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    queryDto.setTenantCode(TenantUtils.getTenantCode());
    queryDto.setSpuCodes(Sets.newHashSet(spuCodeList));
    List<ProductSpuTagMapping> tagList =
        this.productSpuTagMappingRepository.findByProductSpuTagMappingQueryDto(queryDto);
    List<ProductSpuRelateSku> skuList = this.productSpuRelateSkuService.findBySpuCodes(spuCodeList);
    List<ProductSpuMedia> mediaList = this.productSpuMediaService.findBySpuCodes(spuCodeList);
    List<ProductSpuIntroduction> introductionList =
        this.productSpuIntroductionService.findBySpuCodes(spuCodeList);

    Map<String, List<ProductSpuTagMapping>> mapTag = Maps.newHashMap();
    Map<String, List<ProductSpuRelateSku>> mapSku = Maps.newHashMap();
    Map<String, List<ProductSpuMedia>> mapMedia = Maps.newHashMap();
    Map<String, List<ProductSpuIntroduction>> mapIntro = Maps.newHashMap();

    if (CollectionUtils.isNotEmpty(tagList)) {
      mapTag =
          tagList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
              .collect(Collectors.groupingBy(ProductSpuTagMapping::getSpuCode));
    }

    if (CollectionUtils.isNotEmpty(skuList)) {
      mapSku =
          skuList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
              .collect(Collectors.groupingBy(ProductSpuRelateSku::getSpuCode));
    }

    if (CollectionUtils.isNotEmpty(mediaList)) {
      mapMedia =
          mediaList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
              .collect(Collectors.groupingBy(ProductSpuMedia::getSpuCode));
    }

    if (CollectionUtils.isNotEmpty(introductionList)) {
      mapIntro =
          introductionList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
              .collect(Collectors.groupingBy(ProductSpuIntroduction::getSpuCode));
    }

    for (ProductSpu item : list) {
      List<ProductSpuTagMapping> productSpuTagVos = mapTag.get(item.getSpuCode());
      if (!CollectionUtils.isEmpty(productSpuTagVos)) {
        Collection<ProductSpuTag> productSpuTags =
            nebulaToolkitService.copyCollectionByWhiteList(
                productSpuTagVos,
                ProductSpuTagMapping.class,
                ProductSpuTag.class,
                HashSet.class,
                ArrayList.class);
        item.setTagList(Lists.newArrayList(productSpuTags));
      }
      item.setProductList(mapSku.get(item.getSpuCode()));
      List<ProductSpuMedia> mediaList1 = mapMedia.get(item.getSpuCode());
      if (CollectionUtils.isNotEmpty(mediaList1)) {
        Map<String, List<ProductSpuMedia>> curMap =
            mediaList1.stream()
                .filter(a -> StringUtils.isNotBlank(a.getType()))
                .collect(Collectors.groupingBy(ProductSpuMedia::getType));
        item.setPictureList(curMap.get(MediaTypeEnum.PICTURE.getCode()));
        item.setVideoList(curMap.get(MediaTypeEnum.VIDEO.getCode()));
      }
      List<ProductSpuIntroduction> introductionList1 = mapIntro.get(item.getSpuCode());
      if (CollectionUtils.isNotEmpty(introductionList1)) {
        item.setIntroduction(introductionList1.get(0));
      }
    }
    return list;
  }
}
