package com.biz.crm.mdm.business.product.spu.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuTag;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuTagMapping;
import com.biz.crm.mdm.business.product.spu.local.repository.ProductSpuTagMappingRepository;
import com.biz.crm.mdm.business.product.spu.local.repository.ProductSpuTagRepository;
import com.biz.crm.mdm.business.product.spu.local.service.ProductSpuTagService;
import com.biz.crm.mdm.business.product.spu.sdk.constant.ProductSpuConstant;
import com.biz.crm.mdm.business.product.spu.sdk.dto.ProductSpuTagMappingQueryDto;
import com.biz.crm.mdm.business.product.spu.sdk.dto.ProductSpuTagPaginationDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 商品spu标签信息(ProductSpuTag)表服务实现类
 *
 * @author sunx
 * @date 2021-12-02 17:07:24
 */
@Slf4j
@Service("productSpuTagService")
public class ProductSpuTagServiceImpl implements ProductSpuTagService {

  @Autowired(required = false)
  private ProductSpuTagRepository productSpuTagRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private ProductSpuTagMappingRepository productSpuTagMappingRepository;

  @Override
  public Page<ProductSpuTag> findByConditions(
      Pageable pageable, ProductSpuTagPaginationDto paginationDto) {
    paginationDto = ObjectUtils.defaultIfNull(paginationDto, new ProductSpuTagPaginationDto());
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    paginationDto.setTenantCode(TenantUtils.getTenantCode());
    final Page<ProductSpuTag> page =
        productSpuTagRepository.findByConditions(pageable, paginationDto);
    if (Objects.isNull(page) || CollectionUtils.isEmpty(page.getRecords())) {
      return page;
    }
    final Set<String> tagCodeSet =
        page.getRecords().stream()
            .filter(a -> StringUtils.isNotBlank(a.getTagCode()))
            .map(ProductSpuTag::getTagCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(tagCodeSet)) {
      return page;
    }
    final ProductSpuTagMappingQueryDto queryDto = new ProductSpuTagMappingQueryDto();
    queryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    queryDto.setTenantCode(TenantUtils.getTenantCode());
    queryDto.setTagCodes(tagCodeSet);
    final List<ProductSpuTagMapping> list =
        this.productSpuTagMappingRepository.findByProductSpuTagMappingQueryDto(queryDto);
    final Map<String, Long> map =
        list.stream()
            .collect(
                Collectors.groupingBy(ProductSpuTagMapping::getTagCode, Collectors.counting()));
    for (ProductSpuTag item : page.getRecords()) {
      item.setSpuQuantity(map.getOrDefault(item.getTagCode(), 0L));
    }
    return page;
  }

  /**
   * 创建
   *
   * @param productSpuTag
   * @return
   */
  @Override
  @Transactional
  public ProductSpuTag create(ProductSpuTag productSpuTag) {
    this.createValidate(productSpuTag);
    if (StringUtils.isBlank(productSpuTag.getTagCode())) {
      String tagCode = generateCodeService.generateCode(ProductSpuConstant.TAG_CODE, 1).get(0);
      productSpuTag.setTagCode(tagCode);
    } else {
      ProductSpuTag old = this.productSpuTagRepository.findByTagCode(productSpuTag.getTagCode());
      Validate.isTrue(Objects.isNull(old), "已存在对应编码的标签");
    }
    productSpuTag.setTenantCode(TenantUtils.getTenantCode());
    productSpuTagRepository.save(productSpuTag);
    return productSpuTag;
  }

  /**
   * 修改
   *
   * @param productSpuTag
   * @return
   */
  @Override
  @Transactional
  public ProductSpuTag update(ProductSpuTag productSpuTag) {
    this.updateValidate(productSpuTag);
    final List<ProductSpuTag> list =
        this.productSpuTagRepository.findByIds(Lists.newArrayList(productSpuTag.getId()));
    final ProductSpuTag first = list.stream().findFirst().orElse(null);
    Validate.notNull(first, "标签信息不存在");
    Validate.isTrue(productSpuTag.getTagCode().equals(first.getTagCode()), "标签编码不能修改");
    this.productSpuTagRepository.updateById(productSpuTag);
    return productSpuTag;
  }

  /**
   * 批量起用
   *
   * @param ids
   */
  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.notEmpty(ids, "缺失id");
    this.productSpuTagRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE.getCode());
  }

  /**
   * 批量禁用
   *
   * @param ids
   */
  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.notEmpty(ids, "缺失id");
    this.productSpuTagRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE.getCode());
  }

  /**
   * 批量删除
   *
   * @param ids
   */
  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<ProductSpuTag> list = this.productSpuTagRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    final Set<String> set =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTagCode()))
            .map(ProductSpuTag::getTagCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isNotEmpty(set)) {
      this.productSpuTagMappingRepository.deleteByTagCodes(set);
    }
    this.productSpuTagRepository.deleteBatch(ids);
  }

  /**
   * 查看详情
   *
   * @param id
   * @return
   */
  @Override
  public ProductSpuTag findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    final List<ProductSpuTag> list = productSpuTagRepository.findByIds(Lists.newArrayList(id));
    return list.stream().findFirst().orElse(null);
  }

  /**
   * 修改数据验证
   *
   * @param productSpuTag
   */
  private void updateValidate(ProductSpuTag productSpuTag) {
    Validate.notNull(productSpuTag, "修改商品标签时，数据不能为空");
    Validate.notBlank(productSpuTag.getId(), "修改商品标签时，标签id不能为空");
    Validate.notBlank(productSpuTag.getTagName(), "修改商品标签时，标签名称不能为空");
    Validate.notBlank(productSpuTag.getTagCode(), "修改商品标签时，标签编号不能为空");
    ProductSpuTag tag = productSpuTagRepository.findByTagName(productSpuTag.getTagName());
    if (Objects.nonNull(tag)) {
      Validate.isTrue(tag.getId().equals(productSpuTag.getId()), "修改商品标签时，标签名称不能重复");
    }
  }

  /**
   * 创建数据验证
   *
   * @param productSpuTag
   */
  private void createValidate(ProductSpuTag productSpuTag) {
    Validate.notNull(productSpuTag, "创建商品标签时，数据不能为空");
    productSpuTag.setId(null);
    productSpuTag.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    productSpuTag.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    Validate.notBlank(productSpuTag.getTagName(), "创建商品标签时，标签名称不能为空");
    ProductSpuTag tag = productSpuTagRepository.findByTagName(productSpuTag.getTagName());
    Validate.isTrue(tag == null, "创建商品标签时，标签名称不能重复");
  }
}
