package com.biz.crm.mdm.business.productlevel.local.service.internal;
/**
 * Created by Bao Hongbin on 2021-10-08 17:53.
 */

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.common.local.entity.TenantFlagTreeOpEntity;
import com.biz.crm.mdm.business.common.sdk.dto.TreeDto;
import com.biz.crm.mdm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.mdm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.mdm.business.productlevel.local.entity.ProductLevel;
import com.biz.crm.mdm.business.productlevel.local.repository.ProductLevelRepository;
import com.biz.crm.mdm.business.productlevel.local.service.ProductLevelVoService;
import com.biz.crm.mdm.business.productlevel.local.service.helper.ProductLevelServiceHelper;
import com.biz.crm.mdm.business.productlevel.sdk.common.constant.ProductLevelConstant;
import com.biz.crm.mdm.business.productlevel.sdk.dto.ProductLevelCreateDto;
import com.biz.crm.mdm.business.productlevel.sdk.dto.ProductLevelPaginationDto;
import com.biz.crm.mdm.business.productlevel.sdk.dto.ProductLevelUpdateDto;
import com.biz.crm.mdm.business.productlevel.sdk.vo.ProductLevelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @program: crm
 * @description: 产品层级服务实现
 * @author: Bao Hongbin
 * @create: 2021-10-08 17:53
 **/
@Service
public class ProductLevelVoServiceImpl implements ProductLevelVoService {
  @Autowired
  private ProductLevelRepository productLevelRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private GenerateCodeService generateCode;
  @Autowired(required = false)
  private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;
  @Autowired
  private ProductLevelServiceHelper helper;


  @Override
  public Page<ProductLevelVo> findByConditions(Pageable pageable, ProductLevelPaginationDto productLevelPaginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    productLevelPaginationDto = Optional.ofNullable(productLevelPaginationDto).orElse(new ProductLevelPaginationDto());
    productLevelPaginationDto.setTenantCode(TenantUtils.getTenantCode());
    //白名单ruleCode
    if (StringUtils.isNotEmpty(productLevelPaginationDto.getUnderProductLevelCode())) {
      //如果条件成立，说明本次查询只查询该白名单里的数据及其子集
      ProductLevel one = productLevelRepository.findDetailsByCode(
          productLevelPaginationDto.getUnderProductLevelCode(),
          productLevelPaginationDto.getTenantCode());
      if (one != null) {
        productLevelPaginationDto.setUnderThisRuleCode(one.getRuleCode());
      }
    }
    //黑名单ruleCode
    if (StringUtils.isNotEmpty(productLevelPaginationDto.getNotUnderProductLevelCode())) {
      //如果条件成立，说明本次查询不查询该黑名单里的数据及其子集
      ProductLevel one = productLevelRepository.findDetailsByCode(
          productLevelPaginationDto.getNotUnderProductLevelCode(),
          productLevelPaginationDto.getTenantCode());
      if (one != null) {
        productLevelPaginationDto.setNotUnderThisRuleCode(one.getRuleCode());
      }
    }
    //设置已选择项（如果查询条件符合，那么已选择项需要优先显示）
    List<String> selectedCodeList =
        Optional.ofNullable(productLevelPaginationDto.getSelectedCodeList()).orElse(new ArrayList<>());
    if (StringUtils.isNotEmpty(productLevelPaginationDto.getSelectedCode())) {
      selectedCodeList.add(productLevelPaginationDto.getSelectedCode());
    }
    if (!CollectionUtils.isEmpty(selectedCodeList)) {
      productLevelPaginationDto.setSelectedCodeList(selectedCodeList);
    }
    return productLevelRepository.findByConditions(pageable, productLevelPaginationDto);
  }

  @Override
  public ProductLevelVo findDetailsById(String id) {
    if (!StringUtils.isNotEmpty(id)) {
      return null;
    }
    ProductLevel one = productLevelRepository.findDetailsById(id, TenantUtils.getTenantCode());
    if (one == null) {
      return null;
    }
    ProductLevelVo respVo = nebulaToolkitService.copyObjectByWhiteList(
        one, ProductLevelVo.class, HashSet.class, ArrayList.class);
    if (StringUtils.isNotEmpty(one.getParentCode())) {
      ProductLevel parent = productLevelRepository.findDetailsByCode(one.getParentCode(), TenantUtils.getTenantCode());
      if (parent != null) {
        respVo.setParentName(parent.getProductLevelName());
      }
    }
    return respVo;
  }

  @Override
  @Transactional
  public ProductLevelVo create(ProductLevelCreateDto productLevelCreateDto) {
    return this.createForm(productLevelCreateDto);
  }

  @Override
  @Transactional
  public ProductLevelVo update(ProductLevelUpdateDto productLevelUpdateDto) {
    return this.updateForm(productLevelUpdateDto);
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "当进行启用操作时，业务技术编号信息必须传入");
    //1.获取需要启用的数据
    List<ProductLevel> productLevelList =
        productLevelRepository.findListByIds(ids, TenantUtils.getTenantCode())
            .stream().filter(x -> !EnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(productLevelList)) {
      return;
    }
    //2.根据ruleCode获取其所有的上级
    List<String> collect = productLevelList.stream().map(ProductLevel::getRuleCode).collect(Collectors.toList());
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    Set<String> parentRuleCodesExcludeSelf =
        treeRuleCodeStrategy.findParentRuleCodeByRuleCodesExcludeAnySelf(
            ProductLevelConstant.RULE_CODE_LENGTH, collect);
    if (!parentRuleCodesExcludeSelf.isEmpty()) {
      //3.如果存在未启用的上级层级，不能启用当前层级
      List<ProductLevel> parentList = productLevelRepository.findListByRuleCodes(parentRuleCodesExcludeSelf, TenantUtils.getTenantCode())
          .stream().filter(x -> !EnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
          .collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(parentList), "存在未启用的上级层级，不能启用当前层级");
    }
    //4.更新启用状态
    productLevelList.forEach(productLevel -> productLevel.setEnableStatus(EnableStatusEnum.ENABLE.getCode()));
    productLevelRepository.updateBatchById(productLevelList);
    //5.推送onEnable事件
    ArrayList<ProductLevelVo> productLevelVos =
        Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(productLevelList, ProductLevel.class
            , ProductLevelVo.class, HashSet.class, ArrayList.class));
    helper.sendEnableEvent(productLevelVos);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "当进行禁用操作时，业务技术编号信息必须传入");
    //1.获取需要禁用的数据
    List<ProductLevel> productLevelList = productLevelRepository.findListByIds(ids, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(productLevelList)) {
      return;
    }
    //2.使用ruleCode模糊查询所有需要禁用的数据
    List<String> ruleCodes = productLevelList.stream().map(ProductLevel::getRuleCode).collect(Collectors.toList());
    List<ProductLevel> childrenByRuleCodeList =
        productLevelRepository.findCurAndChildrenByRuleCodeList(ruleCodes, EnableStatusEnum.ENABLE.getCode(), TenantUtils.getTenantCode());
    if (!CollectionUtils.isEmpty(childrenByRuleCodeList)) {
      //3.更新启用状态
      childrenByRuleCodeList.forEach(productLevel -> productLevel.setEnableStatus(EnableStatusEnum.DISABLE.getCode()));
      productLevelRepository.updateBatchById(childrenByRuleCodeList);
      //4.推送onDisable事件
      ArrayList<ProductLevelVo> productLevelVos =
          Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(childrenByRuleCodeList, ProductLevel.class
              , ProductLevelVo.class, HashSet.class, ArrayList.class));
      helper.sendDisableEvent(productLevelVos);
    }
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "当进行删除操作时，业务技术编号信息必须传入");
    //1.获取需要删除的数据
    List<ProductLevel> productLevelList = productLevelRepository.findListByIds(ids, TenantUtils.getTenantCode());
    Validate.isTrue(!CollectionUtils.isEmpty(productLevelList), "无效的业务技术编号信息");
    List<String> productLevelCodeList =
        productLevelList.stream().map(ProductLevel::getProductLevelCode).collect(Collectors.toList());
    //2.判断是否存在子层级
    List<ProductLevel> childrenList =
        productLevelRepository.findChildrenListByParentCodes(productLevelCodeList, TenantUtils.getTenantCode())
            .stream().filter(o -> !ids.contains(o.getId())).collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isEmpty(childrenList), "当前产品层级包含子层级，无法删除，若需要删除请先删除子层级");
    //3.删除产品层级
    productLevelRepository.removeByIds(ids);
    //4.推送onDelete事件
    ArrayList<ProductLevelVo> productLevelVos =
        Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(productLevelList, ProductLevel.class
            , ProductLevelVo.class, HashSet.class, ArrayList.class));
    helper.sendDeleteEvent(productLevelVos);
  }

  @Override
  @Transactional
  public void resetAllRuleCode() {
    //1.查找parentCode不为空但找不到对应上级的数据，将其parentCode设为空
    productLevelRepository.updateOrphanParentCodeNull(TenantUtils.getTenantCode());
    //2.查找所有parentCode为空的数据（相当于第一层数据）
    List<ProductLevel> topList =
        productLevelRepository.findListWithoutParentCode(TenantUtils.getTenantCode());
    //3.递归设置其ruleCode和其所有子级的ruleCode
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    for (int i = 0; i < topList.size(); i++) {
      //递归调用
      updateCurAndChildrenRuleCode(topList.get(i).getProductLevelCode(),
          treeRuleCodeStrategy.generateByNum(ProductLevelConstant.RULE_CODE_LENGTH, i + 1),
          1);
    }
  }

  @Override
  public List<String> findCurAndChildrenCodesByCode(String productLevelCode) {
    if (StringUtils.isBlank(productLevelCode)) {
      return new ArrayList<>();
    }
    ProductLevel productLevel =
        productLevelRepository.findDetailsByCode(productLevelCode, TenantUtils.getTenantCode());
    if (null == productLevel) {
      return new ArrayList<>();
    }
    return productLevelRepository.findCurAndChildrenByRuleCode(productLevel.getRuleCode(), TenantUtils.getTenantCode())
        .stream().map(ProductLevel::getProductLevelCode).collect(Collectors.toList());
  }

  @Override
  public List<ProductLevelVo> findAll() {
    List<ProductLevel> all = productLevelRepository.findAll(TenantUtils.getTenantCode());
    return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(all, ProductLevel.class
        , ProductLevelVo.class, HashSet.class, ArrayList.class));
  }

  private ProductLevelVo createForm(ProductLevelCreateDto productLevelCreateDto) {
    //1.数据验证
    helper.createValidation(productLevelCreateDto);
    ProductLevel productLevel = this.nebulaToolkitService.copyObjectByWhiteList(
        productLevelCreateDto, ProductLevel.class, HashSet.class, ArrayList.class);
    productLevel.setTenantCode(TenantUtils.getTenantCode());
    productLevel.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    productLevel.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //2.判断产品层级编码是否为空，如果不为空那么查询是否存在，如果为空则设置
    if (StringUtils.isBlank(productLevelCreateDto.getProductLevelCode())) {
      //TODO 各业务编码前缀以前可以在系统设置中进行设置，该功能将在后续改造中实现，先设置为固定默认值
      productLevel.setProductLevelCode(generateCode.generateCode(ProductLevelConstant.CODE, 1).get(0));
    }
    //3.根据父级设置层级
    int levelNum = 1;
    if (StringUtils.isNotEmpty(productLevelCreateDto.getParentCode())) {
      ProductLevel parent = productLevelRepository.findDetailsByCode(productLevelCreateDto.getParentCode(), TenantUtils.getTenantCode());
      levelNum = parent.getLevelNum() + 1;
    }
    //4.设置规则（降维）编码
    String ruleCode = getRuleCodeByParentCode(productLevelCreateDto.getParentCode());
    productLevel.setRuleCode(ruleCode);
    productLevel.setLevelNum(levelNum);
    //5.保存数据
    productLevelRepository.save(productLevel);
    //6.推送onCreate事件
    ProductLevelVo productLevelVo = nebulaToolkitService.copyObjectByWhiteList(
        productLevel, ProductLevelVo.class, HashSet.class, ArrayList.class);
    helper.sendCreateEvent(Collections.singletonList(productLevelVo));
    return productLevelVo;
  }


  private ProductLevelVo updateForm(ProductLevelUpdateDto productLevelUpdateDto) {
    //1.更新数据验证
    helper.updateValidation(productLevelUpdateDto);
    ProductLevel productLevel = productLevelRepository.findDetailsById(productLevelUpdateDto.getId(), TenantUtils.getTenantCode());
    //2.根据父级设置层级
    int levelNum = 1;
    if (StringUtils.isNotEmpty(productLevelUpdateDto.getParentCode())) {
      ProductLevel parent = productLevelRepository.findDetailsByCode(productLevelUpdateDto.getParentCode(), TenantUtils.getTenantCode());
      levelNum = parent.getLevelNum() + 1;
    }
    //3.更新数据
    ProductLevel productLevelNew = this.nebulaToolkitService.copyObjectByWhiteList(
        productLevelUpdateDto, ProductLevel.class, HashSet.class, ArrayList.class);
    productLevelRepository.updateById(productLevelNew);
    //4.判断是否需要重置降维编码和更新启用状态
    boolean updateRuleCode = false;
    String enableStatusChangeTo = "";
    if (!(productLevel.getParentCode() == null ? "" : productLevel.getParentCode())
        .equals((productLevelUpdateDto.getParentCode() == null ? "" : productLevelUpdateDto.getParentCode()))) {
      //上级编码发生变化，重置降维编码
      updateRuleCode = true;
    }
    if (!(productLevel.getEnableStatus()).equals(productLevelUpdateDto.getEnableStatus())) {
      //启用状态发生变化
      enableStatusChangeTo = productLevelUpdateDto.getEnableStatus();
    }
    if (updateRuleCode) {
      //4.1更新降维编码
      String ruleCode = getRuleCodeByParentCode(productLevelUpdateDto.getParentCode());
      updateCurAndChildrenRuleCode(productLevelNew.getProductLevelCode(), ruleCode, levelNum);
    }
    if (StringUtils.isNotEmpty(enableStatusChangeTo)) {
      //4.2更新启用禁用状态
      if (EnableStatusEnum.ENABLE.getCode().equals(enableStatusChangeTo)) {
        this.enableBatch(Collections.singletonList(productLevelNew.getId()));
      } else if (EnableStatusEnum.DISABLE.getCode().equals(enableStatusChangeTo)) {
        this.disableBatch(Collections.singletonList(productLevelNew.getId()));
      } else {
        throw new IllegalArgumentException("启用状态错误");
      }
    }
    //6.推送onUpdate事件
    //再次查询更新后的数据
    productLevel = productLevelRepository.findDetailsById(productLevelUpdateDto.getId(), TenantUtils.getTenantCode());
    ProductLevelVo productLevelVo = nebulaToolkitService.copyObjectByWhiteList(
        productLevel, ProductLevelVo.class, HashSet.class, ArrayList.class);
    helper.sendUpdateEvent(Collections.singletonList(productLevelVo));
    return productLevelVo;
  }

  /**
   * 根据父级编码获得当前降维编码
   *
   * @param parentCode
   * @return
   */
  private String getRuleCodeByParentCode(String parentCode) {
    String parentRuleCode = null;
    if (StringUtils.isNotEmpty(parentCode)) {
      ProductLevel parent = productLevelRepository.findDetailsByCode(parentCode, TenantUtils.getTenantCode());
      Validate.notNull(parent, "上级产品层级不存在");
      parentRuleCode = parent.getRuleCode();
    }
    List<ProductLevel> childrenListByParentCode =
        productLevelRepository.findChildrenListByParentCode(parentCode, TenantUtils.getTenantCode());
    List<TreeDto> childrenDto =
        Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(childrenListByParentCode, ProductLevel.class
            , TreeDto.class, HashSet.class, ArrayList.class));
    //TODO 编码降维编码长度以前可以在系统设置中进行设置，该功能将在后续改造中实现，先设置为固定默认值
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    return treeRuleCodeStrategy.generate(
        ProductLevelConstant.RULE_CODE_LENGTH, parentRuleCode, childrenDto);
  }

  /**
   * 更新自己及子集的降维编码
   *
   * @param productLevelCode
   * @param ruleCode
   * @param levelNum
   */
  private void updateCurAndChildrenRuleCode(String productLevelCode, String ruleCode, int levelNum) {
    //更新当前
    ProductLevel productLevel = productLevelRepository.findDetailsByCode(productLevelCode, TenantUtils.getTenantCode());
    productLevel.setRuleCode(ruleCode);
    productLevel.setLevelNum(levelNum);
    productLevelRepository.updateById(productLevel);
    //查询下一层
    List<ProductLevel> childrenList = productLevelRepository.findChildrenListByParentCode(productLevelCode, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(childrenList)) {
      return;
    }
    //遍历下级
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    for (int i = 0; i < childrenList.size(); i++) {
      //递归调用
      updateCurAndChildrenRuleCode(childrenList.get(i).getProductLevelCode(),
          ruleCode + treeRuleCodeStrategy.generateByNum(ProductLevelConstant.RULE_CODE_LENGTH, i + 1),
          levelNum + 1);
    }
  }
}
