package com.biz.crm.mdm.business.promoters.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.promoters.sdk.dto.PromotersDto;
import com.biz.crm.mdm.business.promoters.sdk.dto.PromotersSelectDto;
import com.biz.crm.mdm.business.promoters.sdk.service.PromotersVoService;
import com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 促销人员管理
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.4 23:41
 */
@Slf4j
@Api(tags = "促销人员管理: PromotersVo: 促销人员")
@RestController
@RequestMapping(value = {"/v1/promoters/promotersVo"})
public class PromotersVoController {

    @Autowired(required = false)
    private PromotersVoService promotersVoService;

    /**
     * 查询分页列表
     *
     * @param pageable 分页信息
     * @param dto      查询参数
     * @return com.biz.crm.business.common.sdk.model.Result<com.baomidou.mybatisplus.extension.plugins.pagination.Page < com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo>>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:51
     */
    @ApiOperation(value = "查询分页列表")
    @GetMapping(value = {"/findByConditions"})
    public Result<Page<PromotersVo>> findByConditions(@PageableDefault(50) Pageable pageable, PromotersDto dto) {
        try {
            Page<PromotersVo> result = this.promotersVoService.findByConditions(pageable, dto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "批量新增或修改")
    @PostMapping("createOrUpdate")
    public Result<List<PromotersVo>> createOrUpdate(@RequestBody List<PromotersDto> dtoList) {
        try {
            List<PromotersVo> result = this.promotersVoService.createOrUpdate(dtoList);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "新增")
    @PostMapping("create")
    public Result<PromotersVo> create(@ApiParam(name = "dto", value = "促销人员") @RequestBody PromotersDto dto) {
        try {
            PromotersVo result = this.promotersVoService.create(dto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "修改")
    @PatchMapping("update")
    public Result<PromotersVo> update(@ApiParam(name = "dto", value = "促销人员") @RequestBody PromotersDto dto) {
        try {
            PromotersVo result = this.promotersVoService.update(dto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "删除")
    @DeleteMapping("delete")
    public Result<?> delete(@ApiParam(name = "idList", value = "ID集合") @RequestParam("idList") List<String> idList) {
        try {
            this.promotersVoService.delete(idList);
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 启用
     */
    @ApiOperation(value = "启用")
    @PatchMapping("enable")
    public Result<?> enable(@RequestBody List<String> ids) {
        try {
            this.promotersVoService.enableBatch(ids);
            return Result.ok("启用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 禁用
     */
    @ApiOperation(value = "禁用")
    @PatchMapping("disable")
    public Result<?> disable(@RequestBody List<String> ids) {
        try {
            this.promotersVoService.disableBatch(ids);
            return Result.ok("禁用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据id获取信息
     *
     * @param id 员工ID
     * @return com.biz.crm.business.common.sdk.model.Result<com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:46
     */
    @ApiOperation(value = "根据id获取信息")
    @GetMapping(value = {"/findById"})
    public Result<PromotersVo> findById(@RequestParam("id") String id) {
        try {
            PromotersVo list = this.promotersVoService.findById(id);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据id集合获取信息
     *
     * @param idList ID集合
     * @return com.biz.crm.business.common.sdk.model.Result<java.util.List < com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo>>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:46
     */
    @ApiOperation(value = "根据id集合获取信息")
    @GetMapping(value = {"/findByIds"})
    public Result<List<PromotersVo>> findByIds(@RequestParam("idList") List<String> idList) {
        try {
            List<PromotersVo> list = this.promotersVoService.findByIds(idList);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据编码获取信息
     *
     * @param code 员工编码
     * @return com.biz.crm.business.common.sdk.model.Result<com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:46
     */
    @ApiOperation(value = "根据编码获取信息")
    @GetMapping(value = {"/findByCode"})
    public Result<PromotersVo> findByCode(@RequestParam("code") String code) {
        try {
            PromotersVo list = this.promotersVoService.findByCode(code);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据编码获取信息
     *
     * @param codeList 编码集合
     * @return com.biz.crm.business.common.sdk.model.Result<java.util.List < com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo>>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:46
     */
    @ApiOperation(value = "根据编码集合获取信息")
    @GetMapping(value = {"/findByCodes"})
    public Result<List<PromotersVo>> findByCodes(@RequestParam("codeList") List<String> codeList) {
        try {
            List<PromotersVo> list = this.promotersVoService.findByCodes(codeList);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 按身份证号集合查询详情
     *
     * @param idCardList 身份证号集合
     * @return List<PromotersVo>
     */
    @ApiOperation(value = "按身份证号集合查询详情")
    @PostMapping(value = {"/findByIdCard"})
    public Result<List<PromotersVo>> findByIdCard(@RequestBody List<String> idCardList) {
        try {
            List<PromotersVo> list = this.promotersVoService.findByIdCard(idCardList);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 促销人员下拉
     *
     * @param dto 促销人员查询参数
     */
    @ApiOperation(value = "促销人员下拉")
    @PostMapping("findSelectList")
    public Result<List<PromotersVo>> findSelectList(@ApiParam(name = "CommonSelectDto", value = "查询参数") @RequestBody PromotersSelectDto dto) {
        try {
            List<PromotersVo> list = this.promotersVoService.findSelectList(dto);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("查询促销门店人员")
    @PostMapping("findTerminalPromoters")
    public Result<List<PromotersVo>> findTerminalPromoters(@RequestBody PromotersDto dto){
        try {
            return Result.ok(this.promotersVoService.findTerminalPromoters(dto));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("根据门店查询促销门店人员")
    @PostMapping("findTerminalPromotersByTerminal")
    public Result<List<PromotersVo>> findTerminalPromotersByTerminal(@RequestBody PromotersDto dto) {
        try {
            return Result.ok(this.promotersVoService.findTerminalPromotersByTerminal(dto));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 同步
     * @param dto
     * @return {@link Result}<{@link ?}>
     */
    @ApiOperation("同步")
    @PostMapping("sync")
    public Result<?> sync(@RequestBody PromotersDto dto){
        try {
            Assert.hasText(dto.getEmployeeCode(), "员工号不能为空");
            Page<PromotersVo> byConditions = this.promotersVoService.findByConditions(null, dto);
            byConditions.getRecords().forEach(o -> {
                dto.setEmployeeCode(o.getEmployeeCode());
                this.promotersVoService.sync(dto);
            });
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

}
