package com.biz.crm.mdm.business.promoters.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.promoters.local.entity.TerminalPromoters;
import com.biz.crm.mdm.business.promoters.local.mapper.TerminalPromotersMapper;
import com.biz.crm.mdm.business.promoters.sdk.dto.TerminalPromotersDto;
import com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo;
import com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.List;

/**
 * 促销人员信息(repository)
 *
 * @describe: 简述
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.4 22:53
 */
@Component
public class TerminalPromotersRepository extends ServiceImpl<TerminalPromotersMapper, TerminalPromoters> {

    /**
     * 分页
     *
     * @param page
     * @param dto
     * @return
     */
    public Page<TerminalPromotersVo> findByConditions(Page<TerminalPromotersVo> page, TerminalPromotersDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findByConditions(page, dto);
    }

    /**
     * 根据id获取详情
     *
     * @param id 员工ID
     * @return
     */
    public TerminalPromoters findById(String id) {
        if (!StringUtils.hasText(id)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(TerminalPromoters::getId, id)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
    }

    /**
     * 根据id集合获取详情集合
     *
     * @param idList ID集合
     * @return
     */
    public List<TerminalPromoters> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(TerminalPromoters::getId, idList)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据门店编码获取详情
     *
     * @param code 门店编码
     * @return
     */
    public List<TerminalPromoters> findByTerminalCode(String code) {
        if (!StringUtils.hasText(code)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(TerminalPromoters::getTerminalCode, code)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据门店编码集合获取详情集合
     *
     * @param codeList 门店编码集合
     * @return
     */
    public List<TerminalPromoters> findByTerminalCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(TerminalPromoters::getTerminalCode, codeList)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据员工编码获取详情
     *
     * @param code 员工编码
     * @return
     */
    public List<TerminalPromoters> findByEmployeeCode(String code) {
        if (!StringUtils.hasText(code)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(TerminalPromoters::getEmployeeCode, code)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据员工编码集合获取详情集合
     *
     * @param codeList 员工编码集合
     * @return
     */
    public List<TerminalPromoters> findByEmployeeCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(TerminalPromoters::getEmployeeCode, codeList)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据门店编码获取详情
     *
     * @param code 门店编码
     * @return
     */
    public List<TerminalPromoters> findByCustomerCode(String code) {
        if (!StringUtils.hasText(code)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(TerminalPromoters::getCustomerCode, code)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据经销商编码集合获取详情集合
     *
     * @param codeList 经销商编码集合
     * @return
     */
    public List<TerminalPromoters> findByCustomerCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(TerminalPromoters::getCustomerCode, codeList)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据唯一编码获取详情集合
     *
     * @param code 唯一编码
     * @return
     */
    public TerminalPromoters findByUniqueCode(String code) {
        if (!StringUtils.hasText(code)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(TerminalPromoters::getUniqueCode, code)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .one();
    }

    /**
     * 根据唯一编码集合获取详情集合
     *
     * @param codeList 唯一编码集合
     * @return
     */
    public List<TerminalPromoters> findByUniqueCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(TerminalPromoters::getUniqueCode, codeList)
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据编码查询是否存在
     *
     * @param code 编码
     * @return
     */
    public Integer countByUniqueCode(String code) {
        if (!StringUtils.hasText(code)) {
            return 0;
        }
        return this.lambdaQuery()
                .eq(TerminalPromoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(TerminalPromoters::getUniqueCode, code)
                .count();
    }

    /**
     * 根据主键集合，修改 enable_status
     *
     * @param enable 状态
     * @param ids    ID集合
     */
    public void updateEnableStatusByIdIn(EnableStatusEnum enable, List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        UpdateWrapper<TerminalPromoters> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("enable_status", enable.getCode());
        updateWrapper.in("id", ids);
        this.update(updateWrapper);
    }

    /**
     * 分页查询员工和店铺关系
     * @param page
     * @param terminalCodeList
     * @return
     */
    public Page<PromotersVo> findPromotersByTerminalCodes(Page<TerminalPromotersVo> page, List<String> terminalCodeList) {
        if(CollectionUtils.isEmpty(terminalCodeList)){
            return new Page<>(page.getCurrent(),page.getSize());
        }
        return this.getBaseMapper().findPromotersByTerminalCodes(page,terminalCodeList);
    }
}
