package com.biz.crm.mdm.business.promoters.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.promoters.local.entity.Promoters;
import com.biz.crm.mdm.business.promoters.local.repository.PromotersRepository;
import com.biz.crm.mdm.business.promoters.sdk.constant.PromotersConstant;
import com.biz.crm.mdm.business.promoters.sdk.dto.PromotersDto;
import com.biz.crm.mdm.business.promoters.sdk.dto.PromotersEventDto;
import com.biz.crm.mdm.business.promoters.sdk.dto.PromotersSelectDto;
import com.biz.crm.mdm.business.promoters.sdk.event.PromotersEventListener;
import com.biz.crm.mdm.business.promoters.sdk.service.PromotersVoService;
import com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo;
import com.biz.crm.mn.common.base.constant.CommonSelectConstant;
import com.biz.crm.mn.third.system.two.center.sdk.service.TwoCenterApiService;
import com.biz.crm.mn.third.system.two.center.sdk.vo.ExtraVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 促销人员信息(Promoters)表服务实现类
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.4 23:13
 */
@Slf4j
@Service
public class PromotersVoServiceImpl implements PromotersVoService {

    @Autowired(required = false)
    private PromotersRepository promotersRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private TwoCenterApiService twoCenterApiService;

    /**
     * 查询分页列表
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:52
     */
    @Override
    public Page<PromotersVo> findByConditions(Pageable pageable, PromotersDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new PromotersDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<PromotersVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.promotersRepository.findByConditions(page, dto);
    }

    /**
     * 按id查询详情
     *
     * @param id 员工ID
     * @return java.util.List<com.biz.crm.mdm.business.promoters.local.entity.Promoters>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public PromotersVo findById(String id) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        Promoters promoters = this.promotersRepository.findById(id);
        if (Objects.isNull(promoters)) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByBlankList(
                promoters, PromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按id集合查询详情
     *
     * @param idList 员工ID集合
     * @return java.util.List<com.biz.crm.mdm.business.promoters.local.entity.Promoters>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public List<PromotersVo> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyList();
        }
        List<Promoters> promotersList = this.promotersRepository.findByIds(idList);
        if (CollectionUtils.isEmpty(promotersList)) {
            return Collections.emptyList();
        }

        return (List<PromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                promotersList, Promoters.class, PromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按编码查询详情
     *
     * @param code 编码
     * @return java.util.List<com.biz.crm.mdm.business.promoters.local.entity.Promoters>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public PromotersVo findByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        Promoters entity = this.promotersRepository.findByCode(code);
        if (Objects.isNull(entity)) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByBlankList(
                entity, PromotersVo.class,
                HashSet.class, ArrayList.class);
    }


    /**
     * 按编码集合查询详情
     *
     * @param codeList 编码集合
     * @return java.util.List<com.biz.crm.mdm.business.promoters.local.entity.Promoters>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public List<PromotersVo> findByCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        List<Promoters> entityList = this.promotersRepository.findByCodes(codeList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }

        return (List<PromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, Promoters.class, PromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按身份证号集合查询详情
     *
     * @param idCardList 身份证号集合
     * @return List<PromotersVo>
     */
    @Override
    public List<PromotersVo> findByIdCard(List<String> idCardList) {
        if (CollectionUtils.isEmpty(idCardList)) {
            return Collections.emptyList();
        }
        List<Promoters> entityList = this.promotersRepository.lambdaQuery()
                .eq(Promoters::getTenantCode, TenantUtils.getTenantCode())
                .eq(Promoters::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(Promoters::getIdentityCard, idCardList)
                .list();
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }

        return (List<PromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, Promoters.class, PromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 批量新增或修改
     *
     * @param dtoList 数据集合
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.5 16:42
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<PromotersVo> createOrUpdate(List<PromotersDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return Collections.emptyList();
        }
        //基础验证和构建
        Set<String> codeSet = this.baseValAndBuild(dtoList);
        String redisKey = PromotersConstant.MDM_LOCK_EMPLOYEE_CODE;
        Set<String> hasLockSet = new HashSet<>(codeSet.size());
        try {
            codeSet.forEach(code -> {
                boolean lock = redisMutexService.tryLock(redisKey + code, TimeUnit.MINUTES, 30);
                Assert.isTrue(lock, "促销人员[" + code + "]其他人员正在操作,请稍后再试!");
                hasLockSet.add(code);
            });
            List<Promoters> oldEntityList = this.promotersRepository.findByCodes(new ArrayList<>(codeSet));
            Map<String, Promoters> oldMap = new HashMap<>(codeSet.size());
            if (CollectionUtils.isNotEmpty(oldEntityList)) {
                oldEntityList.forEach(entity -> {
                    oldMap.put(entity.getEmployeeCode(), entity);
                });
            }
            List<Promoters> entityList = (List<Promoters>) this.nebulaToolkitService.copyCollectionByBlankList(
                    dtoList, PromotersDto.class, Promoters.class,
                    HashSet.class, ArrayList.class);
            //新增集合
            List<Promoters> addList = this.buildAddMap(entityList, oldMap);
            //更新集合
            List<Promoters> updateList = this.buildUpdateMap(entityList, oldMap);
            if (CollectionUtils.isNotEmpty(addList)) {
                this.promotersRepository.saveBatch(addList);
            }
            if (CollectionUtils.isNotEmpty(updateList)) {
                this.promotersRepository.updateBatchById(updateList);
            }
            //保存日志
            this.saveLog(addList, updateList, oldMap);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            if (CollectionUtils.isNotEmpty(hasLockSet)) {
                hasLockSet.forEach(code -> {
                    redisMutexService.unlock(redisKey + code);
                });
            }
        }
        List<Promoters> resultList = this.promotersRepository.findByCodes(new ArrayList<>(codeSet));
        if (CollectionUtils.isEmpty(resultList)) {
            return Collections.emptyList();
        }
        return (List<PromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                resultList, Promoters.class, PromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 基础构建和验证
     *
     * @param dtoList
     * @return
     */
    private Set<String> baseValAndBuild(List<PromotersDto> dtoList) {
        Set<String> codeSet = new HashSet<>(dtoList.size());
        dtoList.forEach(dto -> {
            this.commonValidate(dto);
            String code = dto.getEmployeeCode();
            Assert.isTrue(!codeSet.contains(code), "当前集合内编码[" + code + "]重复");
            if (StringUtils.isEmpty(dto.getDelFlag())) {
                dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            }
            if (StringUtils.isEmpty(dto.getEnableStatus())) {
                dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            }
            dto.setTenantCode(TenantUtils.getTenantCode());
            codeSet.add(code);
        });
        return codeSet;
    }

    /**
     * 保存日志
     *
     * @param addList    新增集合
     * @param updateList 更新集合
     * @param oldMap     更新前数据Map
     */
    private void saveLog(List<Promoters> addList,
                         List<Promoters> updateList,
                         Map<String, Promoters> oldMap) {
        if (CollectionUtils.isNotEmpty(addList)) {
            List<PromotersDto> list = (List<PromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                    addList, Promoters.class, PromotersDto.class,
                    HashSet.class, ArrayList.class);
            list.forEach(dto -> {
                PromotersEventDto eventDto = new PromotersEventDto();
                eventDto.setNewest(dto);
                SerializableBiConsumer<PromotersEventListener, PromotersEventDto> consumer = PromotersEventListener::onCreate;
                this.nebulaNetEventClient.publish(eventDto, PromotersEventListener.class, consumer);

            });
        }
        if (CollectionUtils.isNotEmpty(updateList)) {
            List<PromotersDto> list = (List<PromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                    updateList, Promoters.class, PromotersDto.class,
                    HashSet.class, ArrayList.class);
            list.forEach(dto -> {
                PromotersEventDto eventDto = new PromotersEventDto();
                eventDto.setOriginal(this.nebulaToolkitService.copyObjectByBlankList(
                        oldMap.getOrDefault(dto.getEmployeeCode(), new Promoters()), PromotersDto.class,
                        HashSet.class, ArrayList.class));
                eventDto.setNewest(dto);
                SerializableBiConsumer<PromotersEventListener, PromotersEventDto> consumer = PromotersEventListener::onUpdate;
                this.nebulaNetEventClient.publish(eventDto, PromotersEventListener.class, consumer);
            });
        }
    }

    /**
     * 构建新增集合
     *
     * @param entityList 入参
     * @param oldMap     数据库数据
     * @return
     */
    private List<Promoters> buildAddMap(List<Promoters> entityList, Map<String, Promoters> oldMap) {
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        List<Promoters> result = new ArrayList<>(entityList.size());
        entityList.stream()
                .filter(k -> !oldMap.containsKey(k.getEmployeeCode()))
                .filter(k -> Objects.isNull(oldMap.get(k.getEmployeeCode())))
                .forEach(result::add);
        return result;
    }


    /**
     * 构建更新集合
     *
     * @param entityList 入参
     * @param oldMap     数据库数据
     * @return
     */
    private List<Promoters> buildUpdateMap(List<Promoters> entityList, Map<String, Promoters> oldMap) {
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        List<Promoters> result = new ArrayList<>(entityList.size());
        entityList.stream()
                .filter(k -> oldMap.containsKey(k.getEmployeeCode()))
                .filter(k -> Objects.nonNull(oldMap.get(k.getEmployeeCode())))
                .forEach(entity -> {
                    Promoters oldEntity = oldMap.get(entity.getEmployeeCode());
                    entity.setId(oldEntity.getId());
                    entity.setCreateTime(oldEntity.getCreateTime());
                    entity.setCreateName(oldEntity.getCreateName());
                    entity.setCreateAccount(oldEntity.getCreateAccount());
                    result.add(entity);
                });
        return result;
    }


    /**
     * 新增
     *
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public PromotersVo create(PromotersDto dto) {
        this.createValidate(dto);
        String code = dto.getEmployeeCode();
        String redisKey = PromotersConstant.MDM_LOCK_EMPLOYEE_CODE + code;
        boolean lock = redisMutexService.tryLock(redisKey, TimeUnit.MINUTES, 3);
        Assert.isTrue(lock, "促销人员[" + code + "]其他人员正在操作,请稍后再试!");
        try {
            int count = this.promotersRepository.countByCode(code);
            Assert.isTrue(count < 1, "促销人员[" + code + "]已存在,不可新增!");
            Promoters entity = this.nebulaToolkitService.copyObjectByBlankList(
                    dto, Promoters.class,
                    HashSet.class, ArrayList.class);
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            this.promotersRepository.saveOrUpdate(entity);

            PromotersEventDto eventDto = new PromotersEventDto();
            eventDto.setOriginal(null);
            eventDto.setNewest(this.nebulaToolkitService.copyObjectByBlankList(
                    entity, PromotersDto.class,
                    HashSet.class, ArrayList.class));
            SerializableBiConsumer<PromotersEventListener, PromotersEventDto> consumer = PromotersEventListener::onCreate;
            this.nebulaNetEventClient.publish(eventDto, PromotersEventListener.class, consumer);

            //更新数据字典信息
            return this.nebulaToolkitService.copyObjectByBlankList(
                    entity, PromotersVo.class,
                    HashSet.class, ArrayList.class);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            redisMutexService.unlock(redisKey);
        }

    }

    /**
     * 修改
     *
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public PromotersVo update(PromotersDto dto) {
        this.updateValidate(dto);
        String code = dto.getEmployeeCode();
        String redisKey = PromotersConstant.MDM_LOCK_EMPLOYEE_CODE + code;
        boolean lock = redisMutexService.tryLock(redisKey, TimeUnit.MINUTES, 3);
        Assert.isTrue(lock, "促销人员[" + code + "]其他人员正在操作,请稍后再试!");
        try {
            PromotersEventDto eventDto = new PromotersEventDto();
            eventDto.setOriginal(dto);
            Promoters oldEntity = this.promotersRepository.findByCode(code);
            Assert.notNull(oldEntity, "促销人员[" + code + "]不存在,编辑失败!");
            dto.setId(oldEntity.getId());
            Promoters entity = this.nebulaToolkitService.copyObjectByBlankList(
                    dto, Promoters.class,
                    HashSet.class, ArrayList.class);
            this.promotersRepository.saveOrUpdate(entity);

            eventDto.setNewest(this.nebulaToolkitService.copyObjectByBlankList(
                    entity, PromotersDto.class,
                    HashSet.class, ArrayList.class));
            SerializableBiConsumer<PromotersEventListener, PromotersEventDto> consumer = PromotersEventListener::onUpdate;
            this.nebulaNetEventClient.publish(eventDto, PromotersEventListener.class, consumer);


            //更新数据字典信息
            return this.nebulaToolkitService.copyObjectByBlankList(
                    entity, PromotersVo.class,
                    HashSet.class, ArrayList.class);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            redisMutexService.unlock(redisKey);
        }
    }

    /**
     * 删除
     *
     * @param idList
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<Promoters> entityList = this.promotersRepository.findByIds(idList);
        entityList.forEach(entity -> {
            String code = entity.getEmployeeCode();
            entity.setEmployeeCode(code + entity.getId());
            entity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
        });
        this.promotersRepository.updateBatchById(entityList);
        List<PromotersDto> list = (List<PromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, Promoters.class, PromotersDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(dto -> {
            PromotersEventDto eventDto = new PromotersEventDto();
            eventDto.setOriginal(dto);
            eventDto.setNewest(null);
            SerializableBiConsumer<PromotersEventListener, PromotersEventDto> consumer = PromotersEventListener::onDelete;
            this.nebulaNetEventClient.publish(eventDto, PromotersEventListener.class, consumer);

        });
    }

    /**
     * 新建校验
     *
     * @param dto
     */
    private void createValidate(PromotersDto dto) {
        this.commonValidate(dto);
        dto.setId(null);

    }

    /**
     * 修改校验
     *
     * @param dto
     */
    private void updateValidate(PromotersDto dto) {
        this.commonValidate(dto);
        Validate.notBlank(dto.getId(), "更新时数据主键不能为空！");
        Promoters businessUnit = this.promotersRepository.findById(dto.getId());
        Validate.notNull(businessUnit, "更新数据不存在！");
        Validate.isTrue(businessUnit.getEmployeeCode().equals(dto.getEmployeeCode()),
                "编码不可修改！");
    }

    /**
     * 基础共同校验
     *
     * @param dto
     */
    private void commonValidate(PromotersDto dto) {
        Validate.notBlank(dto.getEmployeeCode(), "编码不能为空！");
        Validate.notBlank(dto.getName(), "名称不能为空！");
    }

    /**
     * 批量启用
     *
     * @param ids ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待启用的数据主键不能为空");
        List<Promoters> entityList = this.promotersRepository.findByIds(ids);
        Validate.notEmpty(entityList, "待启用的数据不存在或已删除!");
        this.promotersRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
        List<PromotersDto> list = (List<PromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, Promoters.class, PromotersDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(dto -> {
            PromotersEventDto eventDto = new PromotersEventDto();
            eventDto.setOriginal(dto);
            eventDto.setNewest(null);
            SerializableBiConsumer<PromotersEventListener, PromotersEventDto> consumer = PromotersEventListener::onEnable;
            this.nebulaNetEventClient.publish(eventDto, PromotersEventListener.class, consumer);
        });
    }

    /**
     * 批量禁用
     *
     * @param ids ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待禁用的数据主键不能为空");
        List<Promoters> entityList = this.promotersRepository.findByIds(ids);
        Validate.notEmpty(entityList, "待禁用的数据不存在或已删除!");
        this.promotersRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
        List<PromotersDto> list = (List<PromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, Promoters.class, PromotersDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(entity -> {
            PromotersEventDto eventDto = new PromotersEventDto();
            eventDto.setOriginal(entity);
            eventDto.setNewest(null);
            SerializableBiConsumer<PromotersEventListener, PromotersEventDto> consumer = PromotersEventListener::onDisable;
            this.nebulaNetEventClient.publish(eventDto, PromotersEventListener.class, consumer);
        });
    }

    @Override
    public List<PromotersVo> findSelectList(PromotersSelectDto dto) {

        if (StringUtils.isNotEmpty(dto.getSelectedCode())) {
            List<String> selectCodeList = Optional.ofNullable(dto.getSelectedCodeList()).orElse(Lists.newArrayList());
            selectCodeList.add(dto.getSelectedCode());
            dto.setSelectedCodeList(selectCodeList);
        }
        return promotersRepository.findSelectList(dto, CommonSelectConstant.COMMON_SELECT_DEFAULT_SIZE);
    }

    @Override
    public List<PromotersVo> findTerminalPromoters(PromotersDto dto) {

        Validate.notBlank(dto.getRegionCode(),"区域不能为空");
//        Validate.notBlank(dto.getCustomerRetailerCode(),"零售商不能为空");


        return promotersRepository.findTerminalPromoters(dto);
    }

    /**
     * 根据门店查询促销门店人员
     *
     * @param dto
     * @return
     */
    @Override
    public List<PromotersVo> findTerminalPromotersByTerminal(PromotersDto dto) {
        List<PromotersVo> voList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(dto.getTerminalCodeSet()) || StringUtils.isEmpty(dto.getUserType())) {
            return voList;
        }
        voList = promotersRepository.findTerminalPromotersByTerminal(dto);
        return voList;
    }

    /**
     * 同步
     * @param dto
     */
    @Override
    public void sync(PromotersDto dto) {
        List<com.biz.crm.mn.third.system.two.center.sdk.vo.PromotersVo> promotersVos = twoCenterApiService.pullTwoCenterEmployeeData(dto.getEmployeeCode());
        if(org.springframework.util.CollectionUtils.isEmpty(promotersVos)){
            return;
        }
        List<PromotersDto> promotersDtoList = (List<PromotersDto>) this.nebulaToolkitService.copyCollectionByWhiteList(promotersVos, com.biz.crm.mn.third.system.two.center.sdk.vo.PromotersVo.class, PromotersDto.class,
            HashSet.class,
            ArrayList.class);
        PromotersDto promotersDto = promotersDtoList.get(0);
        com.biz.crm.mn.third.system.two.center.sdk.vo.PromotersVo promotersVo = promotersVos.get(0);
        ExtraVo extra = promotersVo.getExtra();
        promotersDto.setInMengNiuDate(extra.getHireDate());
        promotersDto.setInVerticalDate(extra.getHireCzDate());
        promotersDto.setRegularDate(extra.getEmpConfireDate());
        promotersDto.setQuitDate(extra.getTerminationDate());
        promotersDto.setCityLevel(extra.getCityLevel());
        promotersDto.setResponsibleSupervision(extra.getSupervisionName());
        promotersDto.setResponsibleBusiness(extra.getBusinesserName());
        this.createOrUpdate(promotersDtoList);
    }
}
