package com.biz.crm.mdm.business.promoters.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.promoters.local.entity.TerminalPromoters;
import com.biz.crm.mdm.business.promoters.local.repository.TerminalPromotersRepository;
import com.biz.crm.mdm.business.promoters.sdk.constant.PromotersConstant;
import com.biz.crm.mdm.business.promoters.sdk.dto.TerminalPromotersDto;
import com.biz.crm.mdm.business.promoters.sdk.dto.TerminalPromotersEventDto;
import com.biz.crm.mdm.business.promoters.sdk.event.TerminalPromotersEventListener;
import com.biz.crm.mdm.business.promoters.sdk.service.TerminalPromotersVoService;
import com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo;
import com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 促销人员信息(TerminalPromoters)表服务实现类
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.4 23:13
 */
@Slf4j
@Service
public class TerminalPromotersVoServiceImpl implements TerminalPromotersVoService {

    @Autowired(required = false)
    private TerminalPromotersRepository terminalPromotersRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    /**
     * 查询分页列表
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:52
     */
    @Override
    public Page<TerminalPromotersVo> findByConditions(Pageable pageable, TerminalPromotersDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new TerminalPromotersDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<TerminalPromotersVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.terminalPromotersRepository.findByConditions(page, dto);
    }


    /**
     * 按id查询详情
     *
     * @param id ID
     * @return com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.11 16:02
     */
    @Override
    public TerminalPromotersVo findById(String id) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        TerminalPromoters promoters = this.terminalPromotersRepository.findById(id);
        if (Objects.isNull(promoters)) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByBlankList(
                promoters, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }


    /**
     * 按id集合查询详情
     *
     * @param idList ID集合
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.11 16:03
     */
    @Override
    public List<TerminalPromotersVo> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyList();
        }
        List<TerminalPromoters> promotersList = this.terminalPromotersRepository.findByIds(idList);
        if (CollectionUtils.isEmpty(promotersList)) {
            return Collections.emptyList();
        }

        return (List<TerminalPromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                promotersList, TerminalPromoters.class, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按门店编码查询详情
     *
     * @param code 门店编码
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.11 16:03
     */
    @Override
    public List<TerminalPromotersVo> findByTerminalCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByTerminalCode(code);
        if (Objects.isNull(entityList)) {
            return null;
        }
        return (List<TerminalPromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按门店编码集合查询详情
     *
     * @param codeList 门店编码集合
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.11 16:04
     */
    @Override
    public List<TerminalPromotersVo> findByTerminalCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByTerminalCodes(codeList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }

        return (List<TerminalPromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }


    /**
     * 按员工编码查询详情
     *
     * @param code 员工编码
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.11 16:03
     */
    @Override
    public List<TerminalPromotersVo> findByEmployeeCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByEmployeeCode(code);
        if (Objects.isNull(entityList)) {
            return null;
        }
        return (List<TerminalPromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按员工编码集合查询详情
     *
     * @param codeList 员工编码集合
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.11 16:04
     */
    @Override
    public List<TerminalPromotersVo> findByEmployeeCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByEmployeeCodes(codeList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }

        return (List<TerminalPromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }


    /**
     * 按经销商编码查询详情
     *
     * @param code 经销商编码
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.11 16:03
     */
    @Override
    public List<TerminalPromotersVo> findByCustomerCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByCustomerCode(code);
        if (Objects.isNull(entityList)) {
            return null;
        }
        return (List<TerminalPromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按经销商编码集合查询详情
     *
     * @param codeList 经销商编码集合
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.11 16:04
     */
    @Override
    public List<TerminalPromotersVo> findByCustomerCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByCustomerCodes(codeList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }

        return (List<TerminalPromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 批量新增或修改
     *
     * @param dtoList 数据集合
     * @return java.util.List<com.biz.crm.mdm.business.promoters.sdk.vo.TerminalPromotersVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.5 16:42
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<TerminalPromotersVo> createOrUpdate(List<TerminalPromotersDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return Collections.emptyList();
        }
        //基础验证和构建
        dtoList = this.baseValAndBuild(dtoList);
        if (CollectionUtils.isEmpty(dtoList)) {
            return Collections.emptyList();
        }
        Set<String> codeSet = dtoList.stream()
                .map(TerminalPromotersDto::getUniqueCode)
                .collect(Collectors.toSet());
        String redisKey = PromotersConstant.MDM_LOCK_TERMINAL_EMPLOYEE_CODE;
        Set<String> hasLockSet = new HashSet<>(codeSet.size());
        try {
            codeSet.forEach(code -> {
                boolean lock = redisMutexService.tryLock(redisKey + code, TimeUnit.MINUTES, 30);
                Assert.isTrue(lock, "门店促销人员[" + code + "]其他人员正在操作,请稍后再试!");
                hasLockSet.add(code);
            });
            List<TerminalPromoters> oldEntityList = this.terminalPromotersRepository.findByUniqueCodes(new ArrayList<>(codeSet));
            Map<String, TerminalPromoters> oldMap = new HashMap<>(codeSet.size());
            if (CollectionUtils.isNotEmpty(oldEntityList)) {
                oldEntityList.forEach(entity -> {
                    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    oldMap.put(entity.getUniqueCode(), entity);
                });
            }
            List<TerminalPromoters> entityList = (List<TerminalPromoters>) this.nebulaToolkitService.copyCollectionByBlankList(
                    dtoList, TerminalPromotersDto.class, TerminalPromoters.class,
                    HashSet.class, ArrayList.class);
            //新增集合
            List<TerminalPromoters> addList = this.buildAddMap(entityList, oldMap);
            //更新集合
            List<TerminalPromoters> updateList = this.buildUpdateMap(entityList, oldMap);
            if (CollectionUtils.isNotEmpty(addList)) {
                this.terminalPromotersRepository.saveBatch(addList);
            }
            if (CollectionUtils.isNotEmpty(updateList)) {
                this.terminalPromotersRepository.updateBatchById(updateList);
            }
            //保存日志
            this.saveLog(addList, updateList, oldMap);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            if (CollectionUtils.isNotEmpty(hasLockSet)) {
                hasLockSet.forEach(code -> {
                    redisMutexService.unlock(redisKey + code);
                });
            }
        }
        List<TerminalPromoters> resultList = this.terminalPromotersRepository.findByUniqueCodes(new ArrayList<>(codeSet));
        if (CollectionUtils.isEmpty(resultList)) {
            return Collections.emptyList();
        }
        return (List<TerminalPromotersVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                resultList, TerminalPromoters.class, TerminalPromotersVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 基础构建和验证
     *
     * @param dtoList
     * @return
     */
    private List<TerminalPromotersDto> baseValAndBuild(List<TerminalPromotersDto> dtoList) {
        dtoList.forEach(this::commonValidateAndSetUniqueCode);
        String tenantCode = TenantUtils.getTenantCode();
        dtoList.forEach(dto -> {
            this.commonValidateAndSetUniqueCode(dto);
            if (StringUtils.isEmpty(dto.getDelFlag())) {
                dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            }
            if (StringUtils.isEmpty(dto.getEnableStatus())) {
                dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            }
            dto.setTenantCode(tenantCode);
        });
        dtoList = dtoList.stream().collect(
                Collectors.collectingAndThen(
                        Collectors.toCollection(() ->
                                new TreeSet<>(Comparator.comparing(TerminalPromotersDto::getUniqueCode))), ArrayList::new));
        return dtoList;
    }

    /**
     * 保存日志
     *
     * @param addList    新增集合
     * @param updateList 更新集合
     * @param oldMap     更新前数据Map
     */
    private void saveLog(List<TerminalPromoters> addList,
                         List<TerminalPromoters> updateList,
                         Map<String, TerminalPromoters> oldMap) {
        if (CollectionUtils.isNotEmpty(addList)) {
            List<TerminalPromotersDto> list = (List<TerminalPromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                    addList, TerminalPromoters.class, TerminalPromotersDto.class,
                    HashSet.class, ArrayList.class);
            list.forEach(dto -> {
                TerminalPromotersEventDto eventDto = new TerminalPromotersEventDto();
                eventDto.setNewest(dto);
                SerializableBiConsumer<TerminalPromotersEventListener, TerminalPromotersEventDto> consumer = TerminalPromotersEventListener::onCreate;
                this.nebulaNetEventClient.publish(eventDto, TerminalPromotersEventListener.class, consumer);

            });
        }
        if (CollectionUtils.isNotEmpty(updateList)) {
            List<TerminalPromotersDto> list = (List<TerminalPromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                    updateList, TerminalPromoters.class, TerminalPromotersDto.class,
                    HashSet.class, ArrayList.class);
            list.forEach(dto -> {
                TerminalPromotersEventDto eventDto = new TerminalPromotersEventDto();
                eventDto.setOriginal(this.nebulaToolkitService.copyObjectByBlankList(
                        oldMap.getOrDefault(dto.getEmployeeCode(), new TerminalPromoters()), TerminalPromotersDto.class,
                        HashSet.class, ArrayList.class));
                eventDto.setNewest(dto);
                SerializableBiConsumer<TerminalPromotersEventListener, TerminalPromotersEventDto> consumer = TerminalPromotersEventListener::onUpdate;
                this.nebulaNetEventClient.publish(eventDto, TerminalPromotersEventListener.class, consumer);
            });
        }
    }

    /**
     * 构建新增集合
     *
     * @param entityList 入参
     * @param oldMap     数据库数据
     * @return
     */
    private List<TerminalPromoters> buildAddMap(List<TerminalPromoters> entityList, Map<String, TerminalPromoters> oldMap) {
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        List<TerminalPromoters> result = new ArrayList<>(entityList.size());
        entityList.stream()
                .filter(k -> !oldMap.containsKey(k.getUniqueCode()))
                .filter(k -> Objects.isNull(oldMap.get(k.getUniqueCode())))
                .forEach(result::add);
        return result;
    }


    /**
     * 构建更新集合
     *
     * @param entityList 入参
     * @param oldMap     数据库数据
     * @return
     */
    private List<TerminalPromoters> buildUpdateMap(List<TerminalPromoters> entityList, Map<String, TerminalPromoters> oldMap) {
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        List<TerminalPromoters> result = new ArrayList<>(entityList.size());
        entityList.stream()
                .filter(k -> oldMap.containsKey(k.getUniqueCode()))
                .filter(k -> Objects.nonNull(oldMap.get(k.getUniqueCode())))
                .forEach(entity -> {
                    TerminalPromoters oldEntity = oldMap.get(entity.getUniqueCode());
                    entity.setId(oldEntity.getId());
                    entity.setCreateTime(oldEntity.getCreateTime());
                    entity.setCreateName(oldEntity.getCreateName());
                    entity.setCreateAccount(oldEntity.getCreateAccount());
                    result.add(entity);
                });
        return result;
    }


    /**
     * 新增
     *
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TerminalPromotersVo create(TerminalPromotersDto dto) {
        this.createValidateSetUniqueCode(dto);
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        String code = dto.getUniqueCode();
        String redisKey = PromotersConstant.MDM_LOCK_TERMINAL_EMPLOYEE_CODE + code;
        boolean lock = redisMutexService.tryLock(redisKey, TimeUnit.MINUTES, 3);
        Assert.isTrue(lock, "门店促销人员[" + code + "]其他人员正在操作,请稍后再试!");
        try {
            int count = this.terminalPromotersRepository.countByUniqueCode(code);
            Assert.isTrue(count < 1, "门店促销人员[" + code + "]已存在,不可新增!");
            TerminalPromoters entity = this.nebulaToolkitService.copyObjectByBlankList(
                    dto, TerminalPromoters.class,
                    HashSet.class, ArrayList.class);
            this.terminalPromotersRepository.saveOrUpdate(entity);

            TerminalPromotersEventDto eventDto = new TerminalPromotersEventDto();
            eventDto.setOriginal(null);
            eventDto.setNewest(this.nebulaToolkitService.copyObjectByBlankList(
                    entity, TerminalPromotersDto.class,
                    HashSet.class, ArrayList.class));
            SerializableBiConsumer<TerminalPromotersEventListener, TerminalPromotersEventDto> consumer = TerminalPromotersEventListener::onCreate;
            this.nebulaNetEventClient.publish(eventDto, TerminalPromotersEventListener.class, consumer);

            //更新数据字典信息
            return this.nebulaToolkitService.copyObjectByBlankList(
                    entity, TerminalPromotersVo.class,
                    HashSet.class, ArrayList.class);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            redisMutexService.unlock(redisKey);
        }

    }

    /**
     * 修改
     *
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TerminalPromotersVo update(TerminalPromotersDto dto) {
        this.updateValidateSetUniqueCode(dto);
        String code = dto.getUniqueCode();
        String redisKey = PromotersConstant.MDM_LOCK_TERMINAL_EMPLOYEE_CODE + code;
        boolean lock = redisMutexService.tryLock(redisKey, TimeUnit.MINUTES, 3);
        Assert.isTrue(lock, "门店促销人员[" + code + "]其他人员正在操作,请稍后再试!");
        try {

            TerminalPromoters uniqueOldEntity = this.terminalPromotersRepository.findByUniqueCode(code);

            TerminalPromotersEventDto eventDto = new TerminalPromotersEventDto();
            eventDto.setOriginal(dto);
            TerminalPromoters oldEntity = this.terminalPromotersRepository.findById(dto.getId());
            Assert.notNull(oldEntity, "门店促销人员[" + code + "]不存在,编辑失败!");
            dto.setId(oldEntity.getId());
            if (uniqueOldEntity != null) {
                Assert.isTrue(uniqueOldEntity.getId().equals(oldEntity.getId()), "门店促销人员[" + code + "]已存在,编辑失败!");
            }
            TerminalPromoters entity = this.nebulaToolkitService.copyObjectByBlankList(
                    dto, TerminalPromoters.class,
                    HashSet.class, ArrayList.class);
            this.terminalPromotersRepository.saveOrUpdate(entity);

            eventDto.setNewest(this.nebulaToolkitService.copyObjectByBlankList(
                    entity, TerminalPromotersDto.class,
                    HashSet.class, ArrayList.class));
            SerializableBiConsumer<TerminalPromotersEventListener, TerminalPromotersEventDto> consumer = TerminalPromotersEventListener::onUpdate;
            this.nebulaNetEventClient.publish(eventDto, TerminalPromotersEventListener.class, consumer);

            //更新数据字典信息
            return this.nebulaToolkitService.copyObjectByBlankList(
                    entity, TerminalPromotersVo.class,
                    HashSet.class, ArrayList.class);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            redisMutexService.unlock(redisKey);
        }
    }

    /**
     * 删除
     *
     * @param idList
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByIds(idList);
        entityList.forEach(entity -> {
            String code = entity.getEmployeeCode();
            entity.setEmployeeCode(code + entity.getId());
            entity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
        });
        this.terminalPromotersRepository.updateBatchById(entityList);
        List<TerminalPromotersDto> list = (List<TerminalPromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(dto -> {
            TerminalPromotersEventDto eventDto = new TerminalPromotersEventDto();
            eventDto.setOriginal(dto);
            eventDto.setNewest(null);
            SerializableBiConsumer<TerminalPromotersEventListener, TerminalPromotersEventDto> consumer = TerminalPromotersEventListener::onDelete;
            this.nebulaNetEventClient.publish(eventDto, TerminalPromotersEventListener.class, consumer);

        });
    }

    /**
     * 新建校验
     *
     * @param dto
     */
    private void createValidateSetUniqueCode(TerminalPromotersDto dto) {
        this.commonValidateAndSetUniqueCode(dto);
        dto.setId(null);
    }

    /**
     * 修改校验
     *
     * @param dto
     */
    private void updateValidateSetUniqueCode(TerminalPromotersDto dto) {
        this.commonValidateAndSetUniqueCode(dto);
        TerminalPromoters businessUnit = this.terminalPromotersRepository.findById(dto.getId());
        Validate.notNull(businessUnit, "更新数据不存在！");
    }

    /**
     * 基础共同校验
     *
     * @param dto
     */
    private void commonValidateAndSetUniqueCode(TerminalPromotersDto dto) {
        dto.setTerminalCode(StringUtils.trimToEmpty(dto.getTerminalCode()));
        Validate.notBlank(dto.getTerminalCode(), "门店编码不能为空！");
        dto.setEmployeeCode(StringUtils.trimToEmpty(dto.getEmployeeCode()));
        Validate.notBlank(dto.getEmployeeCode(), "促销人员编码不能为空！");
        dto.setCustomerCode(StringUtils.trimToEmpty(dto.getCustomerCode()));
        dto.setUniqueCode(dto.getTerminalCode() + dto.getEmployeeCode() + dto.getCustomerCode());
    }

    /**
     * 批量启用
     *
     * @param ids ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待启用的数据主键不能为空");
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByIds(ids);
        Validate.notEmpty(entityList, "待启用的数据不存在或已删除!");
        this.terminalPromotersRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
        List<TerminalPromotersDto> list = (List<TerminalPromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(dto -> {
            TerminalPromotersEventDto eventDto = new TerminalPromotersEventDto();
            eventDto.setOriginal(dto);
            eventDto.setNewest(null);
            SerializableBiConsumer<TerminalPromotersEventListener, TerminalPromotersEventDto> consumer = TerminalPromotersEventListener::onEnable;
            this.nebulaNetEventClient.publish(eventDto, TerminalPromotersEventListener.class, consumer);
        });
    }

    /**
     * 批量禁用
     *
     * @param ids ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待禁用的数据主键不能为空");
        List<TerminalPromoters> entityList = this.terminalPromotersRepository.findByIds(ids);
        Validate.notEmpty(entityList, "待禁用的数据不存在或已删除!");
        this.terminalPromotersRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
        List<TerminalPromotersDto> list = (List<TerminalPromotersDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, TerminalPromoters.class, TerminalPromotersDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(entity -> {
            TerminalPromotersEventDto eventDto = new TerminalPromotersEventDto();
            eventDto.setOriginal(entity);
            eventDto.setNewest(null);
            SerializableBiConsumer<TerminalPromotersEventListener, TerminalPromotersEventDto> consumer = TerminalPromotersEventListener::onDisable;
            this.nebulaNetEventClient.publish(eventDto, TerminalPromotersEventListener.class, consumer);
        });
    }

    @Override
    public Page<PromotersVo> findPromotersByTerminalCodes(Pageable pageable, List<String> terminalCodeList) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));

        if (CollectionUtils.isEmpty(terminalCodeList)) {
            return new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        }
        Page<TerminalPromotersVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.terminalPromotersRepository.findPromotersByTerminalCodes(page, terminalCodeList);
    }


}
