package com.biz.crm.mdm.business.promotion.material.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.promotion.material.local.service.OrderQualityLevelService;
import com.biz.crm.mdm.business.promotion.material.local.service.PromotionMaterialImageService;
import com.biz.crm.mdm.business.promotion.material.sdk.dto.OrderQualityLevelDto;
import com.biz.crm.mdm.business.promotion.material.sdk.dto.PromotionMaterialDto;
import com.biz.crm.mdm.business.promotion.material.sdk.service.PromotionMaterialService;
import com.biz.crm.mdm.business.promotion.material.sdk.vo.OrderQualityLevelVO;
import com.biz.crm.mdm.business.promotion.material.sdk.vo.PromotionMaterialVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * 促销物料管理(mdm_promotion_material)表 控制层
 *
 * @author qiancheng
 * @since 2022-11-22
 */
@RestController
@RequestMapping("/v1/promotionMaterial/promotionMaterial")
@Slf4j
@Api(tags = "促销物料管理")
public class PromotionMaterialController {

    @Autowired
    private PromotionMaterialService promotionMaterialService;

    @Autowired
    private OrderQualityLevelService orderQualityLevelService;

    @Autowired
    private PromotionMaterialImageService promotionMaterialImageService;


    /**
     * 分页查询数据
     *
     * @param pageable          分页对象
     * @param promotionMaterial 实体对象
     * @return 分页对象
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("/findByConditions")
    public Result<Page<PromotionMaterialVO>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                              @ApiParam(name = "promotionMaterial", value = "促销物料管理") PromotionMaterialDto promotionMaterial) {
        try {
            Page<PromotionMaterialVO> page = this.promotionMaterialService.findByConditions(pageable, promotionMaterial);
            return Result.ok(page);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 物料下拉
     *
     * @param pageable        分页对象
     * @param promotionMaterial 查询dto
     * @return 分页对象
     */
    @ApiOperation(value = "物料下拉")
    @GetMapping("findMaterialSelectList")
    public Result<Page<PromotionMaterialVO>> findMaterialSelectList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                              @ApiParam(name = "promotionMaterial", value = "促销物料管理") PromotionMaterialDto promotionMaterial) {
        try {
            Page<PromotionMaterialVO> page = this.promotionMaterialService.findMaterialSelectList(pageable, promotionMaterial);
            return Result.ok(page);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过促销物料编码 查询物料详情
     *
     * @param materialCode 促销物料编码
     * @return 促销物料
     */
    @ApiOperation(value = "通过促销物料编码 查询物料详情")
    @GetMapping("/findByMaterialCode")
    public Result<PromotionMaterialVO> findByMaterialCode(@RequestParam("materialCode") @ApiParam(name = "materialCode", value = "促销物料编码") String materialCode) {
        try {
            PromotionMaterialVO promotionMaterialVO = promotionMaterialService.findByIdOrMaterialCode(null,materialCode);
            return Result.ok(promotionMaterialVO);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键 查询物料详情
     *
     * @param id 主键
     * @return 促销物料
     */
    @ApiOperation(value = "通过主键 查询物料详情")
    @GetMapping("/findById")
    public Result<PromotionMaterialVO> findById(@RequestParam("id") @ApiParam(name = "id", value = "主键") String id) {
        try {
            PromotionMaterialVO promotionMaterialVO = promotionMaterialService.findByIdOrMaterialCode(id,null);
            return Result.ok(promotionMaterialVO);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过促销物料编码 查询订单数量坎级
     *
     * @param materialCode 促销物料编码
     * @return 促销物料坎级集合
     */
    @ApiOperation(value = "通过促销物料编码 查询订单数量坎级")
    @GetMapping("/findLevelByMaterialCode")
    public Result<List<OrderQualityLevelVO>> findLevelByMaterialCode(@RequestParam("materialCode") @ApiParam(name = "materialCode", value = "促销物料编码") String materialCode) {
        try {
            List<OrderQualityLevelVO> levelList = orderQualityLevelService.findByMaterialCode(materialCode);
            return Result.ok(levelList);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询数据
     *
     * @param dto 实体对象
     * @return 分页对象
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("/findPriceByMaterialCode")
    public Result<OrderQualityLevelVO> findPriceByMaterialCode(@ApiParam(name = "dto", value = "促销物料") OrderQualityLevelDto dto) {
        try {
            OrderQualityLevelVO vo = this.promotionMaterialService.findPriceByMaterialCode(dto);
            return Result.ok(vo);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 新增数据
     *
     * @param promotionMaterial 实体对象
     */
    @ApiOperation(value = "新增数据")
    @PostMapping()
    public Result<?> create(@ApiParam(name = "promotionMaterial", value = "促销物料管理") @RequestBody PromotionMaterialDto promotionMaterial) {
        try {
            this.promotionMaterialService.create(promotionMaterial);
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 修改新据
     *
     * @param promotionMaterial 实体对象
     */
    @ApiOperation(value = "修改数据")
    @PatchMapping
    public Result<?> update(@ApiParam(name = "promotionMaterial", value = "促销物料管理") @RequestBody PromotionMaterialDto promotionMaterial) {
        try {
            this.promotionMaterialService.update(promotionMaterial);
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除促销物料
     *
     * @param ids 主键集合
     */
    @ApiOperation(value = "删除促销物料")
    @DeleteMapping
    public Result<?> delete(@ApiParam(name = "ids", value = "主键集合") @RequestParam("ids") List<String> ids) {
        try {
            this.promotionMaterialService.delete(ids);
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 启用
     *
     * @param ids 主键集合
     * @return 删除结果
     */
    @ApiOperation(value = "启用")
    @PatchMapping("/enable")
    public Result<?> enable(@ApiParam(name = "ids", value = "主键集合") @RequestBody List<String> ids) {
        try {
            this.promotionMaterialService.updateEnableStatus(ids, EnableStatusEnum.ENABLE.getCode());
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 禁用
     *
     * @param ids 主键集合
     * @return 删除结果
     */
    @ApiOperation(value = "禁用")
    @PatchMapping("/disable")
    public Result<?> disable(@ApiParam(name = "ids", value = "主键集合") @RequestBody List<String> ids) {
        try {
            this.promotionMaterialService.updateEnableStatus(ids, EnableStatusEnum.DISABLE.getCode());
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 获取物料名称
     *
     * @param codes 促销物料编码集合
     * @return Map<String, String>
     */
    @ApiOperation(value = "获取物料名称")
    @PostMapping("/findMaterialNameByCodes")
    public Result<Map<String, String>> findMaterialNameByCodes(@RequestBody List<String> codes) {
        try {
            Map<String, String> map = this.promotionMaterialService.findMaterialNameByCodes(codes);
            return Result.ok(map);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

}