package com.biz.crm.mdm.business.promotion.material.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.promotion.material.local.entity.OrderQualityLevel;
import com.biz.crm.mdm.business.promotion.material.local.repository.OrderQualityLevelRepository;
import com.biz.crm.mdm.business.promotion.material.local.service.OrderQualityLevelService;
import com.biz.crm.mdm.business.promotion.material.sdk.dto.OrderQualityLevelDto;
import com.biz.crm.mdm.business.promotion.material.sdk.vo.OrderQualityLevelVO;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

/**
 * 订单数量坎级(mdm_order_quality_level)表  服务实现类
 *
 * @author : qiancheng
 * @date : 2022-11-22
 */
@Slf4j
@Service("orderQualityLevelService")
public class OrderQualityLevelServiceImpl implements OrderQualityLevelService {

    @Autowired
    private OrderQualityLevelRepository orderQualityLevelRepository;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;


    /**
     * 通过促销物料编码 查询订单数量坎级
     * @param MaterialCode 促销物料编码
     * @return 坎级集合
     */
    @Override
    public List<OrderQualityLevelVO> findByMaterialCode(String MaterialCode) {
        if(!StringUtils.isBlank(MaterialCode)){
            List<OrderQualityLevelVO> levelVoList = orderQualityLevelRepository.findByMaterialCode(MaterialCode);
            return levelVoList;
        }
        return null;
    }


    /**
     * 新增订单数量坎级
     * @param materialCode 促销物料编码
     * @param orderQualityLevelList 实体对象集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void create(String materialCode,List<OrderQualityLevelDto> orderQualityLevelList) {
        if(!CollectionUtils.isEmpty(orderQualityLevelList)){
            orderQualityLevelList.forEach(level ->{
                level.setMaterialCode(materialCode);
                level.setTenantCode(TenantUtils.getTenantCode());
                level.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            });
            Collection<OrderQualityLevel> orderQualityLevels = nebulaToolkitService.copyCollectionByWhiteList(orderQualityLevelList, OrderQualityLevelDto.class, OrderQualityLevel.class, HashSet.class, ArrayList.class);
            orderQualityLevelRepository.saveBatch(orderQualityLevels);
        }
    }


    /**
     * 通过促销物料编码集合 删除数据
     * @param materialCodes 促销物料编码集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByMaterialCode(List<String> materialCodes) {
        if(CollectionUtils.isEmpty(materialCodes)){
            UpdateWrapper<OrderQualityLevel> wrapper = new UpdateWrapper();
            wrapper.set("del_flag", DelFlagStatusEnum.DELETE.getCode());
            wrapper.eq("tenant_code",TenantUtils.getTenantCode());
            wrapper.in("material_code",materialCodes);
            orderQualityLevelRepository.update(wrapper);
        }
    }

    /**
     * 通过订单数量坎级主键集合 删除数据
     * @param materialCode 物料编码
     * @param ids 订单数量坎级集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(String materialCode,Collection<String> ids) {
        if(!CollectionUtils.isEmpty(ids)){
            UpdateWrapper<OrderQualityLevel> wrapper = new UpdateWrapper();
            wrapper.set("del_flag", DelFlagStatusEnum.DELETE.getCode());
            wrapper.eq("material_code",materialCode);
            wrapper.notIn("id",ids);
            orderQualityLevelRepository.update(wrapper);
        }
    }

    /**
     * 新增/修改 订单数量坎级
     * @param levelList 订单数量坎级集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveOrUpdate(List<OrderQualityLevelDto> levelList) {
        if(!CollectionUtils.isEmpty(levelList)){
            Collection<OrderQualityLevel> orderQualityLevels = nebulaToolkitService.copyCollectionByWhiteList(levelList, OrderQualityLevelDto.class, OrderQualityLevel.class, HashSet.class, ArrayList.class);
            orderQualityLevelRepository.saveOrUpdateBatch(orderQualityLevels);
        }
    }
}
