package com.biz.crm.mdm.business.promotion.material.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.promotion.material.local.constant.PromotionMaterialConstant;
import com.biz.crm.mdm.business.promotion.material.local.entity.PromotionMaterial;
import com.biz.crm.mdm.business.promotion.material.local.repository.PromotionMaterialRepository;
import com.biz.crm.mdm.business.promotion.material.local.service.OrderQualityLevelService;
import com.biz.crm.mdm.business.promotion.material.local.service.PromotionMaterialImageService;
import com.biz.crm.mdm.business.promotion.material.sdk.dto.OrderQualityLevelDto;
import com.biz.crm.mdm.business.promotion.material.sdk.dto.PromotionMaterialDto;
import com.biz.crm.mdm.business.promotion.material.sdk.dto.PromotionMaterialImageDto;
import com.biz.crm.mdm.business.promotion.material.sdk.enums.PromotionMaterialEnum;
import com.biz.crm.mdm.business.promotion.material.sdk.service.PromotionMaterialService;
import com.biz.crm.mdm.business.promotion.material.sdk.vo.OrderQualityLevelVO;
import com.biz.crm.mdm.business.promotion.material.sdk.vo.PromotionMaterialImageVO;
import com.biz.crm.mdm.business.promotion.material.sdk.vo.PromotionMaterialVO;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 促销物料管理(mdm_promotion_material)表  服务实现类
 *
 * @author : qiancheng
 * @date : 2022-11-22
 */
@Slf4j
@Service("promotionMaterialService")
public class PromotionMaterialServiceImpl implements PromotionMaterialService {

    @Autowired
    private PromotionMaterialRepository promotionMaterialRepository;

    @Autowired
    private PromotionMaterialImageService promotionMaterialImageService;

    @Autowired
    private OrderQualityLevelService orderQualityLevelService;

    @Autowired
    private GenerateCodeService generateCodeService;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 分页查询数据
     * @param pageable        分页对象
     * @param promotionMaterial 实体对象
     * @return 分页对象
     */
    @Override
    public Page<PromotionMaterialVO> findByConditions(Pageable pageable, PromotionMaterialDto promotionMaterial) {
        if(pageable == null){
            pageable = PageRequest.of(0,20);
        }
        if(promotionMaterial == null){
            promotionMaterial = new PromotionMaterialDto();
        }
        if (StringUtils.isBlank(promotionMaterial.getDelFlag())) {
            promotionMaterial.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }
        if(StringUtils.isBlank(promotionMaterial.getTenantCode())){
            promotionMaterial.setTenantCode(TenantUtils.getTenantCode());
        }
        Page<PromotionMaterialVO> page = this.promotionMaterialRepository.findByConditions(pageable, promotionMaterial);
        page.getRecords().forEach(vo ->{
            List<PromotionMaterialImageVO> imageVoList = promotionMaterialImageService.findByMaterialCode(vo.getMaterialCode());
            vo.setMaterialImageList(imageVoList);
        });
        return page;
    }

    /**
     * 通过 主键 或 促销物料编码 查询 促销物料详情
     * @param id 主键
     * @param materialCode 促销物料编码
     * @return
     */
    @Override
    public PromotionMaterialVO findByIdOrMaterialCode(String id,String materialCode) {
        PromotionMaterial one = null;
        if(!StringUtils.isBlank(id)){
            QueryWrapper wrapper = new QueryWrapper();
            wrapper.eq("tenant_code",TenantUtils.getTenantCode());
            wrapper.eq("id",id);
            one = promotionMaterialRepository.getOne(wrapper);
        }else if(!StringUtils.isBlank(materialCode)){
            QueryWrapper wrapper = new QueryWrapper();
            wrapper.eq("tenant_code",TenantUtils.getTenantCode());
            wrapper.eq("material_code",materialCode);
            one = promotionMaterialRepository.getOne(wrapper);
        }
        if(one != null){
            PromotionMaterialVO materialVO = nebulaToolkitService.copyObjectByWhiteList(one, PromotionMaterialVO.class, HashSet.class, ArrayList.class);
            List<PromotionMaterialImageVO> imageVoList = promotionMaterialImageService.findByMaterialCode(one.getMaterialCode());
            List<OrderQualityLevelVO> levelVoList = orderQualityLevelService.findByMaterialCode(one.getMaterialCode());
            materialVO.setMaterialImageList(imageVoList);
            materialVO.setQualityLevelList(levelVoList);
            return materialVO;
        }
        return null;
    }

    /**
     * 新增数据
     * @param promotionMaterial 实体对象
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void create(PromotionMaterialDto promotionMaterial) {
        this.createValidate(promotionMaterial);
        promotionMaterial.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        promotionMaterial.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        promotionMaterial.setTenantCode(TenantUtils.getTenantCode());
        // redis生成营销策略编码，编码规则为MS+年月日+5位顺序数。每天都从00001开始
        String ruleCode = StringUtils.join(PromotionMaterialConstant.PROMOTION_MATERIAL_CODE_PREFIX, DateFormatUtils.format(new Date(), "yyyyMMdd"));
        String materialCode = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS).get(0);
        Validate.notBlank(materialCode,"物料编码生成失败");
        promotionMaterial.setMaterialCode(materialCode);
        //新增 物料图片
        if(!CollectionUtils.isEmpty(promotionMaterial.getMaterialImageList())){

            promotionMaterialImageService.create(promotionMaterial.getMaterialCode(),promotionMaterial.getMaterialImageList());
        }
        //新增 订单数量坎级
        if(!CollectionUtils.isEmpty(promotionMaterial.getQualityLevelList())){
            orderQualityLevelService.create(promotionMaterial.getMaterialCode(),promotionMaterial.getQualityLevelList());
        }
        PromotionMaterial promotionMaterialEntity = nebulaToolkitService.copyObjectByWhiteList(promotionMaterial, PromotionMaterial.class, HashSet.class, ArrayList.class);
        promotionMaterialRepository.save(promotionMaterialEntity);
    }


    /**
     * 修改新据
     * @param promotionMaterial 实体对象
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(PromotionMaterialDto promotionMaterial) {
        Validate.notNull(promotionMaterial,"对象信息不能为空！");
        PromotionMaterial dbMaterial = promotionMaterialRepository.getById(promotionMaterial.getId());
        if(dbMaterial != null){
            Validate.isTrue(dbMaterial.getMaterialCode().equals(promotionMaterial.getMaterialCode()) ,"促销物料编码不能修改");
            PromotionMaterial materialEntity = nebulaToolkitService.copyObjectByWhiteList(promotionMaterial, PromotionMaterial.class, HashSet.class, ArrayList.class);
            List<PromotionMaterialImageDto> materialImageList = promotionMaterial.getMaterialImageList();
            List<OrderQualityLevelDto> levelList = promotionMaterial.getQualityLevelList();
            if(!CollectionUtils.isEmpty(materialImageList)){
                materialImageList.forEach(image ->{
                    if(StringUtils.isBlank(image.getId())){
                        image.setMaterialCode(dbMaterial.getMaterialCode());
                        image.setTenantCode(TenantUtils.getTenantCode());
                        image.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    }
                });
                Set<String> fileCodes = materialImageList.stream().map(PromotionMaterialImageDto::getFileCode).filter(Objects::nonNull).collect(Collectors.toSet());
                promotionMaterialImageService.deleteByFileCode(dbMaterial.getMaterialCode(),fileCodes);
                promotionMaterialImageService.saveOrUpdate(materialImageList);
            }else {
                List<String> list = new ArrayList<>();
                list.add(dbMaterial.getMaterialCode());
                promotionMaterialImageService.deleteByMaterialCode(list);
            }

            if(!CollectionUtils.isEmpty(levelList)){
                levelList.forEach(level ->{
                    if(StringUtils.isBlank(level.getId())){
                        level.setMaterialCode(dbMaterial.getMaterialCode());
                        level.setTenantCode(TenantUtils.getTenantCode());
                        level.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    }
                });
                Set<String> levelIds = levelList.stream().map(OrderQualityLevelDto::getId).filter(Objects::nonNull).collect(Collectors.toSet());
                orderQualityLevelService.delete(dbMaterial.getMaterialCode(),levelIds);
                orderQualityLevelService.saveOrUpdate(levelList);
            }else{
                List<String> list = new ArrayList<>();
                list.add(dbMaterial.getMaterialCode());
                orderQualityLevelService.deleteByMaterialCode(list);
            }
            promotionMaterialRepository.updateById(materialEntity);
        }
    }

    /**
     * 通过促销物料编码集合 删除数据
     * @param ids 促销物料编码集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(List<String> ids) {
        Validate.isTrue(ids != null && ids.size() > 0,"主键不能为空！");
        List<String> codeList = new ArrayList<>();
        ids.forEach(id->{
            PromotionMaterial dbMaterial = promotionMaterialRepository.getById(id);
            codeList.add(dbMaterial.getMaterialCode());
        });
        promotionMaterialImageService.deleteByMaterialCode(codeList);
        orderQualityLevelService.deleteByMaterialCode(codeList);
        ids.forEach( code ->{
            UpdateWrapper wrapper = new UpdateWrapper();
            wrapper.set("del_flag",DelFlagStatusEnum.DELETE.getCode());
            wrapper.set("enable_status",EnableStatusEnum.DISABLE.getCode());
            wrapper.in("id",ids);
            promotionMaterialRepository.update(wrapper);
        });
    }

    /**
     * 启用、禁用
     * @param ids 主键集合
     * @param enableStatus 启用/禁用状态
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateEnableStatus(List<String> ids, String enableStatus) {
        Validate.isTrue(ids != null && ids.size() > 0 , "主键为空！");
        ids.forEach(id ->{
            UpdateWrapper wrapper = new UpdateWrapper();
            wrapper.set("enable_status",enableStatus);
            wrapper.eq("id",id);
            promotionMaterialRepository.update(wrapper);
        });
    }

    /**
     * 批量导入
     * @param importList 实体集合
     */
    @Override
    public void importSave(List<PromotionMaterialDto> importList) {
        Validate.notEmpty(importList,"导入数据不能为空！");
        importList.forEach( dto -> {
            String id = this.validateRepeat(dto);
            if(!StringUtils.isBlank(id)){
                dto.setId(id);
            }else {
                // redis生成物料编码，编码规则为WL+年月日+5位顺序数。每天都从00001开始
                String ruleCode = StringUtils.join(PromotionMaterialConstant.PROMOTION_MATERIAL_CODE_PREFIX, DateFormatUtils.format(new Date(), "yyyyMMdd"));
                String materialCode = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS).get(0);
                Validate.notBlank(materialCode,"物料编码生成失败");
                dto.setMaterialCode(materialCode);
            }
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setTenantCode(TenantUtils.getTenantCode());
        });

        List<PromotionMaterial> promotionMaterials = (List<PromotionMaterial>) nebulaToolkitService.copyCollectionByWhiteList(importList, PromotionMaterialDto.class, PromotionMaterial.class, HashSet.class, ArrayList.class);
        if(promotionMaterials.size() > 1000){
            Map<String, List<PromotionMaterial>> map = groupList(promotionMaterials);
            Set<String> strings = map.keySet();
            strings.forEach(key -> {
                saveOrUpdateBatch(map.get(key));
            });
        }else{
            saveOrUpdateBatch(promotionMaterials);
        }
    }

    /**
     *  重复校验
     * @param dto
     * @return
     */
    private String validateRepeat(PromotionMaterialDto dto){
        if(ObjectUtils.isEmpty(dto)){
            return null;
        }
        QueryWrapper<PromotionMaterial> wrapper = new QueryWrapper<>();
        wrapper.eq("material_name",dto.getMaterialName());
        PromotionMaterial one = promotionMaterialRepository.getOne(wrapper);
        if(!ObjectUtils.isEmpty(one)){
            return one.getId();
        }
        return null;
    }

    /**
     * 批量保存
     * @param promotionMaterials
     */
    @Transactional(rollbackFor = Exception.class)
    public void saveOrUpdateBatch( List<PromotionMaterial> promotionMaterials){
        if(!CollectionUtils.isEmpty(promotionMaterials)) {
            return;
        }
        promotionMaterialRepository.saveOrUpdateBatch(promotionMaterials);
    }

    /**
     * 分片
     * @param list
     * @return
     */
    public  static Map<String,List<PromotionMaterial>> groupList(List list){
        int groupSize = 1000;
        int listSize = list.size();
        int toIndex = groupSize;
        //用map存起来新的分组后数据
        Map<String,List<PromotionMaterial>> map = new HashMap();
        int keyToken = 0;
        for(int i = 0 ; i < list.size() ; i+=groupSize){
            //作用为toIndex最后没有100条数据则剩余几条newList中就装几条
            if(i+groupSize > listSize){
                toIndex=listSize-i;
            }
            List<PromotionMaterial> newList = list.subList(i,i+toIndex);
            map.put("wl"+keyToken, newList);
            keyToken++;
        }
        return map;
    }


    /**
     * 物料下拉
     *
     * @param pageable        分页对象
     * @param promotionMaterial 查询dto
     * @return 分页对象
     */
    @Override
    public Page<PromotionMaterialVO> findMaterialSelectList(Pageable pageable, PromotionMaterialDto promotionMaterial) {
        if(pageable == null){
            pageable = PageRequest.of(0,50);
        }
        if(promotionMaterial == null){
            promotionMaterial = new PromotionMaterialDto();
        }
        if (StringUtils.isBlank(promotionMaterial.getDelFlag())) {
            promotionMaterial.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }
        if(StringUtils.isBlank(promotionMaterial.getTenantCode())){
            promotionMaterial.setTenantCode(TenantUtils.getTenantCode());
        }
        Page<PromotionMaterialVO> page = this.promotionMaterialRepository.findMaterialSelectList(pageable, promotionMaterial);
        return page;
    }

    /**
     * 根据物料编码&数量找单价
     * feign未实现，禁止调用
     *
     * @param dto
     * @return
     */
    @Override
    public OrderQualityLevelVO findPriceByMaterialCode(OrderQualityLevelDto dto) {
        Validate.notBlank(dto.getMaterialCode(), "请填写物料编码！");
        String materialQuantityStr = dto.getMaterialQuantityStr();
        Validate.notBlank(materialQuantityStr, "请填写物料数量！");
        int quantity;
        try {
            quantity = Integer.parseInt(materialQuantityStr);
        } catch (Exception e) {
            throw new IllegalArgumentException("物料数量不为整数，请重新填写！");
        }
        List<OrderQualityLevelVO> levelList = orderQualityLevelService.findByMaterialCode(dto.getMaterialCode());
        if (CollectionUtils.isEmpty(levelList)) {
            throw new IllegalArgumentException("该物料未找到坎级，请维护！");
        }
        List<OrderQualityLevelVO> voList = Lists.newArrayList();
        levelList.forEach(level -> {
            if (level.getQualityOne() > quantity) {
                return;
            }
            if (level.getQualityOne() == quantity) {
                if (PromotionMaterialEnum.Symbol.LessThanSign.getValue().equals(level.getSymbolOne())) {
                    return;
                }
            }
            if (level.getQualityTwo() < quantity) {
                return;
            }
            if (level.getQualityOne() == quantity) {
                if (PromotionMaterialEnum.Symbol.LessThanSign.getValue().equals(level.getSymbolTwo())) {
                    return;
                }
            }
            voList.add(level);
        });
        if (CollectionUtils.isEmpty(voList)) {
            throw new IllegalArgumentException("根据物料+数量未找到坎级，请维护！");
        }
        log.info("voList:{}", voList);
        return voList.get(0);
    }

    /**
     * 获取物料名称
     *
     * @param codes 促销物料编码集合
     * @return Map<String, String>
     */
    @Override
    public Map<String, String> findMaterialNameByCodes(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)) {
            return Maps.newHashMap();
        }
        List<PromotionMaterial> list = this.promotionMaterialRepository.lambdaQuery()
                .eq(PromotionMaterial::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(PromotionMaterial::getTenantCode, TenantUtils.getTenantCode())
                .in(PromotionMaterial::getMaterialCode, codes)
                .list();
        Map<String, String> map = new HashMap<>();
        if (!CollectionUtils.isEmpty(list)) {
            map = list.stream().collect(Collectors.toMap(PromotionMaterial::getMaterialCode, PromotionMaterial::getMaterialName, (oldValue, newValue) -> newValue));
        }
        return map;
    }


    /**
     * 新增验证
     * @param promotionMaterial
     */
    private void createValidate(PromotionMaterialDto promotionMaterial) {
        Validate.notNull(promotionMaterial,"对象信息不能为空！");
        Validate.notBlank(promotionMaterial.getMaterialName(),"物料名称不能为空！");
        Validate.notBlank(promotionMaterial.getMaterialType(),"物料类别不能为空！");
        Validate.notNull(promotionMaterial.getPackageQuantityMin(),"最小包装量不能为空！");
        Validate.notBlank(promotionMaterial.getQualityStandard(),"质量标准不能为空！");
        Validate.notNull(promotionMaterial.getBoxSpecification(),"箱规不能为空！");
        Validate.notBlank(promotionMaterial.getMaterialSpecification(),"物料规格不能为空！");
        Validate.notBlank(promotionMaterial.getUnit(),"单位不能为空！");
        Validate.notNull(promotionMaterial.getBeginDate(),"有效期开始时间不能为空!");
        Validate.notNull(promotionMaterial.getEndDate(),"有效期结束时间不能为空！");
        Validate.notBlank(promotionMaterial.getSalesInstitutionCode(),"销售机构编码不能为空！");
        Validate.notBlank(promotionMaterial.getSalesInstitutionCode(),"销售机构名称不能为空！");
        Validate.notBlank(promotionMaterial.getSupplierCode(),"供应商编码不能为空！");
        Validate.notBlank(promotionMaterial.getSupplierName(),"供应商名称不能为空！");
        Validate.notBlank(promotionMaterial.getContractCode(),"关联合同编号不能为空！");
        Validate.notBlank(promotionMaterial.getContractName(),"关联合同名称不能为空！");
    }

}
