package com.biz.crm.mdm.business.region.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.service.RegionService;
import com.biz.crm.mdm.business.region.sdk.dto.RegionPaginationDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.List;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 行政区域管理
 *
 * @author sunx
 * @date 2021/10/8
 */
@Slf4j
@RestController
@RequestMapping("/v1/region/region")
@Api(tags = "行政区域管理: Region: 行政区域管理")
public class RegionController {

  @Autowired(required = false) private RegionService regionService;

  @ApiOperation(value = "查询分页列表", httpMethod = "GET")
  @GetMapping("/findByConditions")
  public Result<Page<Region>> findByConditions(
      @PageableDefault(50) Pageable pageable,
      @ApiParam(name = "regionDto", value = "请求参数") RegionPaginationDto regionPaginationDto) {
    try {
      Page<Region> result = this.regionService.findByConditions(pageable, regionPaginationDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "根据行政区域编码查询详情", httpMethod = "GET")
  @ApiImplicitParams({
    @ApiImplicitParam(
        name = "regionCode",
        value = "regionCode",
        required = false,
        dataType = "String",
        paramType = "query"),
  })
  @GetMapping("/findDetailByRegionCode")
  public Result<Region> findDetailByRegionCode(
      @RequestParam(value = "regionCode") String regionCode) {
    try {
      return Result.ok(this.regionService.findDetailByIdOrCode(null, regionCode));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据id获取行政区域信息
   *
   * @param id 主键
   */
  @ApiOperation(value = "根据id获取行政区域信息")
  @GetMapping(value = "/findDetailById")
  public Result<Region> findDetailById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      Region result = this.regionService.findDetailByIdOrCode(id, null);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /** 新增行政区域 */
  @ApiOperation(value = "新增行政区域")
  @PostMapping
  public Result<Region> create(
      @RequestBody @ApiParam(name = "region", value = "行政区域") Region region) {
    try {
      Region current = this.regionService.create(region);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /** 编辑行政区域 */
  @ApiOperation(value = "编辑行政区域")
  @PatchMapping
  public Result<Region> update(
      @RequestBody @ApiParam(name = "region", value = "行政区域") Region region) {
    try {
      Region current = this.regionService.update(region);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /** 删除 */
  @ApiOperation(value = "删除")
  @PatchMapping("/delete")
  public Result<?> delete(@RequestBody List<String> ids) {
    try {
      this.regionService.delete(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /** 启用 */
  @ApiOperation(value = "启用")
  @PatchMapping("/enable")
  public Result<?> enable(@RequestBody List<String> ids) {
    try {
      this.regionService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /** 禁用 */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody List<String> ids) {
    try {
      this.regionService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /** 行政区域下拉 */
  @ApiOperation(value = "行政区域下拉,不传parentCode默认查询的是中国,中国regionCode=00")
  @GetMapping("/findRegionSelect")
  public Result<List<Region>> findRegionSelect(
      @RequestParam(value = "parentCode", required = false) String parentCode,
      @RequestParam(value = "regionName", required = false) String regionName) {
    try {
      return Result.ok(this.regionService.findRegionSelect(parentCode, regionName));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
