package com.biz.crm.mdm.business.region.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.mapper.RegionMapper;
import com.biz.crm.mdm.business.region.sdk.dto.RegionPaginationDto;
import com.google.common.collect.Lists;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

/**
 * 行政区域
 *
 * @author sunx
 * @date 2021/10/8
 */
@Component
public class RegionRepository extends ServiceImpl<RegionMapper, Region> {

  /**
   * 分页
   *
   * @param page
   * @param regionPaginationDto
   * @return
   */
  public Page<Region> findByConditions(Page<Region> page, RegionPaginationDto regionPaginationDto) {
    return baseMapper.findByConditions(page, regionPaginationDto);
  }

  /**
   * 根据parentCode 获取所有下级
   *
   * @param parentCode
   * @return
   */
  public List<Region> findByParentCode(String parentCode) {
    if (StringUtils.isBlank(parentCode)) {
      return Lists.newLinkedList();
    }
    return baseMapper.findByParentCodeAndRegionNameLike(parentCode, StringUtils.EMPTY);
  }

  /**
   * 根据parentCode及regionName获取匹配数据
   *
   * @param parentCode
   * @return
   */
  public List<Region> findByParentCodeAndRegionNameLike(String parentCode, String regionName) {
    parentCode = Optional.ofNullable(parentCode).orElse(StringUtils.EMPTY);
    regionName = Optional.ofNullable(regionName).orElse(StringUtils.EMPTY);
    return baseMapper.findByParentCodeAndRegionNameLike(parentCode, regionName);
  }

  /**
   * 获取所有省的信息
   *
   * @return
   */
  public List<Region> findAllProvince() {
    return baseMapper.findByParentCodeAndRegionNameLike("00", StringUtils.EMPTY);
  }

  public Region findById(String id) {
    return this.baseMapper.selectById(id);
  }

  public Region findByRegionCode(String regionCode) {
    return lambdaQuery().eq(Region::getRegionCode, regionCode).one();
  }

  /**
   * 更新状态
   *
   * @param enable
   * @param ids
   */
  @Transactional
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    UpdateWrapper<Region> updateWrapper = new UpdateWrapper<>();
    updateWrapper.set("enable_status", enable.getCode());
    updateWrapper.in("id", ids);
    this.update(updateWrapper);
  }

  /**
   * 更新删除状态
   *
   * @param ids
   */
  @Transactional
  public void updateDeleteStatusByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    this.lambdaUpdate()
        .in(Region::getId, ids)
        .set(Region::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids
   * @return
   */
  public List<Region> findByIds(List<String> ids) {
    return this.lambdaQuery().in(Region::getId, ids).list();
  }

  /**
   * 根据编码获取对应集合详情
   *
   * @param regionCodeList
   * @return
   */
  public List<Region> findByRegionCodes(List<String> regionCodeList) {
    if (CollectionUtils.isEmpty(regionCodeList)) {
      return Lists.newLinkedList();
    }
    return lambdaQuery().in(Region::getRegionCode, regionCodeList).list();
  }

  /**
   * 根据行政区域名称模糊查询，返回对应层级的降维码
   *
   * @param levelNum
   * @param ruleCodeList
   * @param regionName
   * @return
   */
  public List<String> findRuleCodeListByParams(
      Integer levelNum, List<String> ruleCodeList, String regionName) {
    return this.baseMapper.findRuleCodeListByParams(levelNum, ruleCodeList, regionName);
  }

  /**
   * 根据降维码获取行政区域信息
   *
   * @param ruleCodeList
   * @return
   */
  public List<Region> findByRuleCods(List<String> ruleCodeList) {
    return this.lambdaQuery()
        .in(Region::getRuleCode, ruleCodeList)
        .orderByAsc(Region::getRegionCode)
        .list();
  }

  /**
   * 根据父级编码+名称获取匹配的行政区域信息
   *
   * @param parentCode
   * @param regionName
   * @return
   */
  public Region findByParentCodeAndRegionName(String parentCode, String regionName) {
    return this.lambdaQuery()
        .eq(Region::getParentCode, parentCode)
        .eq(Region::getRegionName, regionName)
        .one();
  }

  public List<Region> findDetailByCodes(List<String> regionCodeList) {
    return this.lambdaQuery()
        .in(Region::getRegionCode,regionCodeList)
        .list();
  }

  public List<Region> findSelfAndLowerByRuleCode(String ruleCode) {
    return this.baseMapper.findSelfAndLowerByRuleCode(ruleCode);
  }
}
