package com.biz.crm.mdm.business.region.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.dto.TreeDto;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.entity.RegionLabel;
import com.biz.crm.mdm.business.region.local.repository.RegionRepository;
import com.biz.crm.mdm.business.region.local.service.RegionLabelService;
import com.biz.crm.mdm.business.region.local.service.RegionService;
import com.biz.crm.mdm.business.region.sdk.constant.RegionConstant;
import com.biz.crm.mdm.business.region.sdk.dto.RegionPaginationDto;
import com.biz.crm.mdm.business.region.sdk.event.RegionEventListener;
import com.biz.crm.mdm.business.region.sdk.vo.RegionVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 行政区域接口实现
 *
 * @author sunx
 * @date 2021/10/8
 */
@Service(value = "regionService")
public class RegionServiceImpl implements RegionService {

  @Autowired(required = false)
  private RegionRepository regionRepository;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private List<RegionEventListener> regionEventListenerList;

  @Autowired(required = false)
  private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;

  @Autowired
  private RegionLabelService regionLabelService;

  /**
   * 行政区域分页信息
   *
   * @param pageable
   * @param regionPaginationDto
   * @return
   */
  @Override
  public Page<Region> findByConditions(Pageable pageable, RegionPaginationDto regionPaginationDto) {
    regionPaginationDto =
        Optional.ofNullable(regionPaginationDto).orElse(new RegionPaginationDto());
    regionPaginationDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Validate.isTrue(StringUtils.isBlank(regionPaginationDto.getRegionCode()) ? true : regionPaginationDto.getRegionCode().length() < 64, "行政区域编码，在进行搜索时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(StringUtils.isBlank(regionPaginationDto.getRegionName()) ? true : regionPaginationDto.getRegionName().length() < 128, "行政区域名称，在进行搜索时填入值超过了限定长度(128)，请检查!");
    Integer levelNum = 1;
    String ruleCode = "";
    if (StringUtils.isNotBlank(regionPaginationDto.getRegionCode())) {
      Region region = this.regionRepository.findByRegionCode(regionPaginationDto.getRegionCode());
      if (Objects.nonNull(region)) {
        levelNum = region.getLevelNum() + 1;
        ruleCode = region.getRuleCode();
      }
    }
    levelNum = Integer.min(levelNum, 5);

    List<String> ruleCodeList =
        this.regionRepository.findRuleCodeListByParams(
            levelNum, Lists.newArrayList(ruleCode), regionPaginationDto.getRegionName());
    //如果存在标签条件
    if (CollectionUtils.isNotEmpty(regionPaginationDto.getLabelList())){
      List<String> labelList = regionPaginationDto.getLabelList();
      List<String> ruleCodeByLabelList = this.regionLabelService.findRuleCodeByLabelList(labelList);
      Integer length=3*levelNum;
      List<String> collect = ruleCodeByLabelList.stream().filter(e -> e.length() >= (length)).collect(Collectors.toList());
      List<String> newRuleCodeByLabelList = new ArrayList<>();
      //根据级别裁剪
      if (CollectionUtils.isNotEmpty(collect)) {
        for (String code : collect) {
          code =code.substring(0,3*levelNum);
          newRuleCodeByLabelList.add(code);
        }
      }
      ruleCodeList.retainAll(newRuleCodeByLabelList);
    }

    if (CollectionUtils.isEmpty(ruleCodeList)) {
      return this.getPageRegionByList(null);
    }
    ruleCodeList =
        ruleCodeList.stream().filter(StringUtils::isNotBlank).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(ruleCodeList)) {
      return this.getPageRegionByList(null);
    }

    List<Region> list = this.regionRepository.findByRuleCods(ruleCodeList);
    if (CollectionUtils.isNotEmpty(list)) {
      list = this.selectLabel(list);
      //补充标签
      for (Region region : list) {
        region.setHasChild(true);
        if (levelNum == 5) {
          region.setHasChild(false);
        }
      }
    }
    return this.getPageRegionByList(list);
  }

  /**
   * 查询关联标签
   *
   * @param list
   */
  private List<Region> selectLabel(List<Region> list) {
    List<String> regionCodes = list.stream().map(Region::getRegionCode).collect(Collectors.toList());
    List<RegionLabel> regionLabels = this.regionLabelService.findByRegionCodes(regionCodes);
    if (CollectionUtils.isEmpty(regionLabels)){
      return list;
    }
    Map<String, List<RegionLabel>> map = regionLabels.stream().collect(Collectors.groupingBy(RegionLabel::getRegionCode));
    for (Region region : list) {
      region.setLableList(map.get(region.getRegionCode()));
    }
    return list;
  }

  /**
   * 根据行政区域id或编码获取详情
   *
   * @param id
   * @param regionCode
   * @return
   */
  @Override
  public Region findDetailByIdOrCode(String id, String regionCode) {
    if (StringUtils.isBlank(id) && StringUtils.isBlank(regionCode)) {
      return null;
    }
    if (StringUtils.isNotBlank(id)) {
      return regionRepository.findById(id);
    }
    return regionRepository.findByRegionCode(regionCode);
  }

  /**
   * 新增行政区域
   *
   * @param region
   * @return
   */
  @Override
  @Transactional
  public Region create(Region region) {
    Validate.isTrue(StringUtils.isNotBlank(region.getRegionCode()), "区域编码不能为空");
    Validate.isTrue(StringUtils.isNotBlank(region.getRegionName()), "区域名称不能为空");
    Validate.isTrue(region.getRegionCode().length() < 64, "行政区域编码，在进行输入时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(region.getRegionName().length() < 128, "行政区域名称，在进行输入时填入值超过了限定长度(128)，请检查!");
    Region cur = this.regionRepository.findByRegionCode(region.getRegionCode());
    Validate.isTrue(Objects.isNull(cur), "已存在当前行政区域编码的数据");
    region.setTenantCode(TenantUtils.getTenantCode());
    region.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    region.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    region.setParentCode(Optional.ofNullable(region.getParentCode()).orElse("00"));
    // 根据父级设置层级
    int levelNum = 1;
    if (StringUtils.isNotEmpty(region.getParentCode())) {
      Region parent = this.regionRepository.findByRegionCode(region.getParentCode());
      levelNum = parent.getLevelNum() + 1;
    }
    // 设置规则（降维）编码
    String ruleCode = this.getRuleCodeByParentCode(region.getParentCode());
    region.setRuleCode(ruleCode);
    region.setLevelNum(levelNum);
    regionRepository.save(region);
    return region;
  }

  /**
   * 编辑行政区域
   *
   * @param region
   * @return
   */
  @Override
  @Transactional
  public Region update(Region region) {
    Validate.isTrue(StringUtils.isNotBlank(region.getId()), "区域id不能为空");
    Validate.isTrue(StringUtils.isNotBlank(region.getRegionCode()), "区域编码不能为空");
    Validate.isTrue(StringUtils.isNotBlank(region.getRegionName()), "区域名称不能为空");
    Validate.isTrue(region.getRegionCode().length() < 64, "行政区域编码，在进行输入时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(region.getRegionName().length() < 128, "行政区域名称，在进行输入时填入值超过了限定长度(128)，请检查!");
    region.setParentCode(Optional.ofNullable(region.getParentCode()).orElse("00"));
    String currentId = region.getId();
    Region current = this.regionRepository.getById(currentId);
    Validate.notNull(current, "未发现指定的原始模型对象信");
    Validate.isTrue(region.getRegionCode().equals(current.getRegionCode()), "行政区域编码不能编辑");
    // 2.根据父级设置层级
    int levelNum = 1;
    if (StringUtils.isNotBlank(region.getParentCode())) {
      Region parent = this.regionRepository.findByRegionCode(region.getParentCode());
      levelNum = parent.getLevelNum() + 1;
    }
    region.setLevelNum(levelNum);
    this.regionRepository.updateById(region);

    if (!region.getParentCode().equals(current.getRegionCode())) {
      String ruleCode = getRuleCodeByParentCode(region.getParentCode());
      updateCurAndChildrenRuleCode(region.getRegionCode(), ruleCode, levelNum);
    }
    return current;
  }

  /**
   * 删除
   *
   * @param ids
   */
  @Override
  @Transactional
  public void delete(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.regionRepository.updateDeleteStatusByIds(ids);
    if (CollectionUtils.isNotEmpty(regionEventListenerList)) {
      List<Region> regions = this.regionRepository.findByIds(ids);
      List<RegionVo> voList =
          (List<RegionVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  regions, Region.class, RegionVo.class, HashSet.class, ArrayList.class);
      for (RegionEventListener event : regionEventListenerList) {
        event.onDelete(voList);
      }
    }
  }

  /**
   * 启用
   *
   * @param ids
   */
  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.regionRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  /**
   * 禁用
   *
   * @param ids
   */
  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.regionRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    if (CollectionUtils.isNotEmpty(regionEventListenerList)) {
      List<Region> regions = this.regionRepository.findByIds(ids);
      List<RegionVo> voList =
          (List<RegionVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  regions, Region.class, RegionVo.class, HashSet.class, ArrayList.class);
      for (RegionEventListener event : regionEventListenerList) {
        event.onDisable(voList);
      }
    }
  }

  @Override
  public List<Region> findRegionSelect(String parentCode, String regionName) {
    return regionRepository.findByParentCodeAndRegionNameLike(parentCode, regionName);
  }

  @Override
  public List<Region> findDetailByCodes(List<String> regionCodeList) {
    if (CollectionUtils.isEmpty(regionCodeList)) {
      return new ArrayList<>(0);
    }
    return regionRepository.findDetailByCodes(regionCodeList);
  }

  @Override
  public List<Region> findSelfAndLowerByRuleCode(String ruleCode) {
    if (StringUtils.isBlank(ruleCode)) {
      return new ArrayList<>(0);
    }
    return regionRepository.findSelfAndLowerByRuleCode(ruleCode);
  }

  /**
   * 更新自己及子集的降维编码
   *
   * @param regionCode
   * @param ruleCode
   * @param levelNum
   */
  @Transactional
  public void updateCurAndChildrenRuleCode(String regionCode, String ruleCode, int levelNum) {
    // 更新当前
    Region region = this.regionRepository.findByRegionCode(regionCode);
    region.setRuleCode(ruleCode);
    region.setLevelNum(levelNum);
    this.regionRepository.updateById(region);
    // 查询下一层
    List<Region> childrenList = regionRepository.findByParentCode(region.getRegionCode());
    if (org.springframework.util.CollectionUtils.isEmpty(childrenList)) {
      return;
    }
    // 遍历下级
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    for (int i = 0; i < childrenList.size(); i++) {
      // 递归调用
      updateCurAndChildrenRuleCode(
          childrenList.get(i).getRegionCode(),
          ruleCode + treeRuleCodeStrategy.generateByNum(RegionConstant.RULE_CODE_LENGTH, i + 1),
          levelNum + 1);
    }
  }

  /**
   * 根据集合返回page信息
   *
   * @param list
   * @return
   */
  private Page<Region> getPageRegionByList(List<Region> list) {
    if (CollectionUtils.isEmpty(list)) {
      list = Lists.newLinkedList();
    }
    Page<Region> page = new Page<>();
    page.setTotal(list.size());
    page.setSize(list.size());
    page.setRecords(list);
    return page;
  }

  /**
   * 根据父级编码获得当前降维编码
   *
   * @param parentCode
   * @return
   */
  private String getRuleCodeByParentCode(String parentCode) {
    parentCode = Optional.ofNullable(parentCode).orElse(StringUtils.EMPTY);
    String parentRuleCode = null;
    if (StringUtils.isNotEmpty(parentCode)) {
      Region parent = this.regionRepository.findByRegionCode(parentCode);
      Validate.notNull(parent, "上级行政区域不存在");
      parentRuleCode = parent.getRuleCode();
    }
    List<Region> childrenListByParentCode = this.regionRepository.findByParentCode(parentCode);
    List<TreeDto> childrenDto =
        Lists.newArrayList(
            nebulaToolkitService.copyCollectionByWhiteList(
                childrenListByParentCode,
                Region.class,
                TreeDto.class,
                HashSet.class,
                ArrayList.class));
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    return treeRuleCodeStrategy.generate(
        RegionConstant.RULE_CODE_LENGTH, parentRuleCode, childrenDto);
  }
}
