package com.biz.crm.mdm.business.region.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.mapper.RegionMapper;
import com.biz.crm.mdm.business.region.sdk.dto.RegionPaginationDto;
import com.biz.crm.mdm.business.region.sdk.vo.LazyTreeVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;

/**
 * 行政区域
 *
 * @author sunx
 * @date 2021/10/8
 */
@Component
public class RegionRepository extends ServiceImpl<RegionMapper, Region> {

  /**
   * 分页
   *
   * @param page
   * @param regionPaginationDto
   * @return
   */
  public Page<Region> findByConditions(Page<Region> page, RegionPaginationDto regionPaginationDto) {
    return baseMapper.findByConditions(page, regionPaginationDto, TenantUtils.getTenantCode());
  }

  /**
   * 根据parentCode 获取所有下级
   *
   * @param parentCode
   * @return
   */
  public List<Region> findByParentCode(String parentCode) {
    if (StringUtils.isBlank(parentCode)) {
      return Lists.newLinkedList();
    }
    return baseMapper.findByParentCodeAndRegionNameLike(parentCode, StringUtils.EMPTY, TenantUtils.getTenantCode());//新增租户编号条件
  }

  /**
   * 根据parentCode及regionName获取匹配数据
   *
   * @param parentCode
   * @return
   */
  public List<Region> findByParentCodeAndRegionNameLike(String parentCode, String regionName) {
    parentCode = Optional.ofNullable(parentCode).orElse(StringUtils.EMPTY);
    regionName = Optional.ofNullable(regionName).orElse(StringUtils.EMPTY);
    return baseMapper.findByParentCodeAndRegionNameLike(parentCode, regionName, TenantUtils.getTenantCode());//新增租户编号条件
  }

  /**
   * 获取所有省的信息
   *
   * @return
   */
  public List<Region> findAllProvince() {
    return baseMapper.findByParentCodeAndRegionNameLike("00", StringUtils.EMPTY, TenantUtils.getTenantCode());//新增租户编号条件
  }

  /**
   * 重构查询条件
   * @param id
   * @return
   */
  public Region findById(String id) {
    return this.lambdaQuery()
        .eq(Region::getTenantCode, TenantUtils.getTenantCode())
        .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(Region::getId, id)
        .one();
  }

  public Region findByRegionCode(String regionCode) {
    return lambdaQuery()
        .eq(Region::getRegionCode, regionCode)
        .eq(Region::getTenantCode, TenantUtils.getTenantCode())
        .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .one(); // 新增租户编号条件
  }

  public Region findByGovRegionCode(String govRegionCode) {
    return lambdaQuery().eq(Region::getGovRegionCode, govRegionCode).eq(Region::getTenantCode, TenantUtils.getTenantCode()).one();//新增租户编号条件
  }

  /**
   * 更新状态
   *
   * @param enable
   * @param ids
   */
  @Transactional
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    UpdateWrapper<Region> updateWrapper = new UpdateWrapper<>();
    updateWrapper.set("enable_status", enable.getCode());
    updateWrapper.in("id", ids);
    updateWrapper.in("tenant_code", TenantUtils.getTenantCode());//新增租户编号条件
    this.update(updateWrapper);
  }

  /**
   * 更新删除状态
   *
   * @param ids
   */
  @Transactional
  public void updateDeleteStatusByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    this.lambdaUpdate()
            .in(Region::getId, ids)
            .eq(Region::getTenantCode,TenantUtils.getTenantCode())   //新增租户编号
            .set(Region::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
            .update();
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids
   * @return
   */
  public List<Region> findByIds(List<String> ids) {

    return this.lambdaQuery()
            .eq(Region::getTenantCode,TenantUtils.getTenantCode())   //新增租户编号
            .in(Region::getId, ids)
            .list();
  }

  /**
   * 根据编码获取对应集合详情
   *
   * @param regionCodeList
   * @return
   */
  public List<Region> findByRegionCodes(List<String> regionCodeList) {
    if (CollectionUtils.isEmpty(regionCodeList)) {
      return Lists.newLinkedList();
    }
    return lambdaQuery()
        .in(Region::getRegionCode, regionCodeList)
        .in(Region::getTenantCode, TenantUtils.getTenantCode())
        .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .list(); // 新增租户编号条件
  }

  /**
   * 根据行政区域名称模糊查询，返回对应层级的降维码
   *
   * @param levelNum
   * @param ruleCodeList
   * @param regionName
   * @return
   */
  public List<String> findRuleCodeListByParams(
      Integer levelNum,
      List<String> ruleCodeList,
      String regionName,
      String regionCode,
      String regionLevel,
      String tenantCode) {
    return this.baseMapper.findRuleCodeListByParams(
        levelNum, ruleCodeList, regionName, regionCode, regionLevel, tenantCode);
  }

  /**
   * 根据降维码获取行政区域信息
   *
   * @param ruleCodeList
   * @return
   */
  public List<Region> findByRuleCods(List<String> ruleCodeList) {
    return this.lambdaQuery()
        .in(Region::getRuleCode, ruleCodeList)
        .eq(Region::getTenantCode, TenantUtils.getTenantCode()) // 新增租户编号条件
        .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()) // 新增租户编号条件
        .orderByAsc(Region::getRegionCode)
        .list();
  }

  /**
   * 根据父级编码+名称获取匹配的行政区域信息
   *
   * @param parentCode
   * @param regionName
   * @return
   */
  public Region findByParentCodeAndRegionName(String parentCode, String regionName) {
    return this.lambdaQuery()
        .eq(Region::getParentCode, parentCode)
        .eq(Region::getRegionName, regionName)
        .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Region::getTenantCode, TenantUtils.getTenantCode()) // 新增租户编号条件
        .one();
  }

  public List<Region> findDetailByCodes(List<String> regionCodeList) {
    return this.lambdaQuery()
        .in(Region::getRegionCode, regionCodeList)
        .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Region::getTenantCode, TenantUtils.getTenantCode()) // 新增租户编号条件
        .list();
  }

  public List<Region> findSelfAndLowerByRuleCode(String ruleCode) {
    return this.baseMapper.findSelfAndLowerByRuleCode(ruleCode, TenantUtils.getTenantCode());//新增租户编号条件
  }

  /**
   * 重构查询方法
   * @param currentId
   * @param tenantCode
   * @return
   */
  public Region findByIdAndTenantCode(String currentId, String tenantCode) {
    return this.lambdaQuery()
        .eq(Region::getTenantCode, tenantCode)
        .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Region::getId, currentId)
        .one();
  }

  /**
   * 重构修改方法
   * @param region
   * @param tenantCode
   */
  public void updateByIdAndTenantCode(Region region, String tenantCode) {
    LambdaUpdateWrapper<Region>lambdaUpdateWrapper= Wrappers.lambdaUpdate();
    lambdaUpdateWrapper.eq(Region::getTenantCode,tenantCode);
    lambdaUpdateWrapper.in(Region::getId,region.getId());
    this.baseMapper.update(region,lambdaUpdateWrapper);
  }

  public List<Region> findByRegionLevel(String regionLevel, String regionName) {
    return this.lambdaQuery()
            .eq(Region::getTenantCode,TenantUtils.getTenantCode())
            .eq(Region::getRegionLevel, regionLevel)
            .like(StringUtils.isNotEmpty(regionName), Region::getRegionName, regionName)
            .orderByAsc(Region::getRegionCode)
            .list();
  }

  /**
   * 根据编码获取对应集合详情
   *
   * @param govRegionCodeList
   * @return
   */
  public List<Region> findByGovRegionCodes(List<String> govRegionCodeList) {
    if (CollectionUtils.isEmpty(govRegionCodeList)) {
      return Lists.newLinkedList();
    }
    return lambdaQuery().in(Region::getGovRegionCode, govRegionCodeList).eq(Region::getTenantCode, TenantUtils.getTenantCode()).list();
  }

  /**
   * 根据当前组织编码查询下级组织信息
   *
   * @param parentCode
   * @param tenantCode
   * @return
   */
  public List<Region> findChildByParentCode(String parentCode, String tenantCode) {
    return this.baseMapper.findChildByParentCode(parentCode, tenantCode);
  }

  /**
   * 根据客户组织编码查询详情
   *
   * @param regionCode
   * @param tenantCode
   * @return
   */
  public Region findDetailsByCode(String regionCode, String tenantCode) {
    Validate.notBlank(regionCode, "行政区域编码不能为空");
    return this.lambdaQuery()
        .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Region::getTenantCode, tenantCode)
        .eq(Region::getRegionCode, regionCode)
        .one();
  }

  /**
   * 查询行政区域懒加载数据
   *
   * @param enableStatus
   * @param topOnly
   * @param parentCode
   * @param codeList
   * @param ruleCodeList
   * @param excludeRuleCode
   * @param tenantCode
   */
  public List<LazyTreeVo> findLazyTreeList(
      String enableStatus,
      Boolean topOnly,
      String parentCode,
      List<String> codeList,
      List<String> ruleCodeList,
      String excludeRuleCode,
      String tenantCode) {
    return this.baseMapper.findLazyTreeList(
        enableStatus,
        topOnly,
        parentCode,
        codeList,
        ruleCodeList,
        excludeRuleCode,
        tenantCode,
        DelFlagStatusEnum.NORMAL);
  }
}
