package com.biz.crm.mdm.business.region.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.dto.TreeDto;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.business.common.sdk.utils.TreeRuleCode;
import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.entity.RegionLabel;
import com.biz.crm.mdm.business.region.local.imports.vo.RegionCityLevelImportVO;
import com.biz.crm.mdm.business.region.local.repository.RegionRepository;
import com.biz.crm.mdm.business.region.local.service.RegionLabelService;
import com.biz.crm.mdm.business.region.local.service.RegionService;
import com.biz.crm.mdm.business.region.sdk.constant.RegionConstant;
import com.biz.crm.mdm.business.region.sdk.dto.RegionPaginationDto;
import com.biz.crm.mdm.business.region.sdk.enums.RegionLevelEnum;
import com.biz.crm.mdm.business.region.sdk.event.RegionEventListener;
import com.biz.crm.mdm.business.region.sdk.vo.RegionVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * 行政区域接口实现
 *
 * @author sunx
 * @date 2021/10/8
 */
@Service(value = "regionService")
public class RegionServiceImpl implements RegionService {

    @Autowired(required = false)
    private RegionRepository regionRepository;
    private static final int PROVINCE_SIZE = 4;
    private static final int MAX_RULECODE_IN_ONE_Level = 999;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    @Lazy
    private List<RegionEventListener> regionEventListenerList;

    @Autowired(required = false)
    private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;

    @Autowired
    private RegionLabelService regionLabelService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    /**
     * 行政区域分页信息
     *
     * @param pageable
     * @param regionPaginationDto
     * @return
     */
//    @Override
//    public Page<Region> findByConditions(Pageable pageable, RegionPaginationDto regionPaginationDto) {
//        regionPaginationDto =
//                Optional.ofNullable(regionPaginationDto).orElse(new RegionPaginationDto());
//        regionPaginationDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
//        regionPaginationDto.setTenantCode(TenantUtils.getTenantCode());//增加租户编号条件
//        Validate.isTrue(StringUtils.isBlank(regionPaginationDto.getRegionCode()) ? true : regionPaginationDto.getRegionCode().length() < 64, "行政区域编码，在进行搜索时填入值超过了限定长度(64)，请检查!");
//        Validate.isTrue(StringUtils.isBlank(regionPaginationDto.getRegionName()) ? true : regionPaginationDto.getRegionName().length() < 128, "行政区域名称，在进行搜索时填入值超过了限定长度(128)，请检查!");
//      Integer levelNum = 1;
////        if (null!= regionPaginationDto.getRegionLevel() && regionPaginationDto.getRegionLevel()>=0) {
////          levelNum = regionPaginationDto.getRegionLevel();
////        }
//        String ruleCode = "";
//        if (StringUtils.isNotBlank(regionPaginationDto.getRegionCode())) {
//            Region region = this.regionRepository.findByRegionCode(regionPaginationDto.getRegionCode());
//            if (Objects.nonNull(region)) {
//                levelNum = region.getLevelNum() + 1;
//                ruleCode = region.getRuleCode();
//            }
//        }else if(StringUtils.isNotBlank(regionPaginationDto.getGovRegionCode())){
//            Region region = this.regionRepository.findByGovRegionCode(regionPaginationDto.getGovRegionCode());
//            if (Objects.nonNull(region)) {
//                levelNum = region.getLevelNum() + 1;
//                ruleCode = region.getRuleCode();
//            }
//        }
//        levelNum = Integer.min(levelNum, 5);
//
//        List<String> ruleCodeList =
//                this.regionRepository.findRuleCodeListByParams(
//                        levelNum, Lists.newArrayList(ruleCode), regionPaginationDto.getRegionName(),regionPaginationDto.getParentCode());
//        //如果存在标签条件
//        if (CollectionUtils.isNotEmpty(regionPaginationDto.getLabelList())) {
//            List<String> labelList = regionPaginationDto.getLabelList();
//            List<String> ruleCodeByLabelList = this.regionLabelService.findRuleCodeByLabelList(labelList);
//            Integer length = 3 * levelNum;
//            List<String> collect = ruleCodeByLabelList.stream().filter(e -> e.length() >= (length)).collect(Collectors.toList());
//            List<String> newRuleCodeByLabelList = new ArrayList<>();
//            //根据级别裁剪
//            if (CollectionUtils.isNotEmpty(collect)) {
//                for (String code : collect) {
//                    code = code.substring(0, 3 * levelNum);
//                    newRuleCodeByLabelList.add(code);
//                }
//            }
//            ruleCodeList.retainAll(newRuleCodeByLabelList);
//        }
//
//        if (CollectionUtils.isEmpty(ruleCodeList)) {
//            return this.getPageRegionByList(null);
//        }
//        ruleCodeList =
//                ruleCodeList.stream().filter(StringUtils::isNotBlank).collect(Collectors.toList());
//        if (CollectionUtils.isEmpty(ruleCodeList)) {
//            return this.getPageRegionByList(null);
//        }
//
//        List<Region> list = this.regionRepository.findByRuleCods(ruleCodeList);
////        if (CollectionUtils.isNotEmpty(list)) {
////            list = this.selectLabel(list);
////            //补充标签
////            for (Region region : list) {
////                List<Region> childs = this.regionRepository.findByParentCode(region.getRegionCode());
////                if (CollectionUtils.isEmpty(childs)) {
////                    region.setHasChild(false);
////                } else {
////                    region.setHasChild(true);
////                }
////                if (levelNum == 5) {
////                    region.setHasChild(false);
////                }
////            }
////        }
//        return this.getPageRegionByList(list);
//    }
    @Override
    public Page<Region> findByConditions(Pageable pageable, RegionPaginationDto regionPaginationDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 200));
        if (Objects.isNull(regionPaginationDto)) {
            regionPaginationDto = new RegionPaginationDto();
        }
        //增加租户编号条件
        regionPaginationDto.setTenantCode(TenantUtils.getTenantCode());

        Validate.isTrue(StringUtils.isBlank(regionPaginationDto.getRegionCode()) ? true : regionPaginationDto.getRegionCode().length() < 64, "行政区域编码，在进行搜索时填入值超过了限定长度(64)，请检查!");
        Validate.isTrue(StringUtils.isBlank(regionPaginationDto.getRegionName()) ? true : regionPaginationDto.getRegionName().length() < 128, "行政区域名称，在进行搜索时填入值超过了限定长度(128)，请检查!");
//        Integer levelNum = 1;
        if (StringUtils.isNotBlank(regionPaginationDto.getParentCode())) {
            Region region = this.regionRepository.lambdaQuery()
                    .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                    .eq(Region::getRegionCode, regionPaginationDto.getParentCode())
                    .one();
            if (Objects.isNull(region)) {
                return this.getPageRegionByList(null);
            } else {
                regionPaginationDto.setRegionLevel(region.getRegionLevel() + 1);
                regionPaginationDto.setRegionCode(null);
                regionPaginationDto.setGovRegionCode(null);
                regionPaginationDto.setRegionName(null);
            }
        }

        if (null == regionPaginationDto.getRegionLevel() && StringUtils.isBlank(regionPaginationDto.getGovRegionCode())
                && StringUtils.isBlank(regionPaginationDto.getRegionCode()) && StringUtils.isBlank(regionPaginationDto.getRegionName())) {
            regionPaginationDto.setRegionLevel(1);
        }
        Page<Region> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<Region> result = this.regionRepository.findByConditions(page, regionPaginationDto);
        if (CollectionUtils.isEmpty(result.getRecords())) {
            return this.getPageRegionByList(null);
        }
        return this.getPageRegionByList(result.getRecords());
    }

    /**
     * 查询关联标签
     *
     * @param list
     */
    private List<Region> selectLabel(List<Region> list) {
        List<String> regionCodes = list.stream().map(Region::getRegionCode).collect(Collectors.toList());
        List<RegionLabel> regionLabels = this.regionLabelService.findByRegionCodes(regionCodes);
        if (CollectionUtils.isEmpty(regionLabels)) {
            return list;
        }
        Map<String, List<RegionLabel>> map = regionLabels.stream().collect(Collectors.groupingBy(RegionLabel::getRegionCode));
        for (Region region : list) {
            region.setLableList(map.get(region.getRegionCode()));
        }
        return list;
    }

    /**
     * 根据行政区域id或编码获取详情
     *
     * @param id
     * @param regionCode
     * @return
     */
    @Override
    public Region findDetailByIdOrCode(String id, String regionCode) {
        if (StringUtils.isBlank(id) && StringUtils.isBlank(regionCode)) {
            return null;
        }
      //获取多租户编码
      String tenantCode = TenantUtils.getTenantCode();
      tenantCode = StringUtils.isBlank(tenantCode) ? "default" : tenantCode;
      if (StringUtils.isNotBlank(id)) {
        Region region =  regionRepository.findById(id);
        //补充parentCode  provinceCode cityCode districtCode streetCode
        Region parentRegion = regionRepository.lambdaQuery()
                .eq(Region::getRegionCode, region.getParentCode())
                .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(Region::getTenantCode,tenantCode)
                .one();
        region.setParentName(parentRegion.getRegionName());

        List<String> codeList = new ArrayList<>(PROVINCE_SIZE);

        String tempParentCode = region.getParentCode();
        for (int i = 1; i < region.getRegionLevel(); i++) {
         Region tempRegion = regionRepository.lambdaQuery()
                  .eq(Region::getRegionCode, tempParentCode)
                  .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                  .eq(Region::getTenantCode, tenantCode)
                  .one();
          tempParentCode = tempRegion.getParentCode();
          codeList.add(tempRegion.getRegionName());
        }
        //数据库不存下面字段(省市区街道) 只展示给前端使用
        Collections.reverse(codeList);
        for (int i = 0; i < codeList.size(); i++) {
          switch (String.valueOf(i+1)){
            case "1":
              region.setProvinceCode(codeList.get(i));
              break;
            case "2":
              region.setCityCode(codeList.get(i));
              break;
            case "3":
              region.setDistrictCode(codeList.get(i));
              break;
            case "4":
              region.setStreetCode(codeList.get(i));
              break;
          }
        }

        return region;
      }
      return regionRepository.findByRegionCode(regionCode);
  }

    /**
     * 新增行政区域
     *
     * @param region
     * @return
     */
    @Override
    @Transactional
    public Region create(Region region) {
        Validate.isTrue(StringUtils.isNotBlank(region.getRegionCode()), "区域编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(region.getRegionName()), "区域名称不能为空");
        Validate.isTrue(region.getRegionCode().length() < 64, "行政区域编码，在进行输入时填入值超过了限定长度(64)，请检查!");
        Validate.isTrue(region.getRegionName().length() < 128, "行政区域名称，在进行输入时填入值超过了限定长度(128)，请检查!");
        Region cur = this.regionRepository.findByRegionCode(region.getRegionCode());
        Validate.isTrue(Objects.isNull(cur), "已存在当前行政区域编码的数据");
        region.setTenantCode(TenantUtils.getTenantCode());
        region.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        region.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        region.setParentCode(Optional.ofNullable(region.getParentCode()).orElse("00"));
        // 根据父级设置层级
        int levelNum = 1;
        if (StringUtils.isNotEmpty(region.getParentCode())) {
            Region parent = this.regionRepository.findByRegionCode(region.getParentCode());
            levelNum = parent.getLevelNum() + 1;
        }
      // 设置规则（降维）编码
      String ruleCode = null;
      //国家不允许创建 省份创建默认为1级ruleCode 1*3
      if(RegionLevelEnum.ONE.getDictCode().equals(String.valueOf(region.getRegionLevel()))) {
        ruleCode = this.getRuleCodeByProvince();
      }else if(!RegionLevelEnum.ZERO.getDictCode().equals(String.valueOf(region.getRegionLevel()))){
       ruleCode = this.getRuleCodeByParentCode(region.getParentCode());
      }
      region.setRuleCode(ruleCode);
      region.setLevelNum(levelNum);
      regionRepository.save(region);
      return region;
  }

    /**
     * 编辑行政区域
     *
     * @param region
     * @return
     */
    @Override
    @Transactional
    public Region update(Region region) {
        Validate.isTrue(StringUtils.isNotBlank(region.getId()), "区域id不能为空");
        Validate.isTrue(StringUtils.isNotBlank(region.getRegionCode()), "区域编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(region.getRegionName()), "区域名称不能为空");
        Validate.isTrue(region.getRegionCode().length() < 64, "行政区域编码，在进行输入时填入值超过了限定长度(64)，请检查!");
        Validate.isTrue(region.getRegionName().length() < 128, "行政区域名称，在进行输入时填入值超过了限定长度(128)，请检查!");
        region.setParentCode(Optional.ofNullable(region.getParentCode()).orElse("00"));
        String currentId = region.getId();
        //重构查询方法
        Region current = this.regionRepository.findByIdAndTenantCode(currentId, TenantUtils.getTenantCode());
        Validate.notNull(current, "未发现指定的原始模型对象信");
        Validate.isTrue(region.getRegionCode().equals(current.getRegionCode()), "行政区域编码不能编辑");
        // 2.根据父级设置层级
        int levelNum = 1;
        if (StringUtils.isNotBlank(region.getParentCode())) {
            Region parent = this.regionRepository.findByRegionCode(region.getParentCode());
            levelNum = parent.getLevelNum() + 1;
        }
        region.setLevelNum(levelNum);
        this.regionRepository.updateByIdAndTenantCode(region, TenantUtils.getTenantCode());

        if (!region.getParentCode().equals(current.getRegionCode())) {
            String ruleCode = getRuleCodeByParentCode(region.getParentCode());
            updateCurAndChildrenRuleCode(region.getRegionCode(), ruleCode, levelNum);
        }
        return current;
    }

    /**
     * 启用
     *
     * @param ids
     */
    @Override
    @Transactional
    public void enableBatch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.regionRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    }

    /**
     * 删除
     *
     * @param ids
     */
    @Override
    @Transactional
    public void delete(List<String> ids) {
          Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
          List<Region> regionList = this.regionRepository.findByIds(ids);
          for (Region item : regionList) {
            //找到子集
            int parentSize = this.regionRepository.lambdaQuery()
                .eq(Region::getParentCode, item.getRegionCode())
                .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .count();
            Validate.isTrue(parentSize <= 0,
                "区域名称:【" + item.getRegionName() + "】存在【" + parentSize + "】条下级关系，禁止删除!");
          }
          this.regionRepository.updateDeleteStatusByIds(ids);
          if (CollectionUtils.isNotEmpty(regionEventListenerList)) {
            List<Region> regions = this.regionRepository.findByIds(ids);
            List<RegionVo> voList =
                (List<RegionVo>)
                    this.nebulaToolkitService.copyCollectionByWhiteList(
                        regions, Region.class, RegionVo.class, HashSet.class, ArrayList.class);
            for (RegionEventListener event : regionEventListenerList) {
              event.onDelete(voList);
            }
          }
        }
    /**
     * 禁用
     *
     * @param ids
     */
    @Override
    @Transactional
    public void disableBatch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.regionRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
        if (CollectionUtils.isNotEmpty(regionEventListenerList)) {
            List<Region> regions = this.regionRepository.findByIds(ids);
            List<RegionVo> voList =
                    (List<RegionVo>)
                            this.nebulaToolkitService.copyCollectionByWhiteList(
                                    regions, Region.class, RegionVo.class, HashSet.class, ArrayList.class);
            for (RegionEventListener event : regionEventListenerList) {
                event.onDisable(voList);
            }
        }
    }

    @Override
    public List<Region> findRegionSelect(String parentCode, String regionName) {
        return regionRepository.findByParentCodeAndRegionNameLike(parentCode, regionName);
    }

    @Override
    public List<Region> findDetailByCodes(List<String> regionCodeList) {
        if (CollectionUtils.isEmpty(regionCodeList)) {
            return new ArrayList<>(0);
        }
        return regionRepository.findDetailByCodes(regionCodeList);
    }

    @Override
    public List<Region> findSelfAndLowerByRuleCode(String ruleCode) {
        if (StringUtils.isBlank(ruleCode)) {
            return new ArrayList<>(0);
        }
        return regionRepository.findSelfAndLowerByRuleCode(ruleCode);
    }

    @Override
    public List<RegionVo> findByRegionLevel(String regionLevel, String regionName){
              //默认查省一级数据
              if (StringUtils.isEmpty(regionLevel)) {
                regionLevel = RegionLevelEnum.ONE.getDictCode();
              }
              //直辖市没有市一级数据，目前暂时带出省数据
              List<Region> result = regionRepository.findByRegionLevel(regionLevel, regionName);
              if (RegionLevelEnum.TWO.getDictCode().equals(regionLevel)) {
                result.addAll(regionRepository.findByRegionLevel(RegionLevelEnum.ONE.getDictCode(),
                    regionName));
              }
              return (List<RegionVo>) nebulaToolkitService.copyCollectionByWhiteList(result,
                  Region.class, RegionVo.class, LinkedHashSet.class, ArrayList.class);
    }
    /**
     * 根据当前组织编码查询下级组织信息
     *
     * @param parentCode
     * @return
     */
    @Override
    public List<Region> findChildByParentCode(String parentCode) {
      String tenantCode = TenantUtils.getTenantCode();
      tenantCode = StringUtils.isBlank(tenantCode) ? "default" : tenantCode;
      List<Region> list = this.regionRepository.findChildByParentCode(parentCode, tenantCode);
      list.stream()
              .peek(a -> a.setHasChild(a.getHasChildFlag()==1))
              .collect(Collectors.toList());

      return list;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveCityLevel(LinkedHashMap<Integer, RegionCityLevelImportVO> data) {
        List<RegionCityLevelImportVO> importList = new ArrayList<>(data.values());
        Set<String> regionCodeSet = importList.stream().map(RegionCityLevelImportVO::getRegionCode).filter(Objects::nonNull).collect(Collectors.toSet());
        if (regionCodeSet.size() != importList.size()) {
            throw new IllegalArgumentException("导入数据有重复的行政区域编码或没有行政区域编码");
        }
        String CITY_LEVEL_DICT_CODE = "region_city_level";
        List<DictDataVo> cityLevelDict = dictDataVoService.findByDictTypeCode(CITY_LEVEL_DICT_CODE);
        Map<String, String> cityLevelDictMap = cityLevelDict.stream().collect(Collectors.toMap(DictDataVo::getDictValue, DictDataVo::getDictCode));
        List<Region> regionList = regionRepository.findByGovRegionCodes(new ArrayList<>(regionCodeSet));
        if(regionList.size() != importList.size()){
            throw new IllegalArgumentException("导入数据的行政区域编码存在无关联数据");
        }
        //数据类型转换
        Map<String, RegionCityLevelImportVO> importMap = importList.stream().collect(Collectors.toMap(RegionCityLevelImportVO::getRegionCode, o -> o));
        for(Region region : regionList){
            //暂时不限制区域等级
//            if(Integer.parseInt(RegionLevelEnum.ONE.getDictCode()) != region.getRegionLevel() &&
//                    Integer.parseInt(RegionLevelEnum.TWO.getDictCode()) != region.getRegionLevel()){
//                throw new IllegalArgumentException("行政区域不能设置城市等级");
//            }
            RegionCityLevelImportVO levelImportVO = importMap.get(region.getGovRegionCode());
            if(levelImportVO != null){
                region.setCityLevel(cityLevelDictMap.get(levelImportVO.getCityLevel()));
            }
        }
        regionRepository.updateBatchById(regionList);
    }

    /**
     * 更新自己及子集的降维编码
     *
     * @param regionCode
     * @param ruleCode
     * @param levelNum
     */
    @Transactional
    public void updateCurAndChildrenRuleCode(String regionCode, String ruleCode, int levelNum) {
        // 更新当前
        Region region = this.regionRepository.findByRegionCode(regionCode);
        region.setRuleCode(ruleCode);
        region.setLevelNum(levelNum);
        //重构修改方法
        this.regionRepository.updateByIdAndTenantCode(region, TenantUtils.getTenantCode());
        // 查询下一层
        List<Region> childrenList = regionRepository.findByParentCode(region.getRegionCode());
        if (org.springframework.util.CollectionUtils.isEmpty(childrenList)) {
            return;
        }
        // 遍历下级
        TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
        for (int i = 0; i < childrenList.size(); i++) {
            // 递归调用
            updateCurAndChildrenRuleCode(
                    childrenList.get(i).getRegionCode(),
                    ruleCode + treeRuleCodeStrategy.generateByNum(RegionConstant.RULE_CODE_LENGTH, i + 1),
                    levelNum + 1);
        }
    }

    /**
     * 根据集合返回page信息
     *
     * @param list
     * @return
     */
    private Page<Region> getPageRegionByList(List<Region> list) {
        if (CollectionUtils.isEmpty(list)) {
            list = Lists.newLinkedList();
        }
        Page<Region> page = new Page<>();
        page.setTotal(list.size());
        page.setSize(list.size());
        page.setRecords(list);
        return page;
    }

    /**
     * 根据父级编码获得当前降维编码
     *
     * @param parentCode
     * @return
     */
    private String getRuleCodeByParentCode(String parentCode){
          parentCode = Optional.ofNullable(parentCode).orElse(StringUtils.EMPTY);
          String parentRuleCode = null;
          if (StringUtils.isNotEmpty(parentCode)) {
            Region parent = this.regionRepository.findByRegionCode(parentCode);
            Validate.notNull(parent, "上级行政区域不存在");
            parentRuleCode = parent.getRuleCode();
          }
          List<Region> childrenListByParentCode = this.regionRepository.findByParentCode(
              parentCode);
          List<TreeDto> childrenDto =
              Lists.newArrayList(
                  nebulaToolkitService.copyCollectionByWhiteList(
                      childrenListByParentCode,
                      Region.class,
                      TreeDto.class,
                      HashSet.class,
                      ArrayList.class));
          TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(
              null);
          return treeRuleCodeStrategy.generate(
              RegionConstant.RULE_CODE_LENGTH, parentRuleCode, childrenDto);
    }
  /**
   * 省份不需要获取上级ruleCode 默认依次排序
   * @return
   */
  private String getRuleCodeByProvince(){
    String ruleCode = null;
    List<Region> regionList = this.regionRepository.lambdaQuery()
            .eq(Region::getRegionLevel, RegionLevelEnum.ONE.getDictCode())
            .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(Region::getTenantCode, TenantUtils.getTenantCode())
            .orderByAsc(Region::getRuleCode)
            .list();
    //获取当前层级的ruleCode 去重
    List<String> codeList = regionList.stream().map(Region::getRuleCode).collect(Collectors.toList());
    Set<Integer> set = new HashSet<>();
    for(String item : codeList){
      set.add(Integer.parseInt(item));
    }
    //ruleCode 在当前层级最大值下 寻找空位
    for (int i = 1; i <= MAX_RULECODE_IN_ONE_Level; i++) {
      Validate.isTrue(i != MAX_RULECODE_IN_ONE_Level, "降维编码越界，请联系管理员处理");
      if (!set.contains(i)) {
        ruleCode =  String.format("%03d", i);
        break;
      }
    }
    return ruleCode;
  }



}
