package com.biz.crm.mdm.business.region.local.service.internal;

import com.biz.crm.business.common.sdk.utils.TreeUtil;
import com.biz.crm.business.common.sdk.vo.TreeVo;
import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.repository.RegionRepository;
import com.biz.crm.mdm.business.region.sdk.enums.RegionLevelEnum;
import com.biz.crm.mdm.business.region.sdk.constant.RegionConstant;
import com.biz.crm.mdm.business.region.sdk.service.RegionVoService;
import com.biz.crm.mdm.business.region.sdk.vo.RegionVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Maps;

import java.util.*;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * region模块sdk实现
 *
 * @author sunx
 * @date 2021/10/9
 */
@Service("regionVoService")
public class RegionVoServiceImpl implements RegionVoService {

  @Autowired(required = false)
  private RegionRepository regionRepository;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  /**
   * 获取regionCode对应的regionName的map
   *
   * <p>key->regionCode,value->regionName
   *
   * @param regionCodeList
   * @return
   */
  @Override
  public Map<String, String> findRegionCodeNameMap(List<String> regionCodeList) {
    List<Region> list = regionRepository.findByRegionCodes(regionCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    return list.stream()
        .collect(Collectors.toMap(Region::getRegionCode, Region::getRegionName, (a, b) -> a));
  }

  @Override
  public String findRegionCode(String regionName, String parentCode) {
    Region region = this.regionRepository.findByParentCodeAndRegionName(parentCode, regionName);
    if (Objects.isNull(region)) {
      return null;
    }
    return region.getRegionCode();
  }

  @Override
  public List<RegionVo> findByRegionLevel(String regionLevel, String regionName) {
    //默认查省一级数据
    if (StringUtils.isEmpty(regionLevel)) {
      regionLevel = RegionLevelEnum.ONE.getDictCode();
    }
    //直辖市没有市一级数据，目前暂时带出省数据
    List<Region> result = regionRepository.findByRegionLevel(regionLevel, regionName);
    if (RegionLevelEnum.TWO.getDictCode().equals(regionLevel)) {
      result.addAll(regionRepository.findByRegionLevel(RegionLevelEnum.ONE.getDictCode(), regionName));
    }
    return (List<RegionVo>) nebulaToolkitService.copyCollectionByWhiteList(result, Region.class, RegionVo.class, LinkedHashSet.class, ArrayList.class);
  }

  @Override
  public RegionVo findByRegionCode(String regionCode) {
    Validate.notNull(regionCode, "行政区域编码不能为空！");
    Region region = this.regionRepository.findByRegionCode(regionCode);
    if (ObjectUtils.isNotEmpty(region)) {
      RegionVo regionVo =
          this.nebulaToolkitService.copyObjectByWhiteList(
              region, RegionVo.class, HashSet.class, ArrayList.class);
      return regionVo;
    }
    return null;
  }

  @Override
  public List<RegionVo> findByParentCode(String parentCode) {
    Validate.notNull(parentCode, "父级行政编码不能为空！");
    List<Region> regions = this.regionRepository.findByParentCode(parentCode);
    List<RegionVo> regionVos =
        (List<RegionVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                regions, Region.class, RegionVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isEmpty(regionVos)) {
      return null;
    }
    return regionVos;
  }
  /**
   * 提供给销售区域的行政区域组织树
   *
   * @param regionCode 行政区域编码
   * @param items 过滤条件
   * @return
   */
  @Override
  public List<TreeVo> findRegionTreeByRegionCode(String regionCode, List<String> items) {
    Validate.notNull(regionCode, "行政区域编码不能为空!");
    Validate.notNull(items, "关联销售区域的行政区域编码集合不能为空!");
    List<TreeVo> children = this.findChildrenByRegionCode(regionCode, items);
    if (CollectionUtils.isEmpty(children)) {
      return null;
    }
    return children;
  }

  /**
   * 递归查询一级行政区域编码对应得所有children 行政区域数据量过大，行政区域最下层层级为 ‘5’ 遂只递归到第5级
   * 项目可根据自身的需求，修改RegionConstant.LEVEL_NUM的数据
   *
   * @param regionCode 行政区域编码
   * @param items 过滤条件
   * @return
   */
  private List<TreeVo> findChildrenByRegionCode(String regionCode, List<String> items) {
    // 根据父级的行政编码查询其所有下级
    List<Region> regionList = this.regionRepository.findByParentCode(regionCode);
    // 过滤出满足条件的当前行政区域数据
    List<Region> regions =
        regionList.stream()
            .filter(r -> items.contains(r.getRegionCode()))
            .collect(Collectors.toList());
    List<TreeVo> collect =
        regions.stream()
            .map(
                x -> {
                  TreeVo treeVo = new TreeVo();
                  treeVo.setId(x.getId());
                  treeVo.setParentCode(x.getParentCode());
                  treeVo.setCode(x.getRegionCode());
                  treeVo.setName(x.getRegionName());
                  // 如果当前层级不是最下层，则继续递归过滤，获取其满足items的行政区域数据
                  if (!x.getLevelNum().equals(RegionConstant.LEVEL_NUM)) {
                    List<TreeVo> children = this.findChildrenByRegionCode(x.getRegionCode(), items);
                    // 将过滤后的结果作为当前行政区域的子集
                    treeVo.setChildren(children);
                  }
                  return treeVo;
                })
            .collect(Collectors.toList());
    // 将过滤后的数据构建成树形返回
    List<TreeVo> treeVos = TreeUtil.generateTreeByParentCode(collect);
    return treeVos;
  }
}
