package com.biz.crm.mdm.business.region.local.service.internal;

import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.entity.RegionLabel;
import com.biz.crm.mdm.business.region.local.repository.RegionLabelRepository;
import com.biz.crm.mdm.business.region.local.service.RegionLabelService;
import com.biz.crm.mdm.business.region.local.service.RegionService;
import com.biz.crm.mdm.business.region.sdk.dto.RegionLabelDto;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

/**
 * @author HanJiaJun
 * @describe:
 * @createTime 2022年08月02日 14:45:00
 */
@Service
public class RegionLabelServiceImpl implements RegionLabelService {


  @Autowired
  private RegionService regionService;
  @Autowired
  private RegionLabelRepository regionLabelRepository;

  /**
   * 绑定标签
   *
   * @param dto
   */
  @Override
  public void create(RegionLabelDto dto) {
    Validate.notNull(dto, "绑定标签时对象不能为空");
    Validate.notEmpty(dto.getLabelList(), "绑定标签时标签数据不能为空");
    Validate.notEmpty(dto.getRegionCodeList(), "绑定标签时行政区域编码数据不能为空");
    List<Region> detailByCodes = this.regionService.findDetailByCodes(dto.getRegionCodeList());
    Validate.notEmpty(detailByCodes, "未维护当前行政区域编码的数据");
    //获取全部的规则编码
    List<String> ruleCodes = detailByCodes.stream().filter(e -> StringUtils.isNotBlank(e.getRuleCode())).map(Region::getRuleCode).collect(Collectors.toList());
    //通过规则编码查询当前区域以及所属下级
    List<Region> regions = new ArrayList<>();
    for (String ruleCode : ruleCodes) {
      List<Region> selfAndLowerByRuleCode = this.regionService.findSelfAndLowerByRuleCode(ruleCode);
      regions.addAll(selfAndLowerByRuleCode);
    }
    //获取全部的行政区域编码
    List<String> regionCodes = regions.stream().map(Region::getRegionCode).collect(Collectors.toList());
    //清除已有标签
    this.deleteHandle(regionCodes);
    //添加标签
    List<RegionLabel> res = new ArrayList<>();
    List<String> labelList = dto.getLabelList();
    for (Region region : regions) {
      for (String label : labelList) {
        RegionLabel regionLabel = new RegionLabel();
        regionLabel.setRegionCode(region.getRegionCode());
        regionLabel.setRuleCode(region.getRuleCode());
        regionLabel.setLabel(label);
        res.add(regionLabel);
      }
    }
    this.regionLabelRepository.saveBatch(res);
  }

  /**
   * 执行删除标签操作，参数为最终行政编码
   *
   * @param regionCodes 最终行政编码，即不需要再寻找下级的编码
   */
  private void deleteHandle(List<String> regionCodes) {
    Validate.notEmpty(regionCodes, "删除标签时，编码不能为空");
    this.regionLabelRepository.deleteHandle(regionCodes);
  }

  /**
   * 删除标签
   *
   * @param regionCodes
   */
  @Override
  public void delete(List<String> regionCodes) {
    Validate.notEmpty(regionCodes, "删除标签时，编码不能为空");
    List<Region> detailByCodes = this.regionService.findDetailByCodes(regionCodes);
    Validate.notEmpty(detailByCodes, "未维护当前行政区域编码的数据");
    //获取全部的规则编码
    List<String> ruleCodes = detailByCodes.stream().filter(e -> StringUtils.isNotBlank(e.getRuleCode())).map(Region::getRuleCode).collect(Collectors.toList());
    //通过规则编码查询当前区域以及所属下级
    List<Region> regions = new ArrayList<>();
    for (String ruleCode : ruleCodes) {
      List<Region> selfAndLowerByRuleCode = this.regionService.findSelfAndLowerByRuleCode(ruleCode);
      regions.addAll(selfAndLowerByRuleCode);
    }
    //获取全部的行政区域编码
    List<String> allRegionCodes = regions.stream().map(Region::getRegionCode).collect(Collectors.toList());
    //清除标签
    this.deleteHandle(allRegionCodes);
  }

  /**
   * 通过编码查询标签
   * @param regionCodes
   * @return
   */
  @Override
  public List<RegionLabel> findByRegionCodes(List<String> regionCodes) {
    if (CollectionUtils.isEmpty(regionCodes)){
      return new ArrayList<>(0);
    }
   return this.regionLabelRepository.findByRegionCodes(regionCodes);
  }

  @Override
  public List<String> findRuleCodeByLabelList(List<String> labelList) {
    if (CollectionUtils.isEmpty(labelList)){
      return new ArrayList<>(0);
    }
    List<RegionLabel> labels= this.regionLabelRepository.findByLabelList(labelList);
    if (CollectionUtils.isEmpty(labels)){
      return new ArrayList<>(0);
    }
    Map<String, List<RegionLabel>> map = labels.stream().collect(Collectors.groupingBy(RegionLabel::getRuleCode));
    List<String> res =new ArrayList<>();
    Set labelSet = new TreeSet(labelList);
    map.forEach((k,v)->{
      //对比数据库中的标签和条件中的标签
      List<String> dbLabel  = v.stream().map(RegionLabel::getLabel).collect(Collectors.toList());
      Set dbSet = new TreeSet(dbLabel);
      if (dbSet.equals(labelSet)){
        res.add(k);
      }
    });
    return res;
  }
}
