package com.biz.crm.mdm.business.region.local.service.internal;

import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.repository.RegionRepository;
import com.biz.crm.mdm.business.region.sdk.constant.RegionConstant;
import com.biz.crm.mdm.business.region.sdk.dto.TreeDto;
import com.biz.crm.mdm.business.region.sdk.event.RegionSaleTerritoryEventListener;
import com.biz.crm.mdm.business.region.sdk.service.RegionLazyTreeVoService;
import com.biz.crm.mdm.business.region.sdk.utils.TreeUtil;
import com.biz.crm.mdm.business.region.sdk.vo.LazyTreeVo;
import com.biz.crm.mdm.business.region.sdk.vo.RegionSaleTerritoryVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author hejun
 * @title RegionLazyTreeVoServiceImpl
 * @date 2023/2/24 14:50
 * @description 行政区域懒加载树vo接口实现类
 */
@Service("RegionLazyTreeVoServiceImpl")
public class RegionLazyTreeVoServiceImpl implements RegionLazyTreeVoService {

  @Autowired(required = false)
  private RegionRepository regionRepository;

  @Autowired(required = false)
  private RegionSaleTerritoryEventListener regionSaleTerritoryEventListener;

  /** 基于数据库执行的数据视图执行内容缓存（最多500毫秒） */
  private static volatile Cache<String, List<LazyTreeVo>> cache = null;

  public RegionLazyTreeVoServiceImpl() {
    if (cache == null) {
      synchronized (RegionLazyTreeVoServiceImpl.class) {
        while (cache == null) {
          cache =
              CacheBuilder.newBuilder()
                  .initialCapacity(10000)
                  .expireAfterWrite(500, TimeUnit.MILLISECONDS)
                  .maximumSize(100000)
                  .build();
        }
      }
    }
  }

  @Override
  public List<LazyTreeVo> findLazyTree(TreeDto treeDto) {
    treeDto = Optional.ofNullable(treeDto).orElse(new TreeDto());
    List<LazyTreeVo> list = new ArrayList<>();
    // 设置排除ruleCode
    String excludeRuleCode = "";
    if (StringUtils.isNotEmpty(treeDto.getExcludeCodeAndChildren())) {
      // 如果排除编码不为空，说明存在查询时需要排除的数据及其所有子集，则设置需要排除的ruleCode，通过ruleCode向下排除
      Region one =
          this.regionRepository.findDetailsByCode(
              treeDto.getExcludeCodeAndChildren(), TenantUtils.getTenantCode());
      if (one != null) {
        excludeRuleCode = one.getRuleCode();
      }
    }
    if (StringUtils.isNotEmpty(treeDto.getParentCode())) {
      // 如果父级编码不为空，说明本次只查询该编码的下一级数据构建成树形结构
      List<LazyTreeVo> result =
          this.regionRepository.findLazyTreeList(
              treeDto.getEnableStatus(),
              null,
              treeDto.getParentCode(),
              null,
              null,
              excludeRuleCode,
              TenantUtils.getTenantCode());
      boolean disable = treeDto.getDisable();
      // true 代表父级被禁用 懒加载时 下级状态为全部禁用
      // false代表父级未被禁用 只过滤直接绑定了销售区域的行政区域
      if (CollectionUtils.isNotEmpty(result)) {
        if (disable != true) {
          List<LazyTreeVo> regionTree = this.setStatus(result);
          if (CollectionUtils.isNotEmpty(regionTree)) {
            list.addAll(regionTree);
          }
        } else {
          result.forEach(r -> r.setDisabled(true));
          list.addAll(result);
        }
      }
    } else {
      // 否则，说明本次查询是一级行政区域数据(排除国家级别,parentCode is null)
      // 根据一级的parentCode作为降维懒加载树条件  查询一级行政区域的数据
      // 先从缓存中获取
      List<LazyTreeVo> regionCache = cache.getIfPresent(RegionConstant.REGION_KEY);
      if (CollectionUtils.isNotEmpty(regionCache)) {
        // 执行一级行政区域过滤和修改禁用状态的业务
        List<LazyTreeVo> regionTree = this.setStatus(regionCache);
        if (CollectionUtils.isNotEmpty(regionTree)) {
          list.addAll(regionTree);
        }
      } else {
        // 缓存中没有,则查询数据库 将结果放在缓存中
        List<LazyTreeVo> result =
            this.regionRepository.findLazyTreeList(
                treeDto.getEnableStatus(),
                false,
                RegionConstant.EXCLUDE_REGION_CODE,
                null,
                null,
                excludeRuleCode,
                TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(result)) {
          return null;
        }
        // 将编码结果存放在缓存中
        cache.put(RegionConstant.REGION_KEY, result);
        // 过滤已经关联销售区域的行政区域顶层数据 设置禁用状态
        List<LazyTreeVo> regionTree = this.setStatus(result);
        if (CollectionUtils.isNotEmpty(regionTree)) {
          list.addAll(regionTree);
        }
      }
    }
    if (!CollectionUtils.isEmpty(list)) {
      for (LazyTreeVo lazyTreeVo : list) {
        lazyTreeVo.setHasChild(
            null != lazyTreeVo.getHasChildFlag() && lazyTreeVo.getHasChildFlag().compareTo(0) > 0);
      }
      return TreeUtil.generateLazyTreeByParentCode(list);
    }
    return new ArrayList<>();
  }

  @Override
  public List<LazyTreeVo> findRegionTree() {
    List<LazyTreeVo> list = this.regionRepository.findListByLevelNum(Lists.newArrayList(1,2,3,4));
    List<LazyTreeVo> resultList = list.stream().filter(o -> o.getLevelNum() == 1).collect(Collectors.toList());
    Map<String, List<LazyTreeVo>> parentMap = list.stream().filter(o->StringUtils.isNotEmpty(o.getParentCode())).collect(Collectors.groupingBy(LazyTreeVo::getParentCode));
    list.forEach(lazyTreeVo->{
      buildChildren(lazyTreeVo,parentMap);
    });
    return resultList;
  }

  private  void  buildChildren(LazyTreeVo lazyTreeVo,Map<String, List<LazyTreeVo>> parentMap){
    List<LazyTreeVo> lazyTreeVos = parentMap.get(lazyTreeVo.getCode());
    if(CollectionUtils.isNotEmpty(lazyTreeVos)){
      lazyTreeVo.setChildren(lazyTreeVos);
      lazyTreeVo.setHasChildFlag(1);
      for (LazyTreeVo treeVo : lazyTreeVos) {
        buildChildren(treeVo,parentMap);
      }
    }else{
      lazyTreeVo.setHasChildFlag(0);
      return;
    }
  }

  /**
   * 将设置禁用状态的方法抽出
   *
   * @param result
   * @return
   */
  private List<LazyTreeVo> setStatus(List<LazyTreeVo> result) {
    List<LazyTreeVo> list = new ArrayList<>();
    List<String> regionCodes = new ArrayList<>();
    if (!CollectionUtils.isEmpty(result)) {
      result.forEach(r -> regionCodes.add(r.getCode()));
    }
    // 筛选出直接绑定了销售区域的行政区域
    List<RegionSaleTerritoryVo> saleRegionVos =
        this.regionSaleTerritoryEventListener.findSaleTerritoryCodeByRegionCodes(regionCodes);
    if (CollectionUtils.isEmpty(saleRegionVos)) {
      // 未找到已经关联了行政区域的销售区域数据
      // 默认 禁用标志为false 不禁用
      result.forEach(r -> r.setDisabled(false));
      list.addAll(result);
    } else {
      // 将已经关联销售区域的行政区域数据过滤，开启禁用标记
      Set<String> collect =
          saleRegionVos.stream()
              .map(RegionSaleTerritoryVo::getRegionCode)
              .collect(Collectors.toSet());
      result.stream()
          .forEach(
              r -> {
                if (collect.contains(r.getCode())) {
                  r.setDisabled(true);
                } else {
                  r.setDisabled(false);
                }
              });
      if (CollectionUtils.isNotEmpty(result)) {
        list.addAll(result);
      }
    }
    return list;
  }
}
