package com.biz.crm.mdm.business.sales.org.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mdm.business.sales.org.local.entity.SalesOrg;
import com.biz.crm.mdm.business.sales.org.local.service.SalesOrgMdgService;
import com.biz.crm.mdm.business.sales.org.local.service.SalesOrgService;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgPaginationDto;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgSelectDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import com.google.common.collect.Lists;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 销售组织信息 http 接口
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.13 20:01
 */
@Api(value = "MDM-销售组织", tags = "MDM-销售组织")
@Slf4j
@RestController
@RequestMapping(value = {"/v1/salesOrg/salesOrg"})
public class SalesOrgController {

    @Autowired(required = false)
    private SalesOrgService salesOrgService;

    @Autowired(required = false)
    private SalesOrgMdgService salesOrgMdgService;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @ApiOperation(value = "查询分页列表")
    @GetMapping(value = {"/findByConditions"})
    public Result<Page<SalesOrg>> findByConditions(@PageableDefault(50) Pageable pageable, SalesOrgPaginationDto paginationDto) {
        try {
            Page<SalesOrg> result = this.salesOrgService.findByConditions(pageable, paginationDto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 销售组织信息下拉框分页列表
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     * @return 零售商列表
     */
    @ApiOperation(value = "销售组织下拉框分页列表")
    @GetMapping("/findBySalesSelectDto")
    public Result<Page<SalesOrg>> findBySalesSelectDto(@PageableDefault(50) Pageable pageable,
                                                       @ApiParam(name = "dto", value = "下拉分页Dto") SalesOrgSelectDto dto) {
        try {
            if(StringUtils.isNotBlank(dto.getSelectedCode())){
                dto.setSelectedCodes(Lists.newArrayList(dto.getSelectedCode()));
            }
            return Result.ok(salesOrgService.findBySalesSelectDto(pageable, dto));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    @ApiOperation(value = "通过id/salesOrgCode 查询详情")
    @GetMapping(value = {"/findDetailsByIdOrSalesOrgCode"})
    public Result<SalesOrg> findDetailsByIdOrSalesOrgCode(String id, String salesOrgCode) {
        try {
            SalesOrg org = null;
            if (StringUtils.isNotBlank(id)) {
                org = this.salesOrgService.findDetailsById(id);
            } else if (StringUtils.isNotBlank(salesOrgCode)) {
                org = this.salesOrgService.findByOrgCode(salesOrgCode);
            }
            return Result.ok(org);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "按ids查询详情")
    @GetMapping(value = {"/findDetailsByIds"})
    public Result<List<SalesOrg>> findDetailsByIds(@RequestParam("ids") List<String> ids) {
        try {
            List<SalesOrg> result = this.salesOrgService.findDetailsByIds(ids);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据销售组织编码或销售组织名称查询
     *
     * @param
     * @return
     */
    @ApiOperation(value = "根据销售组织编码或销售组织名称查询")
    @GetMapping("/findBySalesOrgCodeLikeOrSalesOrgNameLike")
    public Result<List<SalesOrg>> findBySalesOrgCodeLikeOrSalesOrgNameLike(@RequestParam(value = "salesOrgCodeLikeOrNameLike") String salesOrgCodeLikeOrNameLike) {
        try {
            return Result.ok(salesOrgService.findBySalesOrgCodeLikeOrSalesOrgNameLike(salesOrgCodeLikeOrNameLike));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "逻辑删除", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @DeleteMapping("/delete")
    public Result<?> delete(@RequestParam("ids") List<String> ids) {
        try {
            this.salesOrgService.deleteByIds(ids);
            return Result.ok("删除成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "启用", httpMethod = "PATCH")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PatchMapping("/enable")
    public Result<?> enable(@RequestBody List<String> ids) {
        try {
            this.salesOrgService.enableBatch(ids);
            return Result.ok("启用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "禁用", httpMethod = "PATCH")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PatchMapping("/disable")
    public Result<?> disable(@RequestBody List<String> ids) {
        try {
            this.salesOrgService.disableBatch(ids);
            return Result.ok("禁用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 相关的创建过程，http接口。请注意该创建过程除了可以创建org中的基本信息以外，还可以对org中属于OneToMany关联的明细信息一同进行创建注意：基于（org）模型的创建操作传入的orgJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
     */
    @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建org中的基本信息以外，还可以对org中属于OneToMany关联的明细信息一同进行创建注意：基于（org）模型的创建操作传入的orgJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
    @PostMapping(value = "")
    public Result<SalesOrg> create(
            @RequestBody @ApiParam(name = "org", value = "相关的创建过程，http接口。请注意该创建过程除了可以创建org中的基本信息以外，还可以对org中属于OneToMany关联的明细信息一同进行创建注意：基于（org）模型的创建操作传入的orgJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") SalesOrg org) {
        try {
            SalesOrg current = this.salesOrgService.create(org);
            return Result.ok(current);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（org）的修改操作传入的orgJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
     */
    @ApiOperation(value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（org）的修改操作传入的orgJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
    @PatchMapping(value = "")
    public Result<SalesOrg> update(
            @RequestBody @ApiParam(name = "org", value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：基于模型（org）的修改操作传入的orgJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") SalesOrg org) {
        try {
            SalesOrg current = this.salesOrgService.update(org);
            return Result.ok(current);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("根据 销售组织id/销售组织编码 查询全部下级（含当前）销售组织列表")
    @GetMapping("/findAllChildrenByIdOrSalesOrgCode")
    public Result<List<SalesOrg>> findAllChildrenByIdOrSalesOrgCode(String id, String salesOrgCode) {
        try {
            List<SalesOrg> list = null;
            if (StringUtils.isNotBlank(id)) {
                list = this.salesOrgService.findAllChildrenById(id);
            } else if (StringUtils.isNotBlank(salesOrgCode)) {
                list = this.salesOrgService.findAllChildrenByOrgCode(salesOrgCode);
            }
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    @ApiOperation("根据 销售组织id/销售组织编码 查询当前下级销售组织列表")
    @GetMapping("/findChildrenByIdOrSalesOrgCode")
    public Result<List<SalesOrg>> findChildrenByIdOrSalesOrgCode(String id, String salesOrgCode) {
        try {
            List<SalesOrg> list = null;
            if (StringUtils.isNotBlank(id)) {
                list = this.salesOrgService.findChildrenById(id);
            } else if (StringUtils.isNotBlank(salesOrgCode)) {
                list = this.salesOrgService.findChildrenByOrgCode(salesOrgCode);
            }
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    @ApiOperation("根据 销售组织id或/编码查询 全部上级（含当前）销售组织列表")
    @GetMapping("/findAllParentByIdOrSalesOrgCode")
    public Result<List<SalesOrg>> findAllParentByIdOrSalesOrgCode(String id, String salesOrgCode) {
        try {
            List<SalesOrg> list = null;
            if (StringUtils.isNotBlank(id)) {
                list = this.salesOrgService.findAllParentById(id);
            } else if (StringUtils.isNotBlank(salesOrgCode)) {
                list = this.salesOrgService.findAllParentByOrgCode(salesOrgCode);
            }
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("根据销售组织id/销售组织salesOrgCode 查询当前销售组织的直接上级销售组织")
    @GetMapping("/findParentByIdOrSalesOrgCode")
    public Result<SalesOrg> findParentByIdOrSalesOrgCode(String id, String salesOrgCode) {
        try {
            SalesOrg parent = null;
            if (StringUtils.isNotBlank(id)) {
                parent = this.salesOrgService.findParentById(id);
            } else if (StringUtils.isNotBlank(salesOrgCode)) {
                parent = this.salesOrgService.findParentBySalesOrgCode(salesOrgCode);
            }
            return Result.ok(parent);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("查询销售组织及下级销售组织，只查询指定条数")
    @GetMapping("/findAllChildrenBySalesOrgCodes")
    public Result<List<SalesOrg>> findAllChildrenBySalesOrgCodes(@PageableDefault(50) Pageable pageable, @RequestParam("salesOrgCodes") List<String> salesOrgCodes) {
        try {
            List<SalesOrg> list = this.salesOrgService.findAllChildrenBySalesOrgCodes(pageable, salesOrgCodes);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("查询所有销售组织（不分页，数据量大，给后台用）(启用状态、销售组织类型、上级编码精确查询，销售组织编码、名称模糊查询)")
    @GetMapping("/findListByConditions")
    public Result<List<SalesOrg>> findListByConditions(SalesOrgPaginationDto dao) {
        try {
            List<SalesOrg> list = this.salesOrgService.findByConditions(dao);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("查询所有销售组织")
    @GetMapping("/findAll")
    public Result<List<SalesOrg>> findAll() {
        try {
            SalesOrgPaginationDto dao = new SalesOrgPaginationDto();
            dao.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            List<SalesOrg> list = this.salesOrgService.findByConditions(dao);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("重置降维编码")
    @PatchMapping("/updateRuleCode")
    public Result<?> updateRuleCode() {
        try {
            if (redisService.hasKey(salesOrgService.getUpdateRuleLockKey())) {
                return Result.error("上次重置降维尚未完成,请耐心等待!");
            }
            UserIdentity userIdentity = loginUserService.getLoginUser();
            this.salesOrgService.updateRuleCode(userIdentity);
            return Result.ok("重置降维中,请耐心等待。");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("根据销售组织编码集合查询这些销售组织（不含自己）的上级销售组织编码，如果参数本身有上下级关系，则会返回处于上级的销售组织编码")
    @GetMapping("/findAllParentSalesOrgCodeExcludeSelf")
    public Result<List<String>> findAllParentSalesOrgCodeExcludeSelf(@RequestParam("salesOrgCodes") List<String> salesOrgCodes) {
        try {
            List<SalesOrg> list = this.salesOrgService.findAllParentSalesOrgCodeExcludeSelf(salesOrgCodes);
            List<String> parentSalesOrgCodes = null;
            if (CollectionUtils.isNotEmpty(list)) {
                parentSalesOrgCodes = list.stream().map(SalesOrg::getSalesOrgCode).collect(Collectors.toList());
            }
            return Result.ok(parentSalesOrgCodes);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("查询销售机构")
    @GetMapping("/findSalesMechanism")
    public Result<Page<SalesOrg>> findSalesMechanism(@PageableDefault(50) Pageable pageable, SalesOrgPaginationDto paginationDto) {
        try {
            Page<SalesOrg> result = this.salesOrgService.findSalesMechanism(pageable, paginationDto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("查询销售组")
    @GetMapping("/findSalesGroup")
    public Result<Page<SalesOrg>> findSalesGroup(@PageableDefault(50) Pageable pageable, SalesOrgPaginationDto paginationDto) {
        try {
            Page<SalesOrg> result = this.salesOrgService.findSalesGroup(pageable, paginationDto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("查询销售组织")
    @GetMapping("/findSalesOrg")
    public Result<Page<SalesOrg>> findSalesOrg(@PageableDefault(50) Pageable pageable, SalesOrgPaginationDto paginationDto) {
        try {
            Page<SalesOrg> result = this.salesOrgService.findSalesOrg(pageable, paginationDto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

//
//    /**
//     * 批量拉取MDG销售组织上级数据
//     *
//     * @param dto 请求参数
//     * @return 销售组织上级数据
//     */
//    @ApiOperation(value = "批量拉取数据总台-MDG销售组织上级数据")
//    @PostMapping(value = {"/pullSalesOrgParentList"})
//    public Result<?> pullSalesOrgParentList(@RequestBody MasterDataMdgBaseDto dto) {
//        try {
//            this.salesOrgMdgService.pullSalesOrgParentList(dto, true);
//            return Result.ok("批量拉取数据总台-MDG销售组织上级数据成功");
//        } catch (RuntimeException e) {
//            log.error(e.getMessage(), e);
//            return Result.error(e.getMessage());
//        }
//    }
//
//    /**
//     * 批量拉取 MDG 销售组织主数据
//     *
//     * @param dto 请求参数
//     * @return 销售组织主数据
//     */
//    @ApiOperation(value = "批量拉取数据总台-MDG销售组织数据")
//    @PostMapping(value = {"/pullSalesOrgList"})
//    public Result<?> pullSalesOrgList(@RequestBody MasterDataMdgBaseDto dto) {
//        try {
//            this.salesOrgMdgService.pullSalesOrgList(dto, true);
//            return Result.ok("批量拉取数据总台-MDG销售组织数据成功");
//        } catch (RuntimeException e) {
//            log.error(e.getMessage(), e);
//            return Result.error(e.getMessage());
//        }
//    }
}
