package com.biz.crm.mdm.business.sales.org.local.controller;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.sales.org.sdk.dto.RelateSalesOrgCodeQueryDto;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgQueryDto;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 销售组织信息
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.13 20:09
 */
@Api(value = "MDM-销售组织", tags = "MDM-销售组织VO")
@Slf4j
@RestController
@RequestMapping(value = {"/v1/salesOrg/salesOrg"})
public class SalesOrgVoController {

    @Autowired(required = false)
    private SalesOrgVoService orgVoService;

    /**
     * 按salesOrgCode集合查询
     *
     * @param salesOrgCode
     * @return
     */
    @GetMapping(value = {"/findBySalesOrgCode"})
    public Result<SalesOrgVo> findBySalesOrgCode(@RequestParam("salesOrgCode") String salesOrgCode) {
        try {
            return Result.ok(this.orgVoService.findBySalesOrgCode(salesOrgCode));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 按salesOrgCode集合查询
     *
     * @param salesOrgCodeList
     * @return
     */
    @GetMapping(value = {"/findBySalesOrgCodes"})
    public Result<List<SalesOrgVo>> findBySalesOrgCodes(@RequestParam("salesOrgCodeList") List<String> salesOrgCodeList) {
        try {
            return Result.ok(this.orgVoService.findBySalesOrgCodes(salesOrgCodeList));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据 销售组织编码查询 全部上级（含当前）销售组织列表
     *
     * @param salesOrgCode
     * @return
     */
    @ApiOperation("根据 销售组织编码查询 全部上级（含当前）销售组织列表")
    @GetMapping("/findAllParentBySalesOrgCode")
    public Result<List<SalesOrgVo>> findAllParentBySalesOrgCode(@RequestParam("salesOrgCode") String salesOrgCode) {
        try {
            List<SalesOrgVo> list = this.orgVoService.findAllParentBySalesOrgCode(salesOrgCode);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据 销售组织编码集合查询 全部上级（含当前）销售组织列表
     *
     * @param salesOrgCodes
     * @return
     */
    @ApiOperation("根据 销售组织编码集合查询 全部上级（含当前）销售组织列表")
    @GetMapping("/findAllParentBySalesOrgCodes")
    public Result<List<SalesOrgVo>> findAllParentBySalesOrgCodes(@RequestParam("salesOrgCodes") List<String> salesOrgCodes) {
        try {
            List<SalesOrgVo> list = this.orgVoService.findAllParentBySalesOrgCodes(salesOrgCodes);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 查询指定salesOrgCode下的所有子节点
     *
     * @param salesOrgCode
     * @return
     */
    @ApiOperation("根据 销售组织编码集合查询 全部上级（含当前）销售组织列表")
    @GetMapping("/findAllChildrenBySalesOrgCode")
    public Result<List<SalesOrgVo>> findAllChildrenBySalesOrgCode(@RequestParam("salesOrgCode") String salesOrgCode) {
        try {
            List<SalesOrgVo> list = this.orgVoService.findAllChildrenBySalesOrgCode(salesOrgCode);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询指定salesOrgCode下的所有子节点
     *
     * @param salesOrgCodes
     * @return
     */
    @ApiOperation("根据 销售组织编码集合查询 全部上级（含当前）销售组织列表")
    @GetMapping("/findAllChildrenVoBySalesOrgCodes")
    public Result<List<SalesOrgVo>> findAllChildrenBySalesOrgCodes(@RequestParam("salesOrgCodes") List<String> salesOrgCodes) {
        try {
            List<SalesOrgVo> list = this.orgVoService.findAllChildrenBySalesOrgCodes(salesOrgCodes);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 获取销售组织编码
     *
     * @param dto
     * @return
     */
    @ApiOperation("获取销售组织编码")
    @GetMapping("/findBySalesOrgQueryDto")
    public Result<Set<String>> findBySalesOrgQueryDto(SalesOrgQueryDto dto) {
        try {
            Set<String> set = this.orgVoService.findBySalesOrgQueryDto(dto);
            return Result.ok(set);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据获取销售组织对应的未删除的上级或下级编码
     *
     * @param dto
     * @return
     */
    @ApiOperation("根据获取销售组织对应的未删除的上级或下级编码")
    @GetMapping("/findByRelateSalesOrgCodeQueryDto")
    public Result<Map<String, String>> findByRelateSalesOrgCodeQueryDto(RelateSalesOrgCodeQueryDto dto) {
        try {
            return Result.ok(this.orgVoService.findByRelateSalesOrgCodeQueryDto(dto));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询指定salesOrgLevel 下的所有销售组织编码和名称
     *
     * @param salesOrgLevel
     * @return
     */
    @ApiOperation("根据 销售组织层级 查询销售组织编码和名称 ")
    @GetMapping(value = {"/findAllSalesOrgLevelBySalesOrg"})
    public Result<List<SalesOrgVo>> findAllSalesOrgLevelBySalesOrg(@RequestParam("salesOrgLevel") String salesOrgLevel,
                                                                   @RequestParam(value = "salesOrgName", required = false) String salesOrgName) {
        try {
            List<SalesOrgVo> list = this.orgVoService.findAllSalesOrgLevelBySalesOrg(salesOrgLevel,salesOrgName);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询销售组织erp编码下全部销售组织
     *
     * @param salesOrgErpCodeList
     * @return
     */
    @ApiOperation("根据 销售组织层级 查询销售组织编码和名称 ")
    @PostMapping(value = {"/findByErpCodeList"})
    public Result<List<SalesOrgVo>> findByErpCodeList(@RequestBody List<String> salesOrgErpCodeList) {
        try {
            List<SalesOrgVo> list = this.orgVoService.findByErpCodeList(salesOrgErpCodeList);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据erp编码查询销售组织全部信息
     *
     * @param erpCodes erp编码列表
     * @return Map<String, SalesOrgVo>
     */
    @ApiOperation("根据erp编码查询销售组织全部信息")
    @PostMapping(value = {"/getSalesMapByErpCodes"})
    public Result<Map<String, SalesOrgVo>> getSalesMapByErpCodes(@RequestBody List<List<String>> erpCodes) {
        try {
            Map<String, SalesOrgVo> map = this.orgVoService.getSalesMapByErpCodes(erpCodes);
            return Result.ok(map);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据tpm编码查询销售组织全部信息
     *
     * @param tpmCodes tpm编码列表
     * @return Map<String, SalesOrgVo>
     */
    @ApiOperation("根据tpm编码查询销售组织全部信息")
    @PostMapping(value = {"/getSalesMapByTpmCodes"})
    public Result<Map<String, SalesOrgVo>> getSalesMapByTpmCodes(@RequestBody List<String> tpmCodes) {
        try {
            Map<String, SalesOrgVo> map = this.orgVoService.getSalesMapByTpmCodes(tpmCodes);
            return Result.ok(map);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据销售组织编码集合查询对应sap编码
     *
     * @param salesOrgCodes 销售组织编码集合
     * @return Map<String, String>
     */
    @ApiOperation("根据销售组织编码集合查询对应sap编码")
    @PostMapping(value = {"/findSapCodesBySalesOrgCodes"})
    public Result<Map<String, String>> findSapCodesBySalesOrgCodes(@RequestBody List<String> salesOrgCodes) {
        try {
            Map<String, String> map = this.orgVoService.findSapCodesBySalesOrgCodes(salesOrgCodes);
            return Result.ok(map);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据销售组织编码集合查询对应名称
     *
     * @param salesOrgCodes 销售组织编码集合
     * @return Map<String, String>
     */
    @ApiOperation("根据销售组织编码集合查询对应名称")
    @PostMapping(value = {"/findNameBySalesOrgCodes"})
    public Result<Map<String, String>> findNameBySalesOrgCodes(@RequestBody List<List<String>> salesOrgCodes) {
        try {
            Map<String, String> map = this.orgVoService.findNameBySalesOrgCodes(salesOrgCodes);
            return Result.ok(map);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据销售组织编码集合查询
     * @param salesOrgCodeList 销售组织编码列表
     * @author jiangyin
     * @date 2023/3/7 19:49
     * @return com.biz.crm.business.common.sdk.model.Result<java.util.List<com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo>>
     */
    @ApiOperation("根据销售组织编码集合查询")
    @PostMapping(value = {"/findBySalesOrgCodesPost"})
    public Result<List<SalesOrgVo>> findBySalesOrgCodesPost(@RequestBody List<String> salesOrgCodeList) {
        try {
            return Result.ok(this.orgVoService.findBySalesOrgCodes(salesOrgCodeList));
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
