package com.biz.crm.mdm.business.sales.org.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.vo.LazyTreeVo;
import com.biz.crm.mdm.business.sales.org.local.entity.SalesOrg;
import com.biz.crm.mdm.business.sales.org.local.mapper.SalesOrgMapper;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgPaginationDto;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgQueryDto;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgSelectDto;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Repository;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;
import com.biz.crm.mdm.business.sales.org.sdk.enums.SalesOrgLevelTypeEnum;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import jodd.util.StringUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Repository;

import java.util.*;
import java.util.stream.Collectors;

import org.springframework.util.Assert;

/**
 * 条件字段分类 数据层
 *
 * @describe: 简述
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.10.28 16:18
 */
@Repository
public class SalesOrgRepository extends ServiceImpl<SalesOrgMapper, SalesOrg> {

    private static final Integer SIZE = 1000;
    private static final Integer DEFAULT_SIZE = 25;

    /**
     * 根据 SalesOrg 主键查询详细信息（包括关联表）
     *
     * @param id
     * @return
     */
    public SalesOrg findDetailsById(String id) {
        if (StringUtil.isEmpty(id)) {
            return null;
        }
        return this.baseMapper.findDetailsById(id, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 根据 orgCode 条件字段分类编码 查询（包括关联表）
     *
     * @param orgCode
     * @param tenantCode
     * @return
     */
    public SalesOrg findByOrgCode(String orgCode, String tenantCode) {
        if (StringUtil.isEmpty(orgCode)) {
            return null;
        }
        return this.baseMapper.findByOrgCode(orgCode, tenantCode, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 分页查询
     *
     * @param pageable
     * @param paginationDto
     * @return
     */
    public Page<SalesOrg> findByConditions(Pageable pageable, SalesOrgPaginationDto paginationDto) {
        Page<SalesOrg> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.baseMapper.findByConditions(page, paginationDto, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 根据 SalesOrg 主键集合查询详细信息（包括关联表）
     *
     * @param ids
     * @return
     */
    public List<SalesOrg> findDetailsByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.emptyList();
        }
        return this.baseMapper.findDetailsByIds(ids, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 根据客户编码或客户名称查询
     *
     * @param
     * @return
     */
    public List<SalesOrg> findByOrgCodeLikeOrOrgNameLike(String codeOrNameLike) {
        if (StringUtils.isEmpty(codeOrNameLike)) {
            return Collections.emptyList();
        }
        Page<SalesOrg> page = new Page<>(0, DEFAULT_SIZE);


        Page<SalesOrg> salesOrgPage = this.baseMapper.findByOrgCodeLikeOrOrgNameLike(page, codeOrNameLike,TenantUtils.getTenantCode());

        return salesOrgPage.getRecords();
    }

    /**
     * 根据 orgCode集合查询详细信息（包括关联表）
     *
     * @param orgCodes
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findByOrgCodes(List<String> orgCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return Collections.emptyList();
        }
        return this.baseMapper.findByOrgCodes(orgCodes, tenantCode, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 按父节点code集合查询
     *
     * @param parentCodes
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findByParentCodes(List<String> parentCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(parentCodes)) {
            return Collections.emptyList();
        }
        return this.baseMapper.findByParentCodes(parentCodes, tenantCode, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 按ruleCode 集合查询所有子集合的销售组织code
     *
     * @param ruleCodes
     * @param tenantCode
     * @return
     */
    public List<String> findChildrenOrgCodeByRuleCodes(List<String> ruleCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return Collections.emptyList();
        }
        return this.baseMapper.findChildrenOrgCodeByRuleCodes(ruleCodes, tenantCode, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 按id集合查询销售组织ruleCode
     *
     * @param ids
     * @return
     */
    public List<String> findRuleCodeByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.emptyList();
        }
        List<SalesOrg> orgs = this.lambdaQuery().eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(CollectionUtils.isNotEmpty(ids), SalesOrg::getId, ids).select(SalesOrg::getRuleCode).list();
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }
        return orgs.stream().map(SalesOrg::getRuleCode).collect(Collectors.toList());
    }

    /**
     * 按orgCode查询销售组织ruleCode
     *
     * @param orgCode
     * @param tenantCode
     * @return
     */
    public String findRuleCodeByOrgCode(String orgCode, String tenantCode) {
        if (StringUtil.isEmpty(orgCode)) {
            return "";
        }
        SalesOrg org = this.lambdaQuery().eq(SalesOrg::getSalesOrgCode, orgCode).eq(SalesOrg::getTenantCode, tenantCode)
                .eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).select(SalesOrg::getRuleCode).one();
        if (Objects.isNull(org)) {
            return null;
        }
        return org.getRuleCode();
    }

    /**
     * 按orgCodes集合查询销售组织ruleCode
     *
     * @param orgCodes
     * @return
     */
    public List<String> findRuleCodeByOrgCodes(List<String> orgCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return Collections.emptyList();
        }
        List<String> list = Lists.newLinkedList();
        for (List<String> item : Lists.partition(orgCodes, SIZE)) {
            List<SalesOrg> cur = this.lambdaQuery()
                    .in(SalesOrg::getSalesOrgCode, item)
                    .eq(SalesOrg::getTenantCode, tenantCode)
                    .eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                    .select(SalesOrg::getRuleCode).list();
            if (CollectionUtils.isEmpty(cur)) {
                continue;
            }
            list.addAll(cur.stream().map(SalesOrg::getRuleCode).collect(Collectors.toSet()));
        }
        return list;
    }

    /**
     * 根据orgCode集合 更新销售组织启用/禁用状态
     *
     * @param orgCodes
     * @param enable
     */
    public void updateEnableStatusByOrgCodes(List<String> orgCodes, EnableStatusEnum enable, String tenantCode) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return;
        }
        UpdateWrapper<SalesOrg> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("enable_status", enable.getCode());
        updateWrapper.in("org_code", orgCodes);
        updateWrapper.eq("tenant_code", tenantCode);
        this.update(updateWrapper);
    }

    /**
     * 根据id集合 更新销售组织启用/禁用状态
     *
     * @param ids
     * @param enable
     */
    public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        UpdateWrapper<SalesOrg> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("enable_status", enable.getCode());
        updateWrapper.in("id", ids);
        this.update(updateWrapper);
    }

    /**
     * 通过父节点code查询
     *
     * @param parentCode
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findByParentCode(String parentCode, String tenantCode) {
        if (StringUtil.isEmpty(parentCode)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery().eq(SalesOrg::getParentCode, parentCode).eq(SalesOrg::getTenantCode, tenantCode)
                .eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
    }

    /**
     * 查询父节点是null的节点（包含已删除已禁用）
     *
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findByParentCodeIsNull(String tenantCode) {
        return this.lambdaQuery()
                .eq(SalesOrg::getParentCode, StringUtils.EMPTY)
                .eq(SalesOrg::getTenantCode, tenantCode)
                .list();
    }

    /**
     * 通过销售组织编码更新ruleCode和levelNum
     *
     * @param orgCode
     * @param ruleCode
     * @param levelNum
     */
    public void updateRuleCodeAndLevelNumByOrgCode(String orgCode, String ruleCode, int levelNum, String tenantCode) {
        if (StringUtil.isEmpty(orgCode)) {
            return;
        }
        this.lambdaUpdate()
                .set(SalesOrg::getRuleCode, ruleCode)
                .set(SalesOrg::getLevelNum, levelNum)
                .eq(SalesOrg::getSalesOrgCode, orgCode)
                .eq(SalesOrg::getTenantCode, tenantCode)
                .update();
    }

    /**
     * 按降维码集合，和状态查询
     *
     * @param ruleCodes
     * @param enable
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findByRuleCodesAndEnableStatus(List<String> ruleCodes, EnableStatusEnum enable,
                                                         String tenantCode) {
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .eq(Objects.nonNull(enable), SalesOrg::getEnableStatus, enable != null ? enable.getCode() : null)
                .in(SalesOrg::getRuleCode, ruleCodes).eq(SalesOrg::getTenantCode, tenantCode)
                .eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
    }

    /**
     * 按ruleCode查询所有子节点
     *
     * @param ruleCode
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findChildrenByRuleCode(String ruleCode, String tenantCode) {
        if (StringUtil.isEmpty(ruleCode)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .likeRight(SalesOrg::getRuleCode, ruleCode)
                .eq(SalesOrg::getTenantCode, tenantCode)
                .eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 按ruleCodes查询所有子节点
     *
     * @param ruleCodes
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findChildrenByRuleCode(List<String> ruleCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return Collections.emptyList();
        }
        return this.baseMapper.findChildrenByRuleCodes(ruleCodes, tenantCode, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 分页通过ruleCodes 查询子节点信息
     *
     * @param pageable
     * @param ruleCodes
     * @param tenantCode
     * @return
     */
    public Page<SalesOrg> findChildrenByRuleCodes(Pageable pageable, List<String> ruleCodes, String tenantCode) {
        Page<SalesOrg> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.baseMapper.findChildrenByRuleCodes(page, ruleCodes, tenantCode, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 按条件查询 启用状态、销售组织类型、上级编码精确查询，销售组织编码、名称模糊查询
     *
     * @param dto
     * @return
     */
    public List<SalesOrg> findByConditions(SalesOrgPaginationDto dto) {
        if (dto.getPageNumber() < 1) {
            dto.setPageNumber(1);
        }
        if (dto.getPageSize() < 1) {
            dto.setPageSize(DEFAULT_SIZE);
        }
        Page<SalesOrg> page = new Page<>(dto.getPageNumber(), dto.getPageSize());
        Page<SalesOrg> orgPage = this.baseMapper.findByConditions(page, dto, DelFlagStatusEnum.NORMAL);
        return orgPage.getRecords();
    }

    /**
     * 根据 【EnableStatus】 orgName模糊 查询orgCode ruleCode
     *
     * @param enableStatus
     * @param name
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findOrgCodeAndRuleCodeByEnableStatusOptAndOrgNameLike(String enableStatus, String name,
                                                                                String tenantCode) {
        if (StringUtil.isEmpty(name)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(enableStatus), SalesOrg::getEnableStatus, enableStatus)
                .like(SalesOrg::getSalesOrgName, name)
                .eq(SalesOrg::getTenantCode, tenantCode)
                .select(SalesOrg::getSalesOrgCode, SalesOrg::getRuleCode)
                .eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 查询企业销售组织懒加载数据
     *
     * @param enableStatus    启用状态
     * @param topOnly         传true只查第一层
     * @param parentCode      只查询该编码下一级
     * @param codeList        只查询这些编码
     * @param ruleCodeList    只查询这些降维编码
     * @param excludeRuleCode 排除这个降维编码的下级
     * @param tenantCode
     * @return
     */
    public List<LazyTreeVo> findOrgLazyTreeList(String enableStatus, Boolean topOnly, String parentCode,
                                                List<String> codeList, List<String> ruleCodeList, String excludeRuleCode, String tenantCode) {
        return this.baseMapper.findOrgLazyTreeList(enableStatus, topOnly, parentCode, codeList, ruleCodeList,
                excludeRuleCode, tenantCode, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 查找parentCode不为空但找不到对应上级的数据,设置parentCode为null
     *
     * @return
     */
    public void updateOrphanParentCodeNull(String tenantCode) {
        if (StringUtil.isEmpty(tenantCode)) {
            return;
        }
        this.baseMapper.updateOrphanParentCodeNull(tenantCode, DelFlagStatusEnum.NORMAL);
    }

    /**
     * 查询所有没有父级编码的子集
     *
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findListWithoutParentCode(String tenantCode) {
        if (StringUtil.isEmpty(tenantCode)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .eq(SalesOrg::getTenantCode, tenantCode)
                // .eq(SalesOrg::getParentCode, StringUtils.EMPTY)
                .isNull(SalesOrg::getParentCode).eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).list();
    }

    /**
     * 根据销售组织编码查询详情
     *
     * @param orgCode
     * @param tenantCode
     * @return
     */
    public SalesOrg findDetailsByCode(String orgCode, String tenantCode) {
        if (StringUtil.isEmpty(orgCode)) {
            return null;
        }
        return this.lambdaQuery().eq(SalesOrg::getTenantCode, tenantCode).eq(SalesOrg::getSalesOrgCode, orgCode)
                .eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode()).one();
    }

    /**
     * 通过父级编码查询所有子级
     *
     * @param parentCode
     * @param tenantCode
     * @return
     */
    public List<SalesOrg> findChildrenListByParentCode(String parentCode, String tenantCode) {
        parentCode = Optional.ofNullable(parentCode).orElse(StringUtils.EMPTY);
        return this.lambdaQuery().eq(SalesOrg::getTenantCode, tenantCode).eq(SalesOrg::getParentCode, parentCode)
                .list();
    }

    /**
     * 手动设置父级编码为空
     *
     * @param id
     */
    public void setParentCodeNull(String id) {
        if (StringUtil.isEmpty(id)) {
            return;
        }
        this.lambdaUpdate().set(SalesOrg::getParentCode, null).eq(SalesOrg::getId, id).update();
    }

    /**
     * 手动设置父级编码，规则编码为空
     *
     * @param id
     */
    public void setParentCodeAndRuleCodeNull(String id) {
        if (StringUtil.isEmpty(id)) {
            return;
        }
        this.lambdaUpdate().set(SalesOrg::getParentCode, null).set(SalesOrg::getRuleCode, "").eq(SalesOrg::getId, id)
                .update();
    }

    /**
     * 通过编码查找销售组织（包括逻辑删除）
     *
     * @param org
     */
    public SalesOrg findAllByOrgCode(SalesOrg org) {
        if (Objects.isNull(org)
                || StringUtil.isEmpty(org.getSalesOrgCode())) {
            return null;
        }
        return this.lambdaQuery().eq(SalesOrg::getSalesOrgCode, org.getSalesOrgCode()).one();
    }

    /**
     * 获取匹配的销售组织编码集合
     *
     * @param dto
     * @return
     */
    public Set<String> findByOrgQueryDto(SalesOrgQueryDto dto) {
        if (Objects.isNull(dto)
                || StringUtil.isEmpty(dto.getSalesOrgName())) {
            return Sets.newHashSet();
        }
        final List<SalesOrg> list = this.lambdaQuery()
                .eq(SalesOrg::getTenantCode, TenantUtils.getTenantCode())
                .eq(SalesOrg::getDelFlag, dto.getDelFlag())
                .like(StringUtils.isNotBlank(dto.getSalesOrgName()), SalesOrg::getSalesOrgName, dto.getSalesOrgName())
                .select(SalesOrg::getSalesOrgCode).list();
        if (CollectionUtils.isEmpty(list)) {
            return Sets.newHashSet();
        }
        return list.stream().filter(a -> StringUtils.isNotBlank(a.getSalesOrgCode())).map(SalesOrg::getSalesOrgCode)
                .collect(Collectors.toSet());
    }

    public Page<SalesOrg> findBySalesSelectDto(Page<SalesOrg> page, SalesOrgSelectDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findBySalesSelectDto(page, dto);

    }

    /**
     * MDG主数据拉取根据销售组织编码获取对应数据
     */
    public List<SalesOrg> findBySalesOrgCodeAllLists(List<String> salesOrgCodeList) {
        if (CollectionUtils.isEmpty(salesOrgCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery().eq(SalesOrg::getTenantCode, TenantUtils.getTenantCode())
                .in(SalesOrg::getSalesOrgCode, salesOrgCodeList).list();
    }

    /**
     * 根据销售组织编码获取有效的销售组织
     */
    public List<SalesOrg> findBySalesOrgCodeLists(List<String> salesOrgCodeList) {
        if (CollectionUtils.isEmpty(salesOrgCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery().eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(SalesOrg::getEnableStatus, DelFlagStatusEnum.NORMAL.getCode())
                .eq(SalesOrg::getTenantCode, TenantUtils.getTenantCode()).in(SalesOrg::getSalesOrgCode, salesOrgCodeList)
                .list();
    }

    /**
     * 查询指定salesOrgLevel 下的所有销售组织编码和名称(MDG)
     *
     * @param salesOrgLevel
     * @param salesOrgName
     * @author 杜涛涛
     * @date 2022/12/5 20:32
     **/
    public List<SalesOrg> findAllSalesOrgLevelBySalesOrg(String salesOrgLevel,String salesOrgName) {
        if (StringUtil.isEmpty(salesOrgLevel)) {
            return Collections.emptyList();
        }
        if (!SalesOrgLevelTypeEnum.MECHANISM.getCode().equals(salesOrgLevel)
                && !SalesOrgLevelTypeEnum.DEPARTMENT.getCode().equals(salesOrgLevel)
                && StringUtil.isEmpty(salesOrgName)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery().eq(SalesOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(SalesOrg::getTenantCode, TenantUtils.getTenantCode())
                .eq(SalesOrg::getSalesOrgLevel, salesOrgLevel)
                .like(StringUtil.isEmpty(salesOrgName), SalesOrg::getSalesOrgName, salesOrgName)
                .list();
    }

    /**
     * 根据所属销售机构编码查询所属销售机构名称
     *
     * @param salesAgencyCode
     * @return
     */
    public List<SalesOrg> findBySalesAgencyCode(String salesAgencyCode) {
        if (StringUtil.isEmpty(salesAgencyCode)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery().in(SalesOrg::getErpCode, salesAgencyCode)
                .eq(SalesOrg::getTenantCode, TenantUtils.getTenantCode()).list();
    }

    public List<SalesOrg> findByErpCodeList(List<String> salesOrgErpCodeList) {
        if (CollectionUtils.isEmpty(salesOrgErpCodeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery().in(SalesOrg::getErpCode, salesOrgErpCodeList)
                .eq(SalesOrg::getTenantCode, TenantUtils.getTenantCode()).list();
    }


    public List<SalesOrg> findByMdgSalesOrgCode(String source2Code) {
        if (StringUtil.isEmpty(source2Code)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(SalesOrg::getErpCode, source2Code)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据erp编码查询销售组织全部信息
     *
     * @param erpCodes   erp编码列表
     * @param tenantCode 租户编码
     * @return List<SalesOrgVo>
     */
    public List<SalesOrgVo> getSalesMapByErpCodes(List<List<String>> erpCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(erpCodes)) {
            return Collections.emptyList();
        }
        return this.baseMapper.getSalesMapByErpCodes(erpCodes, tenantCode);
    }

    /**
     * 根据tpm编码查询销售组织全部信息
     *
     * @param tpmCodes   tpm编码列表
     * @param tenantCode 租户编码
     * @return List<SalesOrgVo>
     */
    public List<SalesOrgVo> getSalesMapByTpmCodes(List<List<String>> tpmCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(tpmCodes)) {
            return Collections.emptyList();
        }
        return this.baseMapper.getSalesMapByTpmCodes(tpmCodes, tenantCode);
    }

    /**
     * 更据销售组织编码查询详情集合
     *
     * @param salesOrgCodes 销售组织编码集合
     * @param tenantCode    租户编码
     * @return List<SalesOrg>
     */
    public List<SalesOrg> findSapCodesByOrgCodes(List<String> salesOrgCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(salesOrgCodes)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(SalesOrg::getSalesOrgCode, salesOrgCodes)
                .eq(TenantFlagOpEntity::getTenantCode, tenantCode)
                .list();
    }

    /**
     * 根据销售组织编码集合查询对应名称
     *
     * @param salesOrgCodes 销售组织编码
     * @param tenantCode    租户编码
     * @return List<SalesOrgVo>
     */
    public List<SalesOrgVo> findNameBySalesOrgCodes(List<List<String>> salesOrgCodes, String tenantCode) {
        if (CollectionUtils.isEmpty(salesOrgCodes)) {
            return Collections.emptyList();
        }
        return this.baseMapper.findNameBySalesOrgCodes(salesOrgCodes, tenantCode);
    }


    /**
     * 根据ERP编码获取所有的销售组织
     *
     * @param erpCodeList
     * @param department
     * @return
     */
    public List<SalesOrg> findAllByErpCodeList(List<String> erpCodeList, SalesOrgLevelTypeEnum department) {
        if (CollectionUtils.isEmpty(erpCodeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(SalesOrg::getErpCode, erpCodeList)
                .eq(Objects.nonNull(department), SalesOrg::getSalesOrgLevel, department.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * erp编码查询名称
     *
     * @param erpCodes
     * @return {@link Map}<{@link String}, {@link String}>
     */
    public Map<String, String> findNameByErpCodes(List<String> erpCodes) {
        Assert.notEmpty(erpCodes, "参数erp编码不能为空");
        String tenantCode = TenantUtils.getTenantCode();
        Map<String, String> resultMap = Maps.newHashMap();
        List<SalesOrgVo> salesOrgVos = this.baseMapper.findNameByErpCodes(erpCodes, tenantCode);
        if (org.springframework.util.CollectionUtils.isEmpty(salesOrgVos)) {
            return resultMap;
        }
        salesOrgVos.forEach(o -> {
            resultMap.put(o.getErpCode(), o.getSalesOrgName());
        });
        return resultMap;
    }

    /**
     * 根据编码或者名称查询
     * @param codeOrName
     * @return {@link List}<{@link SalesOrgVo}>
     */
    public List<SalesOrgVo> findByCodeOrName(List<String> codeOrName) {
        Assert.notEmpty(codeOrName, "编码名称查询条件不能为空");
        SalesOrgSelectDto dto = new SalesOrgSelectDto();
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setCodeOrNameList(codeOrName);
        return this.baseMapper.findBySalesOrgSelectDto(dto);
    }

    /**
     * 根据类型查询销售机构(这里是通过erpCode分组了的)
     * @param pageable
     * @param dto
     * @return {@link Page}<{@link SalesOrg}>
     */
    public Page<SalesOrg> findBySalesOrgLevelType(Pageable pageable, SalesOrgPaginationDto dto) {
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<SalesOrg> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.baseMapper.findBySalesOrgLevelType(page, dto);
    }
}
