package com.biz.crm.mdm.business.sales.org.local.service;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.sales.org.local.entity.SalesOrg;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgPaginationDto;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgSelectDto;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import org.springframework.data.domain.Pageable;

import java.util.List;
import java.util.Map;

/**
 * 销售组织接口
 *
 * @describe: 简述
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.10.28 16:07
 */
public interface SalesOrgService {

    /**
     * 分页条件查询
     *
     * @param pageable
     * @param paginationDto
     * @return
     */
    Page<SalesOrg> findByConditions(Pageable pageable, SalesOrgPaginationDto paginationDto);

    /**
     * 按id查询详情
     *
     * @param id
     * @return
     */
    SalesOrg findDetailsById(String id);

    /**
     * 按销售组织code查询详情
     *
     * @param salesOrgCode
     * @return
     */
    SalesOrg findByOrgCode(String salesOrgCode);

    /**
     * 更据销售组织编码查询对应sap编码
     *
     * @param salesOrgCodes
     * @return
     */
    Map<String,String> findSapCodesByOrgCodes(List<String> salesOrgCodes);

    /**
     * 按id集合查询详情
     *
     * @param ids
     * @return
     */
    List<SalesOrg> findDetailsByIds(List<String> ids);

    /**
     * 根据销售组织编码或销售组织名称查询
     *
     * @param salesOrgCodeLikeOrNameLike
     * @return
     */
    List<SalesOrg> findBySalesOrgCodeLikeOrSalesOrgNameLike(String salesOrgCodeLikeOrNameLike);

    /**
     * 按orgCode集合查询详情
     *
     * @param salesOrgCodes
     * @return
     */
    List<SalesOrg> findByOrgCodes(List<String> salesOrgCodes);

    /**
     * 按id集合删除
     *
     * @param ids
     */
    void deleteByIds(List<String> ids);

    /**
     * 按id集合启用
     *
     * @param ids
     */
    void enableBatch(List<String> ids);

    /**
     * 按id集合禁用
     *
     * @param ids
     */
    void disableBatch(List<String> ids);

    /**
     * 创建销售组织
     *
     * @param salesOrg
     * @return
     */
    SalesOrg create(SalesOrg salesOrg);

    /**
     * 更新销售组织
     *
     * @param salesOrg
     * @return
     */
    SalesOrg update(SalesOrg salesOrg);

    /**
     * 查询id下的所有子节点
     *
     * @param id
     * @return
     */
    List<SalesOrg> findAllChildrenById(String id);

    /**
     * 查询orgCode下的所有子节点
     *
     * @param salesOrgCode
     * @return
     */
    List<SalesOrg> findAllChildrenByOrgCode(String salesOrgCode);

    /**
     * 查询orgCodes下的所有子节点
     *
     * @param salesOrgCodes
     * @return
     */
    List<SalesOrg> findAllChildrenBySalesOrgCodes(List<String> salesOrgCodes);

    /**
     * 按id查询当前节点下的子节点，不包含子节点下面的节点
     *
     * @param id
     * @return
     */
    List<SalesOrg> findChildrenById(String id);

    /**
     * 按orgCode查询当前节点下的子节点，不包含子节点下面的节点
     *
     * @param salesOrgCode
     * @return
     */
    List<SalesOrg> findChildrenByOrgCode(String salesOrgCode);

    /**
     * 通过id查询所有上级
     *
     * @param id
     * @return
     */
    List<SalesOrg> findAllParentById(String id);

    /**
     * 通过orgCode查询所有上级
     *
     * @param salesOrgCode
     * @return
     */
    List<SalesOrg> findAllParentByOrgCode(String salesOrgCode);

    /**
     * 通过orgCodes查询所有上级
     *
     * @param salesOrgCodes
     * @return
     */
    List<SalesOrg> findAllParentByOrgCodes(List<String> salesOrgCodes);

    /**
     * 通过ruleCodes集合查询所有上级
     *
     * @param ruleCodes
     * @return
     */
    List<SalesOrg> findAllParentByRuleCodes(List<String> ruleCodes);

    /**
     * 根据销售组织编码集合查询这些销售组织（不含自己）的上级销售组织编码，如果参数本身有上下级关系，则会返回处于上级的销售组织编码
     *
     * @param salesOrgCodes
     * @return
     */
    List<SalesOrg> findAllParentSalesOrgCodeExcludeSelf(List<String> salesOrgCodes);

    /**
     * 通过id 查询当前销售组织的直接上级销售组织
     *
     * @param id
     * @return
     */
    SalesOrg findParentById(String id);

    /**
     * 通过orgCode 查询当前销售组织的直接上级销售组织
     *
     * @param salesOrgCode
     * @return
     */
    SalesOrg findParentBySalesOrgCode(String salesOrgCode);


    /**
     * 分页按orgCodes查询当前节点下的子节点，包含子节点下面的节点
     *
     * @param pageable
     * @param salesOrgCodes
     * @return
     */
    List<SalesOrg> findAllChildrenBySalesOrgCodes(Pageable pageable, List<String> salesOrgCodes);

    /**
     * 按条件查询 启用状态、销售组织类型、上级编码精确查询，销售组织编码、名称模糊查询
     *
     * @param dto
     * @return
     */
    List<SalesOrg> findByConditions(SalesOrgPaginationDto dto);

    /**
     * 按父节点code 查询
     *
     * @param salesOrgCode
     * @return
     */
    List<SalesOrg> findByParentCode(String salesOrgCode);

    /**
     * 获取redis锁定key
     * @return
     */
    String getUpdateRuleLockKey();

    /**
     * 重置降维编码
     *
     * @param userIdentity
     */
    void updateRuleCode(UserIdentity userIdentity);

    /**
     * 按ruleCodes查询所有子节点
     *
     * @param ruleCodes
     * @return
     */
    List<SalesOrg> findChildrenByRuleCode(List<String> ruleCodes);

    /**
     * 按降维码集合，和状态查询
     *
     * @param ruleCodes
     * @param enable
     * @return
     */
    List<SalesOrg> findByRuleCodesAndEnableStatus(List<String> ruleCodes, EnableStatusEnum enable);


    /**
     * 销售组织信息下拉框分页列表
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     * @return 零售商列表
     */
    Page<SalesOrg> findBySalesSelectDto(Pageable pageable, SalesOrgSelectDto dto);

    /**
     * 批量新增数据（组织渠道产品主数据MDG专用）
     *
     * @param saveList
     * @param updateList
     * @author dutaotao
     * @date 2023/1/3 17:38
     **/
    void saveOrUpdateBatchForMdg(List<SalesOrg> saveList, List<SalesOrg> updateList);

    /**
     * 查询销售机构
     * @param pageable
     * @param paginationDto
     * @return {@link Page}<{@link SalesOrg}>
     */
    Page<SalesOrg> findSalesMechanism(Pageable pageable, SalesOrgPaginationDto paginationDto);

    /**
     * 查询销售机构
     * @param pageable
     * @param paginationDto
     * @return {@link Page}<{@link SalesOrg}>
     */
    Page<SalesOrg> findSalesGroup(Pageable pageable, SalesOrgPaginationDto paginationDto);

    /**
     * 查询销售组织
     * @param pageable
     * @param paginationDto
     * @return {@link Page}<{@link SalesOrg}>
     */
    Page<SalesOrg> findSalesOrg(Pageable pageable, SalesOrgPaginationDto paginationDto);
}
