package com.biz.crm.mdm.business.sales.org.local.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.sales.org.local.entity.SalesOrgBusiness;
import com.biz.crm.mdm.business.sales.org.local.repository.SalesOrgBusinessRepository;
import com.biz.crm.mdm.business.sales.org.local.service.SalesOrgBusinessVoService;
import com.biz.crm.mdm.business.sales.org.sdk.enums.SalesOrgBusinessTypeEnum;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgBusinessVo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.16 16:18
 */
@Service
@Slf4j
public class SalesOrgBusinessVoServiceImpl implements SalesOrgBusinessVoService {

    @Autowired(required = false)
    private SalesOrgBusinessRepository salesOrgBusinessRepository;
    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    /**
     * 根据销售组织ID或销售组织编码;获取业态或业务单元信息
     *
     * @param salesOrgId   销售组织ID
     * @param salesOrgCode 销售组织编码
     * @param typeEnum
     * @return java.util.List<com.biz.crm.mdm.business.org.sdk.vo.SalesOrgBusinessFormatVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.16 16:19
     */
    @Override
    public List<SalesOrgBusinessVo> findBusinessByIdOrCode(String salesOrgId, String salesOrgCode, SalesOrgBusinessTypeEnum typeEnum) {
        if (Objects.isNull(typeEnum)) {
            return Collections.emptyList();
        }
        if (StringUtils.isEmpty(salesOrgId)
                && StringUtils.isEmpty(salesOrgCode)) {
            return Collections.emptyList();
        }
        List<SalesOrgBusinessVo> resultList = salesOrgBusinessRepository.findBusinessByIdOrCode(salesOrgId, salesOrgCode, typeEnum.getCode());
        return this.setName(resultList, typeEnum);
    }

    /**
     * 根据销售组织ID集合或销售组织编码集合;获取业态或业务单元信息
     *
     * @param salesOrgIdList   销售组织ID集合
     * @param salesOrgCodeList 销售组织编码集合
     * @param typeEnum
     * @return java.util.List<com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgBusinessVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.16 17:06
     */
    @Override
    public List<SalesOrgBusinessVo> findBusinessByIdsOrCodes(List<String> salesOrgIdList, List<String> salesOrgCodeList, SalesOrgBusinessTypeEnum typeEnum) {
        if (Objects.isNull(typeEnum)) {
            return Collections.emptyList();
        }
        if (CollectionUtil.isEmpty(salesOrgIdList)
                && CollectionUtil.isEmpty(salesOrgCodeList)) {
            return Collections.emptyList();
        }
        List<SalesOrgBusinessVo> resultList = salesOrgBusinessRepository.findBusinessByIdsOrCodes(salesOrgIdList, salesOrgCodeList, typeEnum.getCode());
        return this.setName(resultList, typeEnum);
    }

    /**
     * 根据销售组织编码和业务类型获取关联信息
     *
     * @param salesOrgCodeList 销售组织编码
     * @param businessTypeEnum 业务类型
     * @return java.util.List<com.biz.crm.mdm.business.sales.org.local.entity.SalesOrgBusiness>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023.1.9 14:34
     */
    @Override
    public List<SalesOrgBusiness> findSalesOrgBusinessByCodeList(List<String> salesOrgCodeList, SalesOrgBusinessTypeEnum businessTypeEnum) {
        if (CollectionUtils.isEmpty(salesOrgCodeList)
                || Objects.isNull(businessTypeEnum)) {
            return Lists.newArrayList();
        }
        return salesOrgBusinessRepository.findSalesOrgBusinessByCodeList(salesOrgCodeList, businessTypeEnum);
    }

    /**
     * 批量保存和更新
     *
     * @param saveList
     * @param updateList
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023.1.9 14:38
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveOrUpdateBatch(List<SalesOrgBusiness> saveList, List<SalesOrgBusiness> updateList) {
        if (!CollectionUtils.isEmpty(saveList)) {
            log.info("销售组织业态业务单元渠道关系新增[{}]条", saveList.size());
            this.salesOrgBusinessRepository.saveBatch(saveList);
        }

        if (!CollectionUtils.isEmpty(updateList)) {
            log.info("销售组织业态业务单元渠道关系更新[{}]条", updateList.size());
            this.salesOrgBusinessRepository.updateBatchById(updateList);
        }
    }

    /**
     * 设置名称
     *
     * @param resultList 数据
     * @param typeEnum   类型
     * @return
     */
    private List<SalesOrgBusinessVo> setName(List<SalesOrgBusinessVo> resultList, SalesOrgBusinessTypeEnum typeEnum) {
        if (CollectionUtil.isEmpty(resultList)
                || Objects.isNull(typeEnum)) {
            return Collections.emptyList();
        }

        Map<String, String> map = null;
        switch (typeEnum) {
            case UNIT:
                map = dictToolkitService.findMapByDictTypeCode("mdm_business_unit");
                break;
            case FORMAT:
                map = dictToolkitService.findMapByDictTypeCode("mdm_business_format");
                break;
            default:
                break;
        }
        if (CollectionUtil.isEmpty(map)) {
            return resultList;
        }
        Map<String, String> finalMap = map;
        resultList.stream().filter(k -> StringUtils.isNotBlank(k.getBusinessCode()))
                .forEach(vo -> vo.setBusinessName(finalMap.getOrDefault(vo.getBusinessCode(), "")));
        return resultList;
    }
}
