package com.biz.crm.mdm.business.sales.org.local.service.impl;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.dictionary.sdk.constant.DictConstant;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.sales.org.local.entity.SalesOrg;
import com.biz.crm.mdm.business.sales.org.local.repository.SalesOrgRepository;
import com.biz.crm.mdm.business.sales.org.local.service.SalesOrgMdgService;
import com.biz.crm.mdm.business.sales.org.local.service.SalesOrgService;
import com.biz.crm.mdm.business.sales.org.sdk.constant.SalesOrgCodeConstant;
import com.biz.crm.mdm.business.sales.org.sdk.enums.SalesOrgLevelTypeEnum;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.constants.MasterDataMdgConstants;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgOrganizeChannelProductVo;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgSalesOfficeVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @describe: 销售组织接口 实现
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.10.28 16:10
 */
@Service
@Slf4j
public class SalesOrgMdgServiceImpl implements SalesOrgMdgService {


    @Autowired(required = false)
    private SalesOrgRepository salesOrgRepository;
    @Autowired(required = false)
    private SalesOrgService salesOrgService;
    @Autowired(required = false)
    private RedisMutexService redisMutexService;
    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;
    @Autowired(required = false)
    private DictToolkitService dictToolkitService;
    @Autowired(required = false)
    private LoginUserService loginUserService;

    /**
     * 默认顶级销售组织编码
     */
    private final static String DEF_SALES_ORG_ONE_CODE = "MN2023";
    /**
     * 默认顶级销售组织名称
     */
    private final static String DEF_SALES_ORG_ONE_NAME = "自定义销售机构";
    /**
     * 电商销售机构不更新
     */
    private final static String NOT_UPDATE_SALES_CODE = "1171";


    /**
     * 批量拉取组织渠道产品主数据 MDG
     *
     * @param dto
     * @param isUpdateRuleCode
     * @author huojia
     * @date 2022/12/6 22:04
     **/
    @Override
    public void pullSalesOrgParentList(MasterDataMdgBaseDto dto, boolean isUpdateRuleCode) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }

        log.info("=====>    销售组织一二级更新 start    <=====");
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        boolean lock = this.lock(lockKey);
        Assert.isTrue(lock, "上次同步销售组织数据尚未完成,请耐心等待!");
        try {
            int thisPage = Integer.parseInt(dto.getPageNum());
            int maxPage;
            if (BooleanEnum.TRUE.getCapital().equals(dto.getFullPullFlag())) {
                maxPage = this.salesOrgParentCountPageMax(dto);
            } else {
                maxPage = thisPage;
            }
            String tenantCode = TenantUtils.getTenantCode();
            Set<String> codeOneSet = new HashSet<>(2048);
            for (int pageNum = thisPage; pageNum <= maxPage; pageNum++) {
                try {
                    dto.setPageNum(Integer.toString(pageNum));
                    List<MasterDataMdgOrganizeChannelProductVo> mdgSalesOrgList = masterDataMdgService.pullOrganizeChannelProductList(dto);
                    log.info("当前拉取进度：" + pageNum + "/" + maxPage + "，每页" + dto.getPageSize() + "条数据");
                    if (CollectionUtils.isEmpty(mdgSalesOrgList)) {
                        return;
                    }
                    // 数据校验
                    List<SalesOrg> pullList = this.buildSalesOrgParentList(codeOneSet, mdgSalesOrgList, tenantCode);
                    // 区分销售组织更新、新增
                    List<SalesOrg> saveList = new ArrayList<>();
                    List<SalesOrg> updateList = new ArrayList<>();
                    this.filterSalesOrg(pullList, saveList, updateList);
                    salesOrgService.saveOrUpdateBatchForMdg(saveList, updateList);
                } catch (Exception e) {
                    log.error("", e);
                }
            }
        } finally {
            if (lock) {
                if (isUpdateRuleCode) {
                    salesOrgService.updateRuleCode(loginUserService.getLoginUser());
                }
                this.unLock(lockKey);
            }
            log.info("=====>    销售组织一二级更新 ent    <=====");
        }
    }


    /**
     * 批量拉取销售办公室关系主数据 MDG
     *
     * @param dto
     * @param isUpdateRuleCode
     * @author dutaotao
     * @date 2022/12/25 15:04
     **/
    @Override
    public void pullSalesOrgList(MasterDataMdgBaseDto dto, boolean isUpdateRuleCode) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }
        log.info("=====>    销售组织三级更新 start    <=====");
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        boolean lock = this.lockSalesOffice(lockKey);
        Assert.isTrue(lock, "上次同步销售组织数据尚未完成,请耐心等待!");
        try {
            int maxPage = 0;
            if (BooleanEnum.TRUE.getCapital().equals(dto.getFullPullFlag())) {
                maxPage = this.salesOrgCountPageMax(dto);
            } else {
                maxPage = Integer.parseInt(dto.getPageNum());
            }
            SalesOrgLevelTypeEnum groupEnum = SalesOrgLevelTypeEnum.GROUP;
            Set<String> codeSet = new HashSet<>(2048);
            String tenantCode = TenantUtils.getTenantCode();
            Map<String, String> businessFormatMap = dictToolkitService.findMapByDictTypeCode(DictConstant.MDM_BUSINESS_FORMAT);
            for (int pageNum = Integer.parseInt(dto.getPageNum()); pageNum <= maxPage; pageNum++) {
                try {
                    dto.setPageNum(Integer.toString(pageNum));
                    log.info("当前拉取进度：" + pageNum + "/" + maxPage + "，每页" + dto.getPageSize() + "条数据");
                    List<MasterDataMdgSalesOfficeVo> mdgList = masterDataMdgService.pullSalesOfficeList(dto);
                    if (CollectionUtils.isEmpty(mdgList)) {
                        return;
                    }
                    // MDG数据校验
                    List<SalesOrg> pullList = this.buildSalesOrgList(codeSet, mdgList, groupEnum, tenantCode);
                    if (CollectionUtils.isEmpty(pullList)) {
                        continue;
                    }
                    // 拼接上级数据
                    pullList = this.buildSalesOrgInfoList(pullList, businessFormatMap, groupEnum, tenantCode);
                    if (CollectionUtils.isEmpty(pullList)) {
                        continue;
                    }
                    // 区分更新、新增
                    List<SalesOrg> saveList = new ArrayList<>();
                    List<SalesOrg> updateList = new ArrayList<>();
                    this.filterSalesOrg(pullList, saveList, updateList);
                    //验证上级是否存在
                    List<SalesOrg> saveNewList = new ArrayList<>();
                    List<SalesOrg> updateNewList = new ArrayList<>();
                    this.filterSalesOrgParent(saveList, updateList, saveNewList, updateNewList);
                    salesOrgService.saveOrUpdateBatchForMdg(saveNewList, updateNewList);
                } catch (Exception e) {
                    log.error("", e);
                }
            }
        } finally {
            if (lock) {
                if (isUpdateRuleCode) {
                    salesOrgService.updateRuleCode(loginUserService.getLoginUser());
                }
                this.unLockSalesOffice(lockKey);
            }
            log.info("=====>    销售组织三级更新 end    <=====");
        }
    }


    /**
     * 过滤销售组织 新增|更新
     *
     * @param list
     * @param saveList
     * @param updateList
     */
    private void filterSalesOrg(List<SalesOrg> list,
                                List<SalesOrg> saveList, List<SalesOrg> updateList) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        List<String> salesOrgCodeList = list.stream()
                .filter(k -> StringUtils.isNotBlank(k.getSalesOrgCode()))
                .map(SalesOrg::getSalesOrgCode)
                .distinct().collect(Collectors.toList());
        // 区分更新、新增操作
        List<SalesOrg> bySalesOrgCodes = salesOrgRepository.findBySalesOrgCodeAllLists(salesOrgCodeList);
        if (CollectionUtils.isEmpty(bySalesOrgCodes)) {
            saveList.addAll(list);
            return;
        }
        Map<String, SalesOrg> map = bySalesOrgCodes.stream()
                .collect(Collectors.toMap(SalesOrg::getSalesOrgCode, v -> v,
                        (oldValue, newValue) -> newValue));
        list.forEach(pull -> {
            SalesOrg oldEntity = map.get(pull.getSalesOrgCode());
            if (Objects.nonNull(oldEntity)) {
                pull.setId(oldEntity.getId());
                pull.setRuleCode(oldEntity.getRuleCode());
                pull.setBusinessUnitCode(oldEntity.getBusinessUnitCode());
                pull.setCreateAccount(oldEntity.getCreateAccount());
                pull.setCreateName(oldEntity.getCreateName());
                pull.setCreateTime(oldEntity.getCreateTime());
                updateList.add(pull);
            } else {
                saveList.add(pull);
            }
        });

    }

    /**
     * 验证数据合法性
     *
     * @param saveList
     * @param updateList
     */
    private void filterSalesOrgParent(List<SalesOrg> saveList, List<SalesOrg> updateList,
                                      List<SalesOrg> saveNewList, List<SalesOrg> updateNewList) {
        if (CollectionUtils.isEmpty(saveList)
                && CollectionUtils.isEmpty(updateList)) {
            return;
        }
        List<String> parentCodeList = saveList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getParentCode()))
                .map(SalesOrg::getParentCode)
                .distinct().collect(Collectors.toList());
        parentCodeList.addAll(updateList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getParentCode()))
                .map(SalesOrg::getParentCode)
                .distinct().collect(Collectors.toList()));
        // 验证他的上级是否存在
        List<SalesOrg> bySalesOrgCodes = salesOrgRepository.findBySalesOrgCodeAllLists(parentCodeList);
        if (CollectionUtils.isEmpty(bySalesOrgCodes)) {
            return;
        }
        Set<String> parentCodeSet = bySalesOrgCodes.stream()
                .filter(k -> StringUtils.isNotBlank(k.getSalesOrgCode()))
                .map(SalesOrg::getSalesOrgCode).collect(Collectors.toSet());
        saveNewList.addAll(saveList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getParentCode()))
                .filter(k -> parentCodeSet.contains(k.getParentCode()))
                .distinct().collect(Collectors.toList()));
        updateNewList.addAll(updateList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getParentCode()))
                .filter(k -> parentCodeSet.contains(k.getParentCode()))
                .distinct().collect(Collectors.toList()));
    }

    /**
     * 最大分页数
     *
     * @param dto
     * @return int
     * @author huojia
     * @date 2023/1/8 1:33
     **/
    private int salesOrgParentCountPageMax(MasterDataMdgBaseDto dto) {
        Integer totalNum = masterDataMdgService.countList(MasterDataMdgConstants.MDG_ORGANIZE_CHANNEL_PRODUCT_TPM);
        int maxPage = 1;
        int sizeInt = Integer.parseInt(dto.getPageSize());
        if (totalNum > sizeInt) {
            maxPage = totalNum % sizeInt > 0 ? (totalNum / sizeInt) + 1 : totalNum / sizeInt;
        }
        return maxPage;
    }


    /**
     * MDG组织渠道产品主数据按年月日解锁MDG
     *
     * @param yearMonthDay
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private void unLock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取组织渠道产品主数据解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(SalesOrgCodeConstant.SALES_ORG_PARENT_LOCK + yearMonthDay);
    }

    /**
     * 拉取销售办公室主数据 按年月日加锁 MDG
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private void unLockSalesOffice(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取组织渠道产品主数据解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(SalesOrgCodeConstant.SALES_ORG_LOCK + yearMonthDay);
    }

    /**
     * 数据校验 MDG组织渠道产品主数据
     *
     * @param list
     * @return java.util.List<com.biz.crm.mdm.business.sales.org.local.entity.salesOrg>
     * @author 杜涛涛
     **/
    private List<SalesOrg> buildSalesOrgParentList(
            Set<String> codeOneSet,
            List<MasterDataMdgOrganizeChannelProductVo> list,
            String tenantCode) {

        List<SalesOrg> pullList = new ArrayList<>();
        SalesOrgLevelTypeEnum oneEnum = SalesOrgLevelTypeEnum.MECHANISM;
        SalesOrgLevelTypeEnum twoEnum = SalesOrgLevelTypeEnum.DEPARTMENT;
        String enableStatus = EnableStatusEnum.ENABLE.getCode();
        String delFlag = DelFlagStatusEnum.NORMAL.getCode();
        list.forEach(mdgVo -> {
            String salesOrgOneErpCode = StringUtils.trimToEmpty(mdgVo.getSaleOrgCode());
            if (StringUtils.isEmpty(salesOrgOneErpCode)) {
                log.error("本次拉取数据[销售组织代码为空]：" + mdgVo);
                return;
            }
            //TODO 电商销售组织  临时不接收
            if (NOT_UPDATE_SALES_CODE.equals(salesOrgOneErpCode)) {
                log.info("=====>    电商销售组织[{}]不更新    <=====", salesOrgOneErpCode);
                return;
            }
            String channelCode = StringUtils.trimToEmpty(mdgVo.getDistributionChannelCode());
            if (StringUtils.isEmpty(channelCode)) {
                log.error("本次拉取数据[分销渠道代码为空]：" + mdgVo);
                return;
            }
            String businessFormatCode = StringUtils.trimToEmpty(mdgVo.getProductTeamCode());
            if (StringUtils.isEmpty(businessFormatCode)) {
                log.error("本次拉取数据[产品组代码为空]：" + mdgVo);
                return;
            }
            String salesOrgTwoErpCode = StringUtils.trimToEmpty(mdgVo.getSaleOfficeCode());
            if (StringUtils.isEmpty(salesOrgTwoErpCode)) {
                log.error("本次拉取数据[销售办公室代码为空]：" + mdgVo);
                return;
            }
            mdgVo.setDistributionChannelCode(StringUtils.trimToEmpty(channelCode));
            mdgVo.setDistributionChannelName(StringUtils.trimToEmpty(mdgVo.getDistributionChannelName()));
            mdgVo.setProductTeamCode(StringUtils.trimToEmpty(businessFormatCode));
            mdgVo.setProductTeamName(StringUtils.trimToEmpty(mdgVo.getProductTeamName()));
            mdgVo.setSaleOrgCode(StringUtils.trimToEmpty(salesOrgOneErpCode));
            mdgVo.setSaleOrgName(StringUtils.trimToEmpty(mdgVo.getSaleOrgName()));
            mdgVo.setSaleOfficeCode(StringUtils.trimToEmpty(salesOrgTwoErpCode));
            mdgVo.setSaleOfficeName(StringUtils.trimToEmpty(mdgVo.getSaleOfficeName()));
            if (salesOrgOneErpCode.equals(salesOrgTwoErpCode)) {
                mdgVo.setSaleOrgCode(DEF_SALES_ORG_ONE_CODE);
                mdgVo.setSaleOrgName(DEF_SALES_ORG_ONE_NAME);
                salesOrgOneErpCode = mdgVo.getSaleOrgCode();
            }
            String salesOrgOneCode = channelCode + businessFormatCode + salesOrgOneErpCode;
            String salesOrgTwoCode = salesOrgOneCode + salesOrgTwoErpCode;
            String salesOrgDesc = "";
            if (StringUtils.isNotBlank(mdgVo.getDistributionChannelName())) {
                salesOrgDesc = mdgVo.getDistributionChannelName();
            }
            if (StringUtils.isNotBlank(mdgVo.getProductTeamName())) {
                if (StringUtils.isNotBlank(salesOrgDesc)) {
                    salesOrgDesc = salesOrgDesc + "-";
                }
                salesOrgDesc = salesOrgDesc + mdgVo.getProductTeamName() + "-";
            }
            if (!codeOneSet.contains(salesOrgOneCode)) {
                codeOneSet.add(salesOrgOneCode);
                SalesOrg salesOrgOne = new SalesOrg();
                // 销售组织编码
                salesOrgOne.setSalesOrgCode(salesOrgOneCode);
                salesOrgOne.setErpCode(salesOrgOneErpCode);
                salesOrgOne.setSalesInstitutionCode(salesOrgOneCode);
                // 销售组织名称
                salesOrgOne.setSalesOrgName(mdgVo.getSaleOrgName());
                salesOrgOne.setSalesOrgDesc(salesOrgDesc + mdgVo.getSaleOrgName());
                salesOrgOne.setChannelCode(channelCode);
                salesOrgOne.setChannelName(mdgVo.getDistributionChannelName());
                salesOrgOne.setBusinessFormatCode(businessFormatCode);
                salesOrgOne.setSalesInstitutionCode(salesOrgOneCode);
                // 销售组织层级
                salesOrgOne.setSalesOrgLevel(oneEnum.getCode());
                salesOrgOne.setLevelNum(oneEnum.getLevel());
                // 租户编号
                salesOrgOne.setTenantCode(tenantCode);
                // 数据业务状态（启用状态）
                salesOrgOne.setEnableStatus(enableStatus);
                // 数据状态（删除状态）
                salesOrgOne.setDelFlag(delFlag);
                pullList.add(salesOrgOne);

            }

            if (!codeOneSet.contains(salesOrgTwoCode)) {
                codeOneSet.add(salesOrgTwoCode);
                SalesOrg salesOrgTwo = new SalesOrg();
                salesOrgTwo.setSalesOrgCode(salesOrgTwoCode);
                salesOrgTwo.setErpCode(salesOrgTwoErpCode);
                salesOrgTwo.setParentErpCode(salesOrgOneErpCode);
                salesOrgTwo.setParentCode(salesOrgOneCode);
                salesOrgTwo.setSalesInstitutionCode(salesOrgOneCode);
                salesOrgTwo.setSalesOrgName(mdgVo.getSaleOfficeName());
                salesOrgTwo.setSalesOrgDesc(salesOrgDesc + mdgVo.getSaleOfficeName());
                salesOrgTwo.setChannelCode(channelCode);
                salesOrgTwo.setChannelName(mdgVo.getDistributionChannelName());
                salesOrgTwo.setBusinessFormatCode(businessFormatCode);
                salesOrgTwo.setSalesInstitutionCode(salesOrgOneCode);
                // 销售组织层级
                salesOrgTwo.setSalesOrgLevel(twoEnum.getCode());
                salesOrgTwo.setLevelNum(twoEnum.getLevel());
                // 租户编号
                salesOrgTwo.setTenantCode(tenantCode);
                // 数据业务状态（启用状态）
                salesOrgTwo.setEnableStatus(enableStatus);
                // 数据状态（删除状态）
                salesOrgTwo.setDelFlag(delFlag);
                pullList.add(salesOrgTwo);
            }
        });
        // 编码去重
        return pullList;

    }

    /**
     * 数据校验 MDG销售办公室主数据
     *
     * @param mdgList
     * @return java.util.List<com.biz.crm.mdm.business.sales.org.local.entity.salesOrg>
     * @author 杜涛涛
     **/
    private List<SalesOrg> buildSalesOrgList(Set<String> codeSet,
                                             List<MasterDataMdgSalesOfficeVo> mdgList,
                                             SalesOrgLevelTypeEnum salesEnum,
                                             String tenantCode) {
        List<SalesOrg> pullList = new ArrayList<>();
        String enableStatus = EnableStatusEnum.ENABLE.getCode();
        String delFlag = DelFlagStatusEnum.NORMAL.getCode();
        mdgList.forEach(mdgVo -> {
            String parentErpCode = StringUtils.trimToEmpty(mdgVo.getZly1code());
            if (StringUtils.isEmpty(parentErpCode)) {
                log.error("本次拉取数据[来源1代码（销售部门）为空]：" + mdgVo);
                return;
            }
            String salesOrgErpCode = StringUtils.trimToEmpty(mdgVo.getZly2code());
            if (StringUtils.isEmpty(salesOrgErpCode)) {
                log.error("本次拉取数据[来源2代码（销售组）为空]：" + mdgVo);
                return;
            }
            if (parentErpCode.equals(salesOrgErpCode)) {
                log.error("本次拉取数据[上级编码是自己]：" + mdgVo);
                return;
            }
            mdgVo.setZly2text(StringUtils.trimToEmpty(mdgVo.getZly2text()));
            String key = parentErpCode + salesOrgErpCode;
            if (!codeSet.contains(key)) {
                codeSet.add(key);
                SalesOrg salesOrgThree = new SalesOrg();
                // 销售组织ERP编码
                salesOrgThree.setErpCode(salesOrgErpCode);
                // 销售组织名称
                salesOrgThree.setSalesOrgName(mdgVo.getZly2text());
                // 上级销售组织ERP编码
                salesOrgThree.setParentErpCode(parentErpCode);

                // 销售组织层级
                salesOrgThree.setSalesOrgLevel(salesEnum.getCode());
                salesOrgThree.setLevelNum(salesEnum.getLevel());
                // 租户编号
                salesOrgThree.setTenantCode(tenantCode);
                // 数据业务状态（启用状态）
                salesOrgThree.setEnableStatus(enableStatus);
                // 数据状态（删除状态）
                salesOrgThree.setDelFlag(delFlag);
                pullList.add(salesOrgThree);
            }
        });
        return pullList;

    }

    /**
     * 数据校验 MDG销售办公室主数据
     *
     * @param pullList
     * @param businessFormatMap
     * @return java.util.List<com.biz.crm.mdm.business.sales.org.local.entity.salesOrg>
     * @author 杜涛涛
     **/
    private List<SalesOrg> buildSalesOrgInfoList(List<SalesOrg> pullList,
                                                 Map<String, String> businessFormatMap,
                                                 SalesOrgLevelTypeEnum salesEnum,
                                                 String tenantCode) {
        if (CollectionUtils.isEmpty(pullList)) {
            return Collections.emptyList();
        }
        Set<String> parentErpCodeSet = pullList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getParentErpCode()))
                .map(SalesOrg::getParentErpCode)
                .collect(Collectors.toSet());
        List<SalesOrg> result = new ArrayList<>();
        if (CollectionUtils.isEmpty(parentErpCodeSet)) {
            return Collections.emptyList();
        }
        List<SalesOrg> parentList = salesOrgRepository.findAllByErpCodeList(new ArrayList<>(parentErpCodeSet), SalesOrgLevelTypeEnum.DEPARTMENT);
        if (CollectionUtils.isEmpty(parentList)) {
            return result;
        }
        String enableStatus = EnableStatusEnum.ENABLE.getCode();
        String delFlag = DelFlagStatusEnum.NORMAL.getCode();
        Map<String, List<SalesOrg>> parentMap = parentList.stream().collect(Collectors.groupingBy(SalesOrg::getErpCode));
        pullList.stream()
                .filter(k -> parentMap.containsKey(k.getParentErpCode()))
                .forEach(item -> {
                    List<SalesOrg> parentInfoList = parentMap.get(item.getParentErpCode());
                    if (CollectionUtils.isEmpty(parentInfoList)) {
                        return;
                    }
                    parentInfoList.forEach(parent -> {
                        SalesOrg salesOrg = new SalesOrg();
                        //基础信息使用上级销售组织信息
                        salesOrg.setChannelCode(parent.getChannelCode());
                        salesOrg.setChannelName(parent.getChannelName());
                        salesOrg.setBusinessUnitCode(parent.getBusinessUnitCode());
                        salesOrg.setBusinessFormatCode(parent.getBusinessFormatCode());
                        salesOrg.setSalesInstitutionCode(parent.getSalesInstitutionCode());
                        salesOrg.setParentCode(parent.getSalesOrgCode());
                        // 销售组织ERP编码
                        salesOrg.setErpCode(item.getErpCode());
                        // 销售组织名称
                        salesOrg.setSalesOrgName(item.getSalesOrgName());
                        // 上级销售组织ERP编码
                        salesOrg.setParentErpCode(item.getParentErpCode());
                        salesOrg.setSalesOrgCode(parent.getSalesOrgCode() + item.getErpCode());
                        // 销售组织名称
                        String salesOrgDesc = "";
                        if (StringUtils.isNotBlank(salesOrg.getChannelName())) {
                            salesOrgDesc = salesOrg.getChannelName();
                        }
                        String businessFormatName = businessFormatMap.get(salesOrg.getBusinessFormatCode());
                        if (StringUtils.isNotBlank(businessFormatName)) {
                            if (StringUtils.isNotBlank(salesOrgDesc)) {
                                salesOrgDesc = salesOrgDesc + "-";
                            }
                            salesOrgDesc = salesOrgDesc + businessFormatName + "-";
                        }
                        salesOrg.setSalesOrgDesc(salesOrgDesc + salesOrg.getSalesOrgName());


                        // 销售组织层级
                        salesOrg.setSalesOrgLevel(salesEnum.getCode());
                        salesOrg.setLevelNum(salesEnum.getLevel());
                        // 租户编号
                        salesOrg.setTenantCode(tenantCode);
                        // 数据业务状态（启用状态）
                        salesOrg.setEnableStatus(enableStatus);
                        // 数据状态（删除状态）
                        salesOrg.setDelFlag(delFlag);

                        result.add(salesOrg);

                    });
                });
        return result;

    }

    /**
     * 拉取组织渠道产品主数据 按年月日加锁 MDG
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private boolean lock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取组织渠道产品主数据 解锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(SalesOrgCodeConstant.SALES_ORG_PARENT_LOCK + yearMonthDay,
                TimeUnit.HOURS, 12);
    }

    /**
     * 拉取销售办公室主数据 按年月日加锁 MDG
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private boolean lockSalesOffice(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取拉取销售办公室主数据 解锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(SalesOrgCodeConstant.SALES_ORG_LOCK + yearMonthDay,
                TimeUnit.HOURS, 12);
    }

    /**
     * 最大分页数
     *
     * @param dto
     * @return int
     * @author huojia
     * @date 2023/1/8 1:33
     **/
    private int salesOrgCountPageMax(MasterDataMdgBaseDto dto) {
        Integer totalNum = masterDataMdgService.countList(MasterDataMdgConstants.MDG_SALES_OFFICE_TPM);
        int maxPage = 1;
        int sizeInt = Integer.parseInt(dto.getPageSize());
        if (totalNum > sizeInt) {
            maxPage = totalNum % sizeInt > 0 ? (totalNum / sizeInt) + 1 : totalNum / sizeInt;
        }
        return maxPage;
    }

}
