package com.biz.crm.mdm.business.sales.org.local.service.impl;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.dto.TreeDto;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.mdm.business.dictionary.sdk.constant.DictConstant;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.sales.org.local.entity.SalesOrg;
import com.biz.crm.mdm.business.sales.org.local.repository.SalesOrgRepository;
import com.biz.crm.mdm.business.sales.org.local.service.SalesOrgService;
import com.biz.crm.mdm.business.sales.org.sdk.constant.SalesOrgCodeConstant;
import com.biz.crm.mdm.business.sales.org.sdk.dto.*;
import com.biz.crm.mdm.business.sales.org.sdk.enums.SalesOrgLevelTypeEnum;
import com.biz.crm.mdm.business.sales.org.sdk.enums.SalesOrgTypeEnum;
import com.biz.crm.mdm.business.sales.org.sdk.event.SalesOrgEventListener;
import com.biz.crm.mdm.business.sales.org.sdk.event.SalesOrgLogEventListener;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import com.google.common.cache.Cache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import java.util.concurrent.TimeUnit;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @describe: 销售组织接口 实现
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.10.28 16:10
 */
@Service
@Slf4j
public class SalesOrgServiceImpl implements SalesOrgService {

    /**
     * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
     */
    private static volatile Cache<String, List<SalesOrg>> cache = null;
    private final int ONE = 1;
    @Autowired(required = false)
    private SalesOrgRepository salesOrgRepository;
    @Autowired(required = false)
    @Lazy
    private List<SalesOrgEventListener> orgEventListeners;
    @Autowired(required = false)
    private GenerateCodeService generateCodeService;
    @Autowired(required = false)
    private LoginUserService loginUserService;
    @Autowired(required = false)
    private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;
    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;
    @Autowired(required = false)
    private DictToolkitService dictToolkitService;
    @Autowired(required = false)
    private RedisMutexService redisMutexService;


    @Override
    public Page<SalesOrg> findByConditions(Pageable pageable, SalesOrgPaginationDto paginationDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
        if (paginationDto == null) {
            paginationDto = new SalesOrgPaginationDto();
        }
        paginationDto.setTenantCode(TenantUtils.getTenantCode());
        // 排除当前销售组织（及全部下级销售组织）
        String excludeAllChildrenRuleCode = null;
        if (StringUtils.isNotBlank(paginationDto.getExcludeAllChildrenOrgCode())) {
            excludeAllChildrenRuleCode = this.salesOrgRepository
                    .findRuleCodeByOrgCode(paginationDto.getExcludeAllChildrenOrgCode(), TenantUtils.getTenantCode());
            paginationDto.setExcludeAllChildrenRuleCode(excludeAllChildrenRuleCode);
        }
        // 查询当前销售组织（及全部下级销售组织）包含自己
        String includeAllChildrenRuleCode = null;
        if (StringUtils.isNotBlank(paginationDto.getIncludeAllChildrenOrgCode())) {
            includeAllChildrenRuleCode = this.salesOrgRepository
                    .findRuleCodeByOrgCode(paginationDto.getIncludeAllChildrenOrgCode(), TenantUtils.getTenantCode());
            paginationDto.setIncludeAllChildrenRuleCode(includeAllChildrenRuleCode);
        }
        // 查询当前销售组织（及全部下级销售组织）排除自己
        String includeAllChildrenOrgCodeExcludeSelf = paginationDto.getIncludeAllChildrenOrgCodeExcludeSelf();
        if (StringUtils.isNotBlank(includeAllChildrenOrgCodeExcludeSelf)) {
            List<SalesOrg> children = this.salesOrgRepository.findByParentCode(includeAllChildrenOrgCodeExcludeSelf,
                    TenantUtils.getTenantCode());
            if (CollectionUtils.isNotEmpty(children)) {
                List<String> targetRuleCodes = children.stream().filter(item -> {
                    return !Objects.equals(includeAllChildrenOrgCodeExcludeSelf, item.getSalesOrgCode());
                }).map(SalesOrg::getRuleCode).collect(Collectors.toList());
                List<String> targetSalesOrgCodes = this.salesOrgRepository
                        .findChildrenOrgCodeByRuleCodes(targetRuleCodes, TenantUtils.getTenantCode());
                paginationDto.setSalesOrgCodes(targetSalesOrgCodes);
            }
        }
        // 查询集合中的所有（及全部下级销售组织）包含自己
        if (CollectionUtils.isNotEmpty(paginationDto.getAllChildrenOrgCodesIncludeSelf())) {
            // TODO 后续改成从缓存中获取
            List<SalesOrg> orgs = this.salesOrgRepository
                    .findByOrgCodes(paginationDto.getAllChildrenOrgCodesIncludeSelf(), TenantUtils.getTenantCode());
            if (CollectionUtils.isNotEmpty(orgs)) {
                List<String> targetRuleCodes = orgs.stream().map(SalesOrg::getRuleCode).collect(Collectors.toList());
                List<String> targetSalesOrgCodes = this.salesOrgRepository
                        .findChildrenOrgCodeByRuleCodes(targetRuleCodes, TenantUtils.getTenantCode());
                paginationDto.setSalesOrgCodes(targetSalesOrgCodes);
            }
        }
        // 查询集合中的所有（及全部下级销售组织）排除自己，如果自己在其他子节点中那么保留
        List<String> allChildrenOrgCodesExcludeSelf = paginationDto.getAllChildrenOrgCodesExcludeSelf();
        if (CollectionUtils.isNotEmpty(paginationDto.getAllChildrenOrgCodesExcludeSelf())) {
            // TODO 后续改成从缓存中获取
            List<SalesOrg> orgs = this.salesOrgRepository
                    .findByOrgCodes(paginationDto.getAllChildrenOrgCodesExcludeSelf(), TenantUtils.getTenantCode());
            if (CollectionUtils.isNotEmpty(orgs)) {
                List<String> targetRuleCodes =
                        orgs.stream().filter(item -> !allChildrenOrgCodesExcludeSelf.contains(item.getSalesOrgCode()))
                                .map(SalesOrg::getRuleCode).collect(Collectors.toList());
                List<String> targetSalesOrgCodes = this.salesOrgRepository
                        .findChildrenOrgCodeByRuleCodes(targetRuleCodes, TenantUtils.getTenantCode());
                paginationDto.setSalesOrgCodes(targetSalesOrgCodes);
            }
        }
        // 查询集合中的所有（及全部下级销售组织）任何情况下都排除自己
        List<String> allChildrenOrgCodesExcludeAnySelf = paginationDto.getAllChildrenOrgCodesExcludeAnySelf();
        if (CollectionUtils.isNotEmpty(allChildrenOrgCodesExcludeAnySelf)) {
            // TODO 后续改成从缓存中获取
            List<SalesOrg> orgs = this.salesOrgRepository
                    .findByOrgCodes(paginationDto.getAllChildrenOrgCodesExcludeAnySelf(), TenantUtils.getTenantCode());
            if (CollectionUtils.isNotEmpty(orgs)) {
                List<String> targetRuleCodes = orgs.stream().map(SalesOrg::getRuleCode).collect(Collectors.toList());
                List<String> targetSalesOrgCodes = this.salesOrgRepository
                        .findChildrenOrgCodeByRuleCodes(targetRuleCodes, TenantUtils.getTenantCode());
                targetSalesOrgCodes = targetSalesOrgCodes.stream().filter(item -> {
                    return !allChildrenOrgCodesExcludeAnySelf.contains(item);
                }).collect(Collectors.toList());
                paginationDto.setSalesOrgCodes(targetSalesOrgCodes);
            }
        }

        Set<String> codeList = Sets.newHashSet();
        if (StringUtils.isNotBlank(paginationDto.getSelectedCode())) {
            codeList.add(paginationDto.getSelectedCode());
        }
        if (CollectionUtils.isNotEmpty(paginationDto.getSelectedCodes())) {
            codeList.addAll(paginationDto.getSelectedCodes());
        }
        if (CollectionUtils.isNotEmpty(codeList)) {
            paginationDto.setSelectedCodes(Lists.newArrayList(codeList));
        }
        //获取业态字典
        Map<String, String> formatMap = dictToolkitService.findMapByDictTypeCode(DictConstant.MDM_BUSINESS_FORMAT);
        Page<SalesOrg> page = this.salesOrgRepository.findByConditions(pageable, paginationDto);
        if (!CollectionUtils.isEmpty(page.getRecords())) {
            page.getRecords().forEach(record -> {
                record.setUnionName(StringUtils.join(record.getSalesOrgCode(), "/", record.getSalesOrgName()));
                record.setUnionNameExt(StringUtils.join(record.getSalesOrgCode(), "/", record.getSalesOrgName(), "/", record.getChannelName(), "/"));
                if (StringUtils.isNotBlank(record.getBusinessFormatCode())) {
                    record.setUnionNameExt(StringUtils.join(record.getUnionNameExt(), formatMap.getOrDefault(record.getBusinessFormatCode(), record.getBusinessFormatCode())));
                }
            });
        }
        return page;
    }

    @Override
    public SalesOrg findDetailsById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        return this.salesOrgRepository.findDetailsById(id);
    }

    @Override
    public SalesOrg findByOrgCode(String orgCode) {
        if (StringUtils.isBlank(orgCode)) {
            return null;
        }
        return this.salesOrgRepository.findByOrgCode(orgCode, TenantUtils.getTenantCode());
    }

    @Override
    public Map<String, String> findSapCodesByOrgCodes(List<String> salesOrgCodes) {
        if (CollectionUtils.isEmpty(salesOrgCodes)) {
            return Maps.newHashMap();
        }
        List<SalesOrg> list = this.salesOrgRepository.findSapCodesByOrgCodes(salesOrgCodes, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(list)) {
            return Maps.newHashMap();
        }
        Map<String, String> sapMap = new HashMap<>();
        list.forEach(salesOrg -> {
            sapMap.put(salesOrg.getSalesOrgCode(), salesOrg.getErpCode());
        });
        return sapMap;
    }

    @Override
    public List<SalesOrg> findDetailsByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return null;
        }
        return this.salesOrgRepository.findDetailsByIds(ids);
    }

    @Override
    public List<SalesOrg> findBySalesOrgCodeLikeOrSalesOrgNameLike(String orgCodeLikeorNameLike) {
        return this.salesOrgRepository.findByOrgCodeLikeOrOrgNameLike(orgCodeLikeorNameLike);

    }

    @Override
    public List<SalesOrg> findByOrgCodes(List<String> orgCodes) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return null;
        }
        return this.salesOrgRepository.findByOrgCodes(orgCodes, TenantUtils.getTenantCode());
    }

    @Override
    @Transactional
    public void deleteByIds(List<String> ids) {
        Validate.notEmpty(ids, "id集合不能为空");
        List<SalesOrg> orgs = this.findDetailsByIds(ids);
        Validate.notEmpty(orgs, "无效的参数");
        List<String> orgCodes = orgs.stream().map(SalesOrg::getSalesOrgCode).collect(Collectors.toList());
        List<SalesOrg> children = this.salesOrgRepository.findByParentCodes(orgCodes, TenantUtils.getTenantCode());
        Validate.isTrue(CollectionUtils.isEmpty(children), "存在下级销售组织，不能删除");
        // 删除事件 TODO /终端/客户/行政区域 需要实现销售组织删除事件，在销售组织删除时校验是否存在关联
        if (Objects.nonNull(this.orgEventListeners)) {
            for (SalesOrgEventListener listener : this.orgEventListeners) {
                listener.onDelete(orgCodes);
            }
        }
        // 逻辑删除
        for (SalesOrg org : orgs) {
            org.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
        }
        this.salesOrgRepository.updateBatchById(orgs);
        // 如果ParentCode为空，则手动设置为空
        for (SalesOrg org : orgs) {
            salesOrgRepository.setParentCodeAndRuleCodeNull(org.getId());
        }
        // 发布引擎事件通知
        List<SalesOrgEventDto> orgEventDtoList =
                Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(children, SalesOrg.class,
                        SalesOrgEventDto.class, HashSet.class, ArrayList.class));
        SalesOrgEventBatchDto orgEventBatchDto = new SalesOrgEventBatchDto();
        orgEventBatchDto.setSalesOrgEventDtoList(orgEventDtoList);
        SerializableBiConsumer<SalesOrgEventListener, SalesOrgEventBatchDto> sf = SalesOrgEventListener::onDeleteBatch;
        this.nebulaNetEventClient.publish(orgEventBatchDto, SalesOrgEventListener.class, sf);

        // TODO 清除销售组织缓存
        // TODO 日志处理
    }

    @Override
    @Transactional
    public void enableBatch(List<String> ids) {
        Validate.notEmpty(ids, "id集合不能为空");
        // 通过ruleCode 取出上级ruleCode 用于判断父级节点是否被禁用
        List<String> ruleCodes = this.salesOrgRepository.findRuleCodeByIds(ids);
        if (CollectionUtils.isNotEmpty(ruleCodes)) {
            Validate.notNull(this.treeRuleCodeStrategyHolder, "系统未配置降维码策略控制器");
            TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
            Validate.notNull(treeRuleCodeStrategy, "系统未配置降维码策略");
            Set<String> parentRuleCodes = treeRuleCodeStrategy
                    .findParentRuleCodeByRuleCodesExcludeAnySelf(SalesOrgCodeConstant.RULE_CODE_LENGTH, ruleCodes);
            if (CollectionUtils.isNotEmpty(parentRuleCodes)) {
                List<SalesOrg> parentDisableList = this.salesOrgRepository
                        .findByRuleCodesAndEnableStatus(new ArrayList<>(parentRuleCodes), EnableStatusEnum.DISABLE,
                                TenantUtils.getTenantCode())
                        .stream().filter(org -> !ids.contains(org.getId())).collect(Collectors.toList());
                Validate.isTrue(CollectionUtils.isEmpty(parentDisableList), "存在未启用的上级销售组织,不能启用当前销售组织");
            }
        }
        this.salesOrgRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
        // 发布引擎事件通知
        List<SalesOrg> orgList = this.salesOrgRepository.getBaseMapper().selectBatchIds(ids);
        List<SalesOrgEventDto> orgEventDtoList =
                Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(orgList, SalesOrg.class,
                        SalesOrgEventDto.class, HashSet.class, ArrayList.class));
        SalesOrgEventBatchDto orgEventBatchDto = new SalesOrgEventBatchDto();
        orgEventBatchDto.setSalesOrgEventDtoList(orgEventDtoList);
        SerializableBiConsumer<SalesOrgEventListener, SalesOrgEventBatchDto> sf = SalesOrgEventListener::onEnableBatch;
        this.nebulaNetEventClient.publish(orgEventBatchDto, SalesOrgEventListener.class, sf);
        // 日志
        List<SalesOrg> orgs = this.salesOrgRepository.findDetailsByIds(ids);
        for (SalesOrg org : orgs) {
            SalesOrgLogEventDto orgLogEventDto = new SalesOrgLogEventDto();
            SalesOrgVo oldObj = new SalesOrgVo();
            SalesOrgVo newObj = new SalesOrgVo();
            oldObj.setId(org.getId());
            newObj.setId(org.getId());
            oldObj.setEnableStatus(org.getEnableStatus());
            newObj.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            orgLogEventDto.setNewest(newObj);
            orgLogEventDto.setOriginal(oldObj);
            SerializableBiConsumer<SalesOrgLogEventListener, SalesOrgLogEventDto> onUpdate =
                    SalesOrgLogEventListener::onUpdate;
            this.nebulaNetEventClient.publish(orgLogEventDto, SalesOrgLogEventListener.class, onUpdate);
        }
    }

    @Override
    @Transactional
    public void disableBatch(List<String> ids) {
        Validate.notEmpty(ids, "id集合不能为空");
        List<String> orgCodes = this.findChildrenOrgCodeByIds(ids);
        if (CollectionUtils.isNotEmpty(orgCodes)) {
            this.salesOrgRepository.updateEnableStatusByOrgCodes(orgCodes, EnableStatusEnum.DISABLE,
                    TenantUtils.getTenantCode());
            // TODO 清除销售组织缓存
        }
        // 发布引擎事件通知
        List<SalesOrg> orgList = this.salesOrgRepository.getBaseMapper().selectBatchIds(ids);
        List<SalesOrgEventDto> orgEventDtoList =
                Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(orgList, SalesOrg.class,
                        SalesOrgEventDto.class, HashSet.class, ArrayList.class));
        SalesOrgEventBatchDto orgEventBatchDto = new SalesOrgEventBatchDto();
        orgEventBatchDto.setSalesOrgEventDtoList(orgEventDtoList);
        SerializableBiConsumer<SalesOrgEventListener, SalesOrgEventBatchDto> sf = SalesOrgEventListener::onDisableBatch;
        this.nebulaNetEventClient.publish(orgEventBatchDto, SalesOrgEventListener.class, sf);
        // 日志
        List<SalesOrg> orgs = this.salesOrgRepository.findDetailsByIds(ids);
        for (SalesOrg org : orgs) {
            SalesOrgLogEventDto orgLogEventDto = new SalesOrgLogEventDto();
            SalesOrgVo oldObj = new SalesOrgVo();
            SalesOrgVo newObj = new SalesOrgVo();
            oldObj.setId(org.getId());
            newObj.setId(org.getId());
            oldObj.setEnableStatus(org.getEnableStatus());
            newObj.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            orgLogEventDto.setNewest(newObj);
            orgLogEventDto.setOriginal(oldObj);
            SerializableBiConsumer<SalesOrgLogEventListener, SalesOrgLogEventDto> onUpdate =
                    SalesOrgLogEventListener::onUpdate;
            this.nebulaNetEventClient.publish(orgLogEventDto, SalesOrgLogEventListener.class, onUpdate);
        }
    }

    @Override
    @Transactional
    public SalesOrg create(SalesOrg org) {
        SalesOrg current = this.createForm(org);
        SalesOrg salesOrgAll = this.salesOrgRepository.findAllByOrgCode(org);
        if (!Objects.isNull(salesOrgAll)) {
            throw new IllegalArgumentException("销售组织编码已存在（已逻辑删除）");
        }
        this.salesOrgRepository.save(current);
        // TODO 清除销售组织缓存
        // 日志处理
        SalesOrgLogEventDto orgLogEventDto = new SalesOrgLogEventDto();
        orgLogEventDto.setNewest(
                this.nebulaToolkitService.copyObjectByWhiteList(current, SalesOrgVo.class, HashSet.class, ArrayList.class));
        SerializableBiConsumer<SalesOrgLogEventListener, SalesOrgLogEventDto> onCreate =
                SalesOrgLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(orgLogEventDto, SalesOrgLogEventListener.class, onCreate);
        return current;
    }

    @Override
    @Transactional
    public SalesOrg update(SalesOrg org) {
        SalesOrg current = this.updateForm(org);
        // TODO 清除销售组织缓存
        return current;
    }

    @Override
    public List<SalesOrg> findAllChildrenById(String id) {
        SalesOrg parent = this.findDetailsById(id);
        if (Objects.isNull(parent)) {
            return null;
        }
        return this.salesOrgRepository.findChildrenByRuleCode(parent.getRuleCode(), TenantUtils.getTenantCode());
    }

    @Override
    public List<SalesOrg> findAllChildrenByOrgCode(String orgCode) {
        SalesOrg parent = this.findByOrgCode(orgCode);
        if (Objects.isNull(parent)) {
            return null;
        }
        return this.salesOrgRepository.findChildrenByRuleCode(parent.getRuleCode(), TenantUtils.getTenantCode());
    }

    @Override
    public List<SalesOrg> findAllChildrenBySalesOrgCodes(List<String> orgCodes) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return null;
        }
        List<String> ruleCodes = this.salesOrgRepository.findRuleCodeByOrgCodes(orgCodes, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return null;
        }
        return this.salesOrgRepository.findChildrenByRuleCode(ruleCodes, TenantUtils.getTenantCode());
    }

    @Override
    public List<SalesOrg> findChildrenById(String id) {
        SalesOrg parent = this.findDetailsById(id);
        if (Objects.isNull(parent)) {
            return null;
        }
        return this.salesOrgRepository.findByParentCode(parent.getSalesOrgCode(), TenantUtils.getTenantCode());
    }

    @Override
    public List<SalesOrg> findChildrenByOrgCode(String orgCode) {
        if (StringUtils.isBlank(orgCode)) {
            return null;
        }
        return this.salesOrgRepository.findByParentCode(orgCode, TenantUtils.getTenantCode());
    }

    @Override
    public List<SalesOrg> findAllParentById(String id) {
        SalesOrg current = this.findDetailsById(id);
        if (Objects.isNull(current)) {
            return null;
        }
        return this.findAllParentByRuleCodes(Lists.newArrayList(current.getRuleCode()));
    }

    @Override
    public List<SalesOrg> findAllParentByOrgCode(String orgCode) {
        SalesOrg current = this.findByOrgCode(orgCode);
        if (Objects.isNull(current)) {
            return null;
        }
        return this.findAllParentByRuleCodes(Lists.newArrayList(current.getRuleCode()));
    }

    @Override
    public List<SalesOrg> findAllParentByOrgCodes(List<String> orgCodes) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return null;
        }
        List<SalesOrg> orgs = this.findByOrgCodes(orgCodes);
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }
        List<String> ruleCodes = orgs.stream().filter(item -> StringUtils.isNotBlank(item.getRuleCode()))
                .map(SalesOrg::getRuleCode).collect(Collectors.toList());
        return this.findAllParentByRuleCodes(ruleCodes);
    }

    @Override
    public List<SalesOrg> findAllParentByRuleCodes(List<String> ruleCodes) {
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return null;
        }
        Validate.notNull(this.treeRuleCodeStrategyHolder, "系统未配置降维码策略控制器");
        TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
        Validate.notNull(treeRuleCodeStrategy, "系统未配置降维码策略");
        Set<String> ruleCodeAll =
                treeRuleCodeStrategy.findParentRuleCodeByRuleCodes(SalesOrgCodeConstant.RULE_CODE_LENGTH, ruleCodes);
        if (CollectionUtils.isEmpty(ruleCodeAll)) {
            return null;
        }
        return this.salesOrgRepository.findByRuleCodesAndEnableStatus(ruleCodeAll.stream().collect(Collectors.toList()),
                null, TenantUtils.getTenantCode());
    }

    @Override
    public List<SalesOrg> findAllParentSalesOrgCodeExcludeSelf(List<String> orgCodes) {
        List<SalesOrg> orgs = this.findByOrgCodes(orgCodes);
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }
        List<String> ruleCodes = orgs.stream().map(SalesOrg::getRuleCode).collect(Collectors.toList());
        TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
        Set<String> ruleCodeAll = treeRuleCodeStrategy
                .findParentRuleCodesByRuleCodesExcludeSelf(SalesOrgCodeConstant.RULE_CODE_LENGTH, ruleCodes);
        if (CollectionUtils.isEmpty(ruleCodeAll)) {
            return null;
        }
        return this.salesOrgRepository.findByRuleCodesAndEnableStatus(ruleCodeAll.stream().collect(Collectors.toList()),
                null, TenantUtils.getTenantCode());
    }

    @Override
    public SalesOrg findParentById(String id) {
        SalesOrg current = this.findDetailsById(id);
        if (Objects.isNull(current)) {
            return null;
        }
        if (StringUtils.isBlank(current.getParentCode())) {
            return null;
        }
        return this.findByOrgCode(current.getParentCode());
    }

    @Override
    public SalesOrg findParentBySalesOrgCode(String orgCode) {
        SalesOrg current = this.findByOrgCode(orgCode);
        if (Objects.isNull(current)) {
            return null;
        }
        if (StringUtils.isBlank(current.getParentCode())) {
            return null;
        }
        return this.findByOrgCode(current.getParentCode());
    }

    @Override
    public List<SalesOrg> findAllChildrenBySalesOrgCodes(Pageable pageable, List<String> orgCodes) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
        if (CollectionUtils.isEmpty(orgCodes)) {
            return null;
        }
        List<String> ruleCodes = this.salesOrgRepository.findRuleCodeByOrgCodes(orgCodes, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return null;
        }
        Page<SalesOrg> page =
                this.salesOrgRepository.findChildrenByRuleCodes(pageable, ruleCodes, TenantUtils.getTenantCode());
        if (Objects.isNull(page)) {
            return null;
        }
        return page.getRecords();
    }

    @Override
    public List<SalesOrg> findByConditions(SalesOrgPaginationDto dto) {
        if (dto == null) {
            dto = new SalesOrgPaginationDto();
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.salesOrgRepository.findByConditions(dto);
    }

    @Override
    public List<SalesOrg> findByParentCode(String orgCode) {
        return this.findChildrenByOrgCode(orgCode);
    }

    /**
     * 获取redis锁定key
     * @return
     */
    @Override
    public String getUpdateRuleLockKey(){
        return SalesOrgCodeConstant.SALES_ORG_UPDATE_RULE_LOCK;
    }

    /**
     * 重置降维编码
     *
     * @param userIdentity
     */
    @Override
    @Async
    @Transactional(rollbackFor = Exception.class)
    public void updateRuleCode(UserIdentity userIdentity) {
        boolean lockFlag = redisMutexService.tryLock(getUpdateRuleLockKey(), TimeUnit.MINUTES, 90);
        Assert.isTrue(lockFlag,"降维编码加锁失败!");
        try {
            loginUserService.refreshAuthentication(userIdentity);
            // 1.查找parentCode不为空但找不到对应上级的数据，将其parentCode设为空
            salesOrgRepository.updateOrphanParentCodeNull(TenantUtils.getTenantCode());
            // 2.查找所有parentCode为空的数据（相当于第一层数据）
            List<SalesOrg> topList = salesOrgRepository.findListWithoutParentCode(TenantUtils.getTenantCode());
            // 3.递归设置其ruleCode和其所有子级的ruleCode
            TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
            for (int i = 0; i < topList.size(); i++) {
                // 递归调用
                this.updateCurAndChildrenRuleCode(topList.get(i).getSalesOrgCode(),
                        treeRuleCodeStrategy.generateByNum(SalesOrgCodeConstant.RULE_CODE_LENGTH, i + ONE), ONE);
            }
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            redisMutexService.unlock(getUpdateRuleLockKey());
        }
    }

    /**
     * 更新自己及子集的降维编码
     *
     * @param orgCode
     * @param ruleCode
     * @param levelNum
     */
    private void updateCurAndChildrenRuleCode(String orgCode, String ruleCode, int levelNum) {
        // 更新当前
        SalesOrg org = salesOrgRepository.findDetailsByCode(orgCode, TenantUtils.getTenantCode());
        if (org == null) {
            return;
        }
        org.setRuleCode(ruleCode);
        org.setLevelNum(levelNum);
        salesOrgRepository.updateById(org);
        // 查询下一层
        List<SalesOrg> childrenList =
                salesOrgRepository.findChildrenListByParentCode(orgCode, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(childrenList)) {
            return;
        }
        // 遍历下级
        TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
        for (int i = 0; i < childrenList.size(); i++) {
            // 递归调用
            this.updateCurAndChildrenRuleCode(childrenList.get(i).getSalesOrgCode(),
                    ruleCode + treeRuleCodeStrategy.generateByNum(SalesOrgCodeConstant.RULE_CODE_LENGTH, i + ONE),
                    levelNum + ONE);
        }
    }

    /**
     * 更新
     *
     * @param org
     * @return
     */
    private SalesOrg updateForm(SalesOrg org) {
        /*
         * 对静态模型的修改操作的过程为：
         * 1、如果当前模型对象不是主模型
         *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
         *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
         *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
         *
         * 2、如果当前模型对象是主业务模型
         *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
         *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
         *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
         *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
         *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
         *    2.3.2、以及验证每个分组的OneToMany明细信息
         */
        org.setTenantCode(TenantUtils.getTenantCode());
        this.updateValidation(org);
        // 这里可根据id或者code更新
        String currentId = org.getId();
        SalesOrg current = this.salesOrgRepository.getById(currentId);
        current = Validate.notNull(current, "未发现指定的原始模型对象信");
        Validate.isTrue(Objects.equals(org.getSalesOrgCode(), current.getSalesOrgCode()), "销售组织编码不能修改");
        int levelNum = ONE;
        String parentCode = org.getParentCode();
        String parentRuleCode = null;
        List<SalesOrg> children = Lists.newArrayList();
        List<TreeDto> childrenDto = Lists.newArrayList();
        if (StringUtils.isNotEmpty(parentCode)) {
            SalesOrg parent = this.findByOrgCode(parentCode);
            Validate.notNull(parent, "上级销售组织不存在");
            Validate.isTrue(!parent.getRuleCode().startsWith(current.getRuleCode()), "上级销售组织不能是当前销售组织的下级销售组织");
            levelNum = parent.getLevelNum() + ONE;
            parentRuleCode = parent.getRuleCode();
            children = this.salesOrgRepository.findByParentCode(parentCode, TenantUtils.getTenantCode());
        } else {
            children = this.salesOrgRepository.findByParentCodeIsNull(TenantUtils.getTenantCode());
        }
        if (CollectionUtils.isNotEmpty(children)) {
            childrenDto = (List<TreeDto>) this.nebulaToolkitService.copyCollectionByWhiteList(children, SalesOrg.class,
                    TreeDto.class, HashSet.class, ArrayList.class);
        }
        String oldParentCode = current.getParentCode();
        BeanUtils.copyProperties(org, current, "id", "modifyTime", "createAccount", "createTime", "tenantCode",
                "ruleCode");
        // 开始赋值——更新时间与更新人
        Date now = new Date();
        String account = this.loginUserService.getLoginAccountName();
        current.setLevelNum(levelNum);
        current.setModifyAccount(account);
        current.setModifyTime(now);
        this.salesOrgRepository.saveOrUpdate(current);
        if (StringUtils.isBlank(org.getParentCode())) {
            // 如果ParentCode为空，则手动设置为空
            salesOrgRepository.setParentCodeNull(org.getId());
        }
        // 判断启用状态是否发生变化
        if (!Objects.equals(org.getEnableStatus(), current.getEnableStatus())) {
            if (Objects.equals(org.getEnableStatus(), EnableStatusEnum.DISABLE.getCode())) {
                this.disableBatch(Lists.newArrayList(org.getId()));
            } else if (Objects.equals(org.getEnableStatus(), EnableStatusEnum.ENABLE.getCode())) {
                this.enableBatch(Lists.newArrayList(org.getId()));
            }
        }
        // 是否重置降维编码 取决于上级有无变化,更新所有下级节点 ruleCode
        if (!Objects.equals(parentCode, oldParentCode)) {
            Validate.notNull(this.treeRuleCodeStrategyHolder, "系统未配置降维码策略控制器");
            TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
            Validate.notNull(treeRuleCodeStrategy, "系统未配置降维码策略");
            String ruleCode =
                    treeRuleCodeStrategy.generate(SalesOrgCodeConstant.RULE_CODE_LENGTH, parentRuleCode, childrenDto);
            this.updateRuleCodeAllChildren(org.getSalesOrgCode(), ruleCode, levelNum);
            current.setRuleCode(ruleCode);
        }
        SalesOrgEventDto orgEventDto =
                nebulaToolkitService.copyObjectByWhiteList(current, SalesOrgEventDto.class, HashSet.class, ArrayList.class);
        orgEventDto.setOldParentCode(Optional.ofNullable(oldParentCode).orElse(StringUtils.EMPTY));
        orgEventDto.setParentCode(Optional.ofNullable(org.getParentCode()).orElse(StringUtils.EMPTY));
        SerializableBiConsumer<SalesOrgEventListener, SalesOrgEventDto> sf = SalesOrgEventListener::onUpdate;
        nebulaNetEventClient.publish(orgEventDto, SalesOrgEventListener.class, sf);
        // 日志
        SalesOrgLogEventDto orgLogEventDto = new SalesOrgLogEventDto();
        orgLogEventDto.setNewest(
                this.nebulaToolkitService.copyObjectByWhiteList(current, SalesOrgVo.class, HashSet.class, ArrayList.class));
        orgLogEventDto.setOriginal(
                this.nebulaToolkitService.copyObjectByWhiteList(org, SalesOrgVo.class, HashSet.class, ArrayList.class));
        SerializableBiConsumer<SalesOrgLogEventListener, SalesOrgLogEventDto> onUpdate =
                SalesOrgLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(orgLogEventDto, SalesOrgLogEventListener.class, onUpdate);
        return current;
    }

    /**
     * 创建
     *
     * @param org
     * @return
     */
    private SalesOrg createForm(SalesOrg org) {
        /*
         * 对静态模型的保存操作过程为：
         * 1、如果当前模型对象不是主模型
         *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
         *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
         *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
         * 2、如果当前模型对象是主业务模型
         *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
         *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
         *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
         * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
         *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
         *   2.3.2、以及验证每个分组的OneToMany明细信息
         * */
        Date now = new Date();
        String account = this.loginUserService.getLoginAccountName();
        org.setCreateAccount(account);
        org.setModifyAccount(account);
        org.setTenantCode(TenantUtils.getTenantCode());
        org.setCreateTime(now);
        org.setModifyTime(now);

        org.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        org.setEnableStatus(null != org.getEnableStatus() ? org.getEnableStatus() : EnableStatusEnum.ENABLE.getCode());
        if (StringUtils.isBlank(org.getSalesOrgCode())) {
            String code = this.generateCodeService.generateCode(SalesOrgCodeConstant.CODE, ONE).get(0);
            org.setSalesOrgCode(code);
        } else {
            SalesOrg exits = this.findByOrgCode(org.getSalesOrgCode());
            Validate.isTrue(exits == null, "销售组织编码已经存在");
        }
        int levelNum = 1;
        if (StringUtils.isNotEmpty(org.getParentCode())) {
            SalesOrg parent =
                    this.salesOrgRepository.findDetailsByCode(org.getParentCode(), TenantUtils.getTenantCode());
            Validate.notNull(parent, "父节点不存在!");
            levelNum = parent.getLevelNum() + 1;
        }
        org.setParentCode(Optional.ofNullable(org.getParentCode()).orElse(StringUtils.EMPTY));
        String parentCode = org.getParentCode();
        String ruleCode = this.getRuleCodeByParentCode(parentCode);
        org.setRuleCode(ruleCode);
        org.setLevelNum(levelNum);
        String orgType = org.getSalesOrgType();
        if (StringUtils.isBlank(orgType)) {
            for (SalesOrgTypeEnum value : SalesOrgTypeEnum.values()) {
                String order = value.getOrder();
                Integer integer = Integer.valueOf(order);
                if (integer == levelNum) {
                    org.setSalesOrgType(value.getDictCode());
                }
            }
        }
        this.createValidation(org);
        return org;
    }

    /**
     * 校验数据
     *
     * @param org
     */
    private void createValidation(SalesOrg org) {
        Validate.notNull(org, "进行当前操作时，信息对象必须传入!!");
        // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
        Validate.isTrue(StringUtils.isBlank(org.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
        org.setId(null);
        Validate.notBlank(org.getTenantCode(), "添加信息时，租户编号不能为空！");
        Validate.notBlank(org.getSalesOrgName(), "添加信息时，销售组织名称不能为空！");
        Validate.notBlank(org.getSalesOrgType(), "添加信息时，销售组织类型不能为空！");
        String parentCode = org.getParentCode();
        if (StringUtils.isNotBlank(parentCode)) {
            SalesOrg parent = this.findByOrgCode(parentCode);
            Validate.notNull(parent, "父节点不存在!");
        }
        if (StringUtils.isNotBlank(org.getSalesOrgCode())) {
            Validate.isTrue(org.getSalesOrgCode().length() < 32, "销售组织编码信息，在进行新增时填入值超过了限定长度(32)，请检查!");
        }
        Validate.isTrue(org.getSalesOrgName().length() < 64, "销售组织名称，在进行新增时填入值超过了限定长度(64)，请检查!");
        if (StringUtils.isNotBlank(org.getSalesOrgDesc())) {
            Validate.isTrue(org.getSalesOrgDesc() == null || org.getSalesOrgDesc().length() < 128,
                    "销售组织描述，在进行新增时填入值超过了限定长度(128)，请检查!");

        }
    }

    /**
     * 更新校验
     *
     * @param org
     */
    private void updateValidation(SalesOrg org) {
        Validate.notNull(org, "进行当前操作时，信息对象必须传入!!");
        // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
        Validate.notBlank(org.getId(), "修改信息时，id不能为空！");
        Validate.notBlank(org.getTenantCode(), "修改信息时，租户编号不能为空！");
        Validate.isTrue(org.getSalesOrgName().length() < 64, "销售组织名称，在进行修改时填入值超过了限定长度(64)，请检查!");
        Validate.isTrue(org.getSalesOrgDesc() == null || org.getSalesOrgDesc().length() < 128,
                "销售组织描述，在进行修改时填入值超过了限定长度(32)，请检查!");
        String parentCode = org.getParentCode();
        if (StringUtils.isNotBlank(parentCode)) {
            SalesOrg parent = this.findByOrgCode(parentCode);
            Validate.notNull(parent, "父节点不存在!");
        }
    }

    /**
     * 通过id集合查询所以子节点的orgCode
     *
     * @param ids
     * @return
     */
    private List<String> findChildrenOrgCodeByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return null;
        }
        List<String> ruleCodes = this.salesOrgRepository.findRuleCodeByIds(ids);
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return null;
        }
        return this.salesOrgRepository.findChildrenOrgCodeByRuleCodes(ruleCodes, TenantUtils.getTenantCode());
    }

    @Override
    public List<SalesOrg> findChildrenByRuleCode(List<String> ruleCodes) {
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return null;
        }
        String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), ruleCodes);
        List<SalesOrg> graph = cache.getIfPresent(cacheKey);
        if (graph == null) {
            graph = this.salesOrgRepository.findChildrenByRuleCode(ruleCodes, TenantUtils.getTenantCode());
            cache.put(cacheKey, graph);
        }
        return graph;
    }

    @Override
    public List<SalesOrg> findByRuleCodesAndEnableStatus(List<String> ruleCodes, EnableStatusEnum enable) {
        if (CollectionUtils.isEmpty(ruleCodes)) {
            return null;
        }
        String cacheKey =
                StringUtils.join(TenantUtils.getTenantCode(), ruleCodes, enable != null ? enable.getCode() : null);
        List<SalesOrg> graph = cache.getIfPresent(cacheKey);
        if (graph == null) {
            graph =
                    this.salesOrgRepository.findByRuleCodesAndEnableStatus(ruleCodes, enable, TenantUtils.getTenantCode());
            cache.put(cacheKey, graph);
        }
        return graph;
    }

    @Override
    public Page<SalesOrg> findBySalesSelectDto(Pageable pageable, SalesOrgSelectDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new SalesOrgSelectDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<SalesOrg> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        if (StringUtils.isNotEmpty(dto.getTopSalesOrgCode())) {
            // 有顶级编码的话要设置下ruleCode
            List<SalesOrg> topList =
                    salesOrgRepository.findBySalesOrgCodeLists(Lists.newArrayList(dto.getTopSalesOrgCode()));
            if (CollectionUtils.isEmpty(topList)) {
                throw new RuntimeException("顶级销售组织编码[" + dto.getTopSalesOrgCode() + "]有误");
            }
            dto.setRuleCode(topList.get(0).getRuleCode());
        }
        page = this.salesOrgRepository.findBySalesSelectDto(page, dto);
        if (!CollectionUtils.isEmpty(page.getRecords())) {
            page.getRecords().forEach(record -> {
                record.setUnionName(StringUtils.join(record.getSalesOrgCode(), "/", record.getSalesOrgName()));
            });
        }
        return page;
    }

    /**
     * 批量新增数据（组织渠道产品主数据MDG用）
     *
     * @param saveList
     * @param updateList
     * @author dutaotao
     * @date 2023/1/3 17:38
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveOrUpdateBatchForMdg(List<SalesOrg> saveList, List<SalesOrg> updateList) {
        if (!CollectionUtils.isEmpty(saveList)) {
            log.info("销售组织新增[{}]条", saveList.size());
            List<List<SalesOrg>> saveGroupList = Lists.partition(saveList, 400);
            saveGroupList.forEach(list -> {
                this.salesOrgRepository.saveBatch(list);
            });
        }

        if (!CollectionUtils.isEmpty(updateList)) {
            log.info("销售组织更新[{}]条", updateList.size());
            List<List<SalesOrg>> updateGroupList = Lists.partition(updateList, 400);
            updateGroupList.forEach(list -> {
                this.salesOrgRepository.updateBatchById(list);
            });
        }
    }

    /**
     * 更新销售组织及销售组织下级降维编码和层级
     *
     * @param orgCode  当前销售组织编码
     * @param ruleCode 当前销售组织降维编码
     * @param levelNum 当前层级
     */
    private void updateRuleCodeAllChildren(String orgCode, String ruleCode, int levelNum) {
        // 更新当前
        this.salesOrgRepository.updateRuleCodeAndLevelNumByOrgCode(orgCode, ruleCode, levelNum,
                TenantUtils.getTenantCode());
        // 查询下一层
        List<SalesOrg> children = this.salesOrgRepository.findByParentCode(orgCode, TenantUtils.getTenantCode());
        if (CollectionUtils.isNotEmpty(children)) {
            // 遍历下级
            for (int i = 0; i < children.size(); i++) {
                // 递归调用
                SalesOrg item = children.get(i);
                Validate.notNull(this.treeRuleCodeStrategyHolder, "系统未配置降维码策略控制器");
                TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
                Validate.notNull(treeRuleCodeStrategy, "系统未配置降维码策略");
                updateRuleCodeAllChildren(item.getSalesOrgCode(),
                        ruleCode + treeRuleCodeStrategy.generateByNum(SalesOrgCodeConstant.RULE_CODE_LENGTH, i + ONE),
                        (levelNum + ONE));
            }
        }
    }

    /**
     * 获取降维码
     *
     * @param parentCode
     * @return
     */
    private String getRuleCodeByParentCode(String parentCode) {
        String parentRuleCode = null;
        if (StringUtils.isNotEmpty(parentCode)) {
            SalesOrg parent = this.salesOrgRepository.findDetailsByCode(parentCode, TenantUtils.getTenantCode());
            Validate.notNull(parent, "上级销售组织不存在");
            parentRuleCode = parent.getRuleCode();
        }
        List<SalesOrg> childrenListByParentCode =
                this.salesOrgRepository.findChildrenListByParentCode(parentCode, TenantUtils.getTenantCode());
        List<TreeDto> childrenDto =
                Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(childrenListByParentCode, SalesOrg.class,
                        TreeDto.class, HashSet.class, ArrayList.class));
        // TODO 编码降维编码长度以前可以在系统设置中进行设置，该功能将在后续改造中实现，先设置为固定默认值
        TreeRuleCodeStrategy treeRuleCodeStrategy = treeRuleCodeStrategyHolder.getStrategy(null);
        return treeRuleCodeStrategy.generate(SalesOrgCodeConstant.RULE_CODE_LENGTH, parentRuleCode, childrenDto);
    }

    /**
     * 查询销售机构
     * @param pageable
     * @param paginationDto
     * @return {@link Page}<{@link SalesOrg}>
     */
    @Override
    public Page<SalesOrg> findSalesMechanism(Pageable pageable, SalesOrgPaginationDto paginationDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
        paginationDto.setSalesOrgLevel(SalesOrgLevelTypeEnum.MECHANISM.getCode());
        return this.salesOrgRepository.findBySalesOrgLevelType(pageable, paginationDto);
    }

    /**
     * 查询销售组
     * @param pageable
     * @param paginationDto
     * @return {@link Page}<{@link SalesOrg}>
     */
    @Override
    public Page<SalesOrg> findSalesGroup(Pageable pageable, SalesOrgPaginationDto paginationDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
        paginationDto.setSalesOrgLevel(SalesOrgLevelTypeEnum.GROUP.getCode());
        return this.salesOrgRepository.findBySalesOrgLevelType(pageable, paginationDto);
    }

    /**
     * 查询销售组织
     * @param pageable
     * @param paginationDto
     * @return {@link Page}<{@link SalesOrg}>
     */
    @Override
    public Page<SalesOrg> findSalesOrg(Pageable pageable, SalesOrgPaginationDto paginationDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
        return this.salesOrgRepository.findBySalesOrgLevelType(pageable, paginationDto);
    }
}
