package com.biz.crm.mdm.business.sales.org.local.service.impl;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.sales.org.local.entity.SalesOrgSubComOrg;
import com.biz.crm.mdm.business.sales.org.local.repository.SalesOrgSubComOrgRepository;
import com.biz.crm.mdm.business.sales.org.local.service.SalesOrgSubComOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.constant.SalesOrgCodeConstant;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.constants.MasterDataMdgConstants;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgCompanyAreaVo;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgSalesOfficeVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * @describe: 销售组织与分子公司组织对应关系 实现
 * @author: dutaotao
 * @version: v1.0.0
 * @date: 2022.10.28 16:10
 */
@Service("salesOrgSubComOrgVoService")
@Slf4j
public class SalesOrgSubComOrgVoServiceImpl implements SalesOrgSubComOrgVoService {

    @Autowired(required = false)
    private RedisMutexService redisMutexService;
    @Autowired(required = false)
    private SalesOrgSubComOrgRepository salesOrgSubComOrgRepository;
    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    /**
     * 批量拉取销售办公室关系主数据 MDG
     *
     * @param dto
     * @author dutaotao
     * @date 2022/12/25 15:04
     **/
    @Override
    public void pullSalesOfficeList(MasterDataMdgBaseDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }
        boolean lock = true;
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        if (StringUtils.isEmpty(dto.getDs())) {
            dto.setDs(lockKey);
            lockKey = dto.getDs();
        }
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            int maxPage = 0;
            if (BooleanEnum.TRUE.getCapital().equals(dto.getFullPullFlag())) {
                maxPage = this.salesOfficeCountPageMax(dto);
            } else {
                maxPage = Integer.parseInt(dto.getPageNum());
            }
            for (int pageNum = Integer.parseInt(dto.getPageNum()); pageNum <= maxPage; pageNum++) {
                dto.setPageNum(Integer.toString(pageNum));
                List<MasterDataMdgSalesOfficeVo> masterDataMdgSalesOfficeVos = new ArrayList<>();
                masterDataMdgSalesOfficeVos = masterDataMdgService.pullSalesOfficeList(dto);
                if (CollectionUtils.isEmpty(masterDataMdgSalesOfficeVos)) {
                    return;
                }
                // 数据校验
                List<SalesOrgSubComOrg> pullList = this.salesOfficeValidate(masterDataMdgSalesOfficeVos);
                List<String> salesOrgCodeList =
                        pullList.stream().map(SalesOrgSubComOrg::getSalesOrgCode).collect(Collectors.toList());
                if (CollectionUtils.isEmpty(salesOrgCodeList)) {
                    throw new RuntimeException("来源数据为空");
                }
                // 区分更新、新增操作
                List<SalesOrgSubComOrg> bySalesOrgCodes =
                        salesOrgSubComOrgRepository.findBySalesOrgCodeLists(salesOrgCodeList);
                if (CollectionUtils.isEmpty(bySalesOrgCodes)) {
                    this.saveOrUpdateBatc(pullList, null);
                    return;
                }
                // 区分更新、新增
                List<SalesOrgSubComOrg> saveList = new ArrayList<>();
                List<SalesOrgSubComOrg> updateList = new ArrayList<>();
                Map<String, SalesOrgSubComOrg> map = bySalesOrgCodes.stream().collect(
                        Collectors.toMap(SalesOrgSubComOrg::getSalesOrgCode, v -> v, (oldValue, newValue) -> newValue));
                pullList.forEach(pull -> {
                    if (map.containsKey(pull.getSalesOrgCode())) {
                        SalesOrgSubComOrg salesOrgSubComOrg = map.get(pull.getSalesOrgCode());
                        updateList.add(salesOrgSubComOrg);
                    } else {
                        saveList.add(pull);
                    }
                });
                this.saveOrUpdateBatc(saveList, updateList);
            }
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }

    /**
     * 数据校验 MDG公司地区和关系数据主数据
     *
     * @param masterDataMdgCompanyAreaVos
     * @return
     * @author dutaotao
     * @date 2022/12/6 21:37
     **/
    private List<SalesOrgSubComOrg> companyAreaValidate(List<MasterDataMdgCompanyAreaVo> masterDataMdgCompanyAreaVos,
                                                        List<MasterDataMdgCompanyAreaVo> failList) {
        Map<String, String> codeMap = new HashMap<>(100000);
        List<SalesOrgSubComOrg> pullList = new ArrayList<>();
        masterDataMdgCompanyAreaVos.forEach(masterDataMdgCompanyAreaVo -> {
            if (StringUtils.isEmpty(masterDataMdgCompanyAreaVo.getSource1Code())) {
                log.info("本次拉取数据：" + masterDataMdgCompanyAreaVos);
                throw new RuntimeException("分子公司编码不能为空，请检查！");
            }
            SalesOrgSubComOrg salesOrgSubComOrg = new SalesOrgSubComOrg();
            if (codeMap.containsKey(masterDataMdgCompanyAreaVo.getSource1Code())) {
                log.info("本次拉取数据" + masterDataMdgCompanyAreaVos);
                throw new RuntimeException("编码" + masterDataMdgCompanyAreaVo.getSource1Code() + "重复拉取，请检查！");
            }
            codeMap.put(masterDataMdgCompanyAreaVo.getSource1Code(), "");

            // 销售机构编码
            salesOrgSubComOrg.setSalesOrgCode(masterDataMdgCompanyAreaVo.getSource2Code());
            List<String> salesOrgName =
                    salesOrgVoService.findByMdgSalesOrgCode(masterDataMdgCompanyAreaVo.getSource2Code());
            if (!Objects.isNull(salesOrgName)) {
                salesOrgSubComOrg.setSalesOrgName(salesOrgName.get(0));
            }
            // 分子公司组织编码
            salesOrgSubComOrg.setSubComOrgCode(masterDataMdgCompanyAreaVo.getSource1Code());
            // 分子公司组织名称
            salesOrgSubComOrg.setSubComOrgName(masterDataMdgCompanyAreaVo.getSource1SystemDesc());
            salesOrgSubComOrg.setTenantCode(TenantUtils.getTenantCode());
            salesOrgSubComOrg.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            salesOrgSubComOrg.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            pullList.add(salesOrgSubComOrg);
        });
        // 编码去重
        return pullList;
    }

    @Override
    public void pullCompanyAreaList(MasterDataMdgBaseDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        boolean lock = true;
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            List<MasterDataMdgCompanyAreaVo> failList = new ArrayList<>();
            List<MasterDataMdgCompanyAreaVo> masterDataMdgCompanyAreaVos = new ArrayList<>();
            int maxPage = 0;
            if (BooleanEnum.TRUE.getCapital().equals(dto.getFullPullFlag())) {
                maxPage = this.countPageMax(dto);
            } else {
                maxPage = Integer.parseInt(dto.getPageNum());
            }
            for (int pageNum = Integer.parseInt(dto.getPageNum()); pageNum <= maxPage; pageNum++) {
                dto.setPageNum(Integer.toString(pageNum));
                masterDataMdgCompanyAreaVos = masterDataMdgService.pullCompanyAreaList(dto);
                if (CollectionUtils.isEmpty(masterDataMdgCompanyAreaVos)) {
                    return;
                }
                // 数据校验
                List<SalesOrgSubComOrg> pullList = this.companyAreaValidate(masterDataMdgCompanyAreaVos, failList);
                log.info("当前拉取进度：" + pageNum + "/" + maxPage + "，每页" + dto.getPageSize() + "条数据，拉取失败的数据共"
                        + failList.size() + "条，---------------------------->" + JSON.toJSONString(failList));
                failList.clear();
                List<String> salesOrgCodeList =
                        pullList.stream().map(SalesOrgSubComOrg::getSubComOrgCode).collect(Collectors.toList());
                if (CollectionUtils.isEmpty(salesOrgCodeList)) {
                    throw new RuntimeException("来源数据为空");
                }
                // 区分更新、新增操作
                List<SalesOrgSubComOrg> bySalesOrgCodes =
                        salesOrgSubComOrgRepository.findBySalesOrgCodeLists(salesOrgCodeList);
                if (CollectionUtils.isEmpty(bySalesOrgCodes)) {
                    this.saveOrUpdateBatc(pullList, null);
                    return;
                }
                // 区分更新、新增
                List<SalesOrgSubComOrg> saveList = new ArrayList<>();
                List<SalesOrgSubComOrg> updateList = new ArrayList<>();
                Map<String, SalesOrgSubComOrg> map = bySalesOrgCodes.stream().collect(
                        Collectors.toMap(SalesOrgSubComOrg::getSalesOrgCode, v -> v, (oldValue, newValue) -> newValue));
                pullList.forEach(pull -> {
                    if (map.containsKey(pull.getSalesOrgCode())) {
                        SalesOrgSubComOrg salesOrgSubComOrg = map.get(pull.getSalesOrgCode());
                        updateList.add(salesOrgSubComOrg);
                    } else {
                        saveList.add(pull);
                    }
                });
                this.saveOrUpdateBatc(saveList, updateList);
            }
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }

    /**
     * 根据客户编码查询分子公司编码
     *
     * @param erpCode
     * @return Boolean
     * @author huojia
     * @date 2022/12/5 20:32
     **/
    @Override
    public Boolean findByCustomerClass(String erpCode) {
        if (Objects.isNull(erpCode)) {
            return false;
        }
        List<SalesOrgSubComOrg> salesOrgSubComOrgList = this.salesOrgSubComOrgRepository.findByCustomerClass(erpCode);
        if (!CollectionUtils.isEmpty(salesOrgSubComOrgList)) {
            return true;
        }
        return false;
    }

    /**
     * 根据客户编码查询分子公司编码
     *
     * @param erpCodeList
     * @return Boolean
     * @author huojia
     * @date 2022/12/5 20:32
     **/
    @Override
    public Set<String> findErpCodeListByCustomerClass(List<String> erpCodeList) {
        if (CollectionUtil.isEmpty(erpCodeList)) {
            return Collections.emptySet();
        }
        List<SalesOrgSubComOrg> salesOrgSubComOrgList = this.salesOrgSubComOrgRepository.findErpCodeListByCustomerClass(erpCodeList);
        if (CollectionUtil.isEmpty(salesOrgSubComOrgList)) {
            return Collections.emptySet();
        }
        return salesOrgSubComOrgList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getSubComOrgCode()))
                .map(SalesOrgSubComOrg::getSubComOrgCode)
                .collect(Collectors.toSet());
    }

    /**
     * 批量新增数据（MDG用）
     *
     * @param saveList
     * @param updateList
     * @author dutaotao
     * @date 2023/1/3 17:38
     */
    @Transactional(rollbackFor = Exception.class)
    void saveOrUpdateBatc(List<SalesOrgSubComOrg> saveList, List<SalesOrgSubComOrg> updateList) {
        if (!CollectionUtils.isEmpty(saveList)) {
            this.salesOrgSubComOrgRepository.saveBatch(saveList);
        }
        if (!CollectionUtils.isEmpty(updateList)) {
            this.salesOrgSubComOrgRepository.updateBatchById(updateList);
        }
    }

    /**
     * 数据校验 MDG销售办公室关系主数据
     *
     * @param masterDataMdgSalesOfficeVos
     * @return java.util.List<com.biz.crm.mdm.business.sales.org.local.entity.SalesOrgSubComOrg>
     * @author 杜涛涛
     **/
    private List<SalesOrgSubComOrg> salesOfficeValidate(List<MasterDataMdgSalesOfficeVo> masterDataMdgSalesOfficeVos) {
        Map<String, String> codeMap = new HashMap<>(100000);
        List<SalesOrgSubComOrg> pullList = new ArrayList<>();
        masterDataMdgSalesOfficeVos.forEach(masterDataMdgSalesOfficeVo -> {
            if (StringUtils.isEmpty(masterDataMdgSalesOfficeVo.getZly1code())) {
                log.info("本次拉取数据：" + masterDataMdgSalesOfficeVos);
                throw new RuntimeException("分子公司编码不能为空，请检查！");
            }
            SalesOrgSubComOrg salesOrgSubComOrg = new SalesOrgSubComOrg();
            if (codeMap.containsKey(masterDataMdgSalesOfficeVo.getZly1code())) {
                log.info("本次拉取数据" + masterDataMdgSalesOfficeVos);
                throw new RuntimeException("编码" + masterDataMdgSalesOfficeVo.getZly1code() + "重复拉取，请检查！");
            }
            codeMap.put(masterDataMdgSalesOfficeVo.getMandt(), "");
            // 分子公司组织编码
            salesOrgSubComOrg.setSubComOrgCode(masterDataMdgSalesOfficeVo.getZly1code());
            // 分子公司组织名称
            salesOrgSubComOrg.setSubComOrgName(masterDataMdgSalesOfficeVo.getZlysy1text());
            // 销售机构编码
            salesOrgSubComOrg.setSalesOrgCode(masterDataMdgSalesOfficeVo.getZly2code());
            salesOrgSubComOrg.setTenantCode(TenantUtils.getTenantCode());
            salesOrgSubComOrg.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            salesOrgSubComOrg.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            pullList.add(salesOrgSubComOrg);
        });
        // 编码去重
        return pullList;
    }

    /**
     * MDG销售办公室关系主数据按年月日解锁MDG
     *
     * @param yearMonthDay
     * @return boolean
     * @author 杜涛涛
     * @date 2022/12/6 21:52
     **/
    private boolean lock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取销售办公室关系主数据解锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(SalesOrgCodeConstant.SALES_ORG_SUB_COM_ORG_LOCK + yearMonthDay,
                TimeUnit.HOURS, 12);
    }

    /**
     * MDG销售办公室关系主数据按年月日解锁MDG
     *
     * @param yearMonthDay
     * @author 杜涛涛
     * @date 2022/12/6 21:52
     **/
    private void unLock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取销售办公室关系主数据解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(SalesOrgCodeConstant.SALES_ORG_SUB_COM_ORG_LOCK + yearMonthDay);
    }

    /**
     * 最大分页数
     *
     * @param dto
     * @return int
     * @author huojia
     * @date 2023/1/8 1:33
     **/
    private int countPageMax(MasterDataMdgBaseDto dto) {
        Integer totalNum = masterDataMdgService.countList(MasterDataMdgConstants.MDG_COMPANY_AREA_TPM);
        int maxPage = 1;
        int sizeInt = Integer.parseInt(dto.getPageSize());
        if (totalNum > sizeInt) {
            maxPage = totalNum % sizeInt > 0 ? (totalNum / sizeInt) + 1 : totalNum / sizeInt;
        }
        return maxPage;
    }

    /**
     * 最大分页数
     *
     * @param dto
     * @return int
     * @author huojia
     * @date 2023/1/8 1:33
     **/
    private int salesOfficeCountPageMax(MasterDataMdgBaseDto dto) {
        Integer totalNum = masterDataMdgService.countList(MasterDataMdgConstants.MDG_SALES_OFFICE_TPM);
        int maxPage = 1;
        int sizeInt = Integer.parseInt(dto.getPageSize());
        if (totalNum > sizeInt) {
            maxPage = totalNum % sizeInt > 0 ? (totalNum / sizeInt) + 1 : totalNum / sizeInt;
        }
        return maxPage;
    }
}
