package com.biz.crm.mdm.business.sales.org.local.service.impl;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.mdm.business.sales.org.local.entity.SalesOrg;
import com.biz.crm.mdm.business.sales.org.local.repository.SalesOrgRepository;
import com.biz.crm.mdm.business.sales.org.local.service.SalesOrgService;
import com.biz.crm.mdm.business.sales.org.sdk.constant.SalesOrgCodeConstant;
import com.biz.crm.mdm.business.sales.org.sdk.dto.RelateSalesOrgCodeQueryDto;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgPaginationDto;
import com.biz.crm.mdm.business.sales.org.sdk.dto.SalesOrgQueryDto;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 销售组织vo serivce 实现
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.13 19:47
 */
@Service
public class SalesOrgVoServiceImpl implements SalesOrgVoService {

    @Autowired(required = false)
    private SalesOrgService salesOrgService;
    @Autowired(required = false)
    private SalesOrgRepository salesOrgRepository;
    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;

    @Override
    public Page<SalesOrgVo> findByConditions(Pageable pageable, SalesOrgPaginationDto paginationDto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
        if (paginationDto == null) {
            paginationDto = new SalesOrgPaginationDto();
        }
        Page<SalesOrg> page = this.salesOrgService.findByConditions(pageable, paginationDto);
        Page<SalesOrgVo> pageVo = new Page<>();
        if (Objects.isNull(page)) {
            return null;
        }
        if (CollectionUtils.isNotEmpty(page.getRecords())) {
            List<SalesOrgVo> vos =
                    (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(page.getRecords(), SalesOrg.class,
                            SalesOrgVo.class, HashSet.class, ArrayList.class);
            this.build(vos);
            pageVo.setRecords(vos);
        }
        pageVo.setCurrent(page.getCurrent());
        pageVo.setPages(page.getPages());
        pageVo.setTotal(page.getTotal());
        return pageVo;
    }

    @Override
    public SalesOrgVo findDetailsById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        SalesOrg org = this.salesOrgService.findDetailsById(id);
        if (Objects.isNull(org)) {
            return null;
        }
        SalesOrgVo vo =
                this.nebulaToolkitService.copyObjectByWhiteList(org, SalesOrgVo.class, HashSet.class, ArrayList.class);
        this.build(vo);
        return vo;
    }

    @Override
    public SalesOrgVo findBySalesOrgCode(String orgCode) {
        if (StringUtils.isBlank(orgCode)) {
            return null;
        }
        SalesOrg org = this.salesOrgService.findByOrgCode(orgCode);
        if (Objects.isNull(org)) {
            return null;
        }
        SalesOrgVo vo =
                this.nebulaToolkitService.copyObjectByWhiteList(org, SalesOrgVo.class, HashSet.class, ArrayList.class);
        this.build(vo);
        return vo;
    }

    @Override
    public List<SalesOrgVo> findDetailsByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return null;
        }
        List<SalesOrg> orgs = this.salesOrgService.findDetailsByIds(ids);
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }
        List<SalesOrgVo> vos = (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(orgs,
                SalesOrg.class, SalesOrgVo.class, HashSet.class, ArrayList.class);
        this.build(vos);
        return vos;
    }

    @Override
    public List<SalesOrgVo> findAllParentBySalesOrgCode(String orgCode) {
        if (StringUtils.isNotEmpty(orgCode)) {
            List<SalesOrg> orgs = salesOrgService.findAllParentByOrgCode(orgCode);
            if (CollectionUtils.isEmpty(orgs)) {
                return null;
            }
            List<SalesOrgVo> vos = (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(orgs,
                    SalesOrg.class, SalesOrgVo.class, HashSet.class, ArrayList.class);
            this.build(vos);
            return vos;
        }
        return null;
    }

    @Override
    public List<SalesOrgVo> findAllParentBySalesOrgCodes(List<String> orgCodes) {
        if (CollectionUtils.isNotEmpty(orgCodes)) {
            List<SalesOrg> orgs = salesOrgService.findAllParentByOrgCodes(orgCodes);
            if (CollectionUtils.isEmpty(orgs)) {
                return null;
            }
            List<SalesOrgVo> vos = (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(orgs,
                    SalesOrg.class, SalesOrgVo.class, HashSet.class, ArrayList.class);
            this.build(vos);
            return vos;
        }
        return null;
    }

    @Override
    public Set<String> findBySalesOrgQueryDto(SalesOrgQueryDto dto) {
        if (Objects.isNull(dto) || StringUtils.isAllBlank(dto.getSalesOrgName())) {
            return Sets.newHashSet();
        }
        return this.salesOrgRepository.findByOrgQueryDto(dto);
    }

    @Override
    public List<SalesOrgVo> findBySalesOrgCodes(List<String> orgCodes) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return null;
        }
        List<SalesOrg> orgs = this.salesOrgService.findByOrgCodes(orgCodes);
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }
        List<SalesOrgVo> vos = (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(orgs,
                SalesOrg.class, SalesOrgVo.class, HashSet.class, ArrayList.class);
        this.build(vos);
        return vos;
    }

    @Override
    public List<SalesOrgVo> findBySalesOrgCodesPost(List<String> orgCodes) {
        return null;
    }

    @Override
    public List<SalesOrgVo> findAllChildrenById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        List<SalesOrg> orgs = this.salesOrgService.findAllChildrenById(id);
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }
        List<SalesOrgVo> vos = (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(orgs,
                SalesOrg.class, SalesOrgVo.class, HashSet.class, ArrayList.class);
        this.build(vos);
        return vos;
    }

    @Override
    public List<SalesOrgVo> findAllChildrenBySalesOrgCode(String orgCode) {
        if (StringUtils.isBlank(orgCode)) {
            return null;
        }
        List<SalesOrg> orgs = this.salesOrgService.findAllChildrenByOrgCode(orgCode);
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }
        List<SalesOrgVo> vos = (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(orgs,
                SalesOrg.class, SalesOrgVo.class, HashSet.class, ArrayList.class);
        this.build(vos);
        return vos;
    }

    @Override
    public List<SalesOrgVo> findAllChildrenBySalesOrgCodes(List<String> orgCodes) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return null;
        }
        List<SalesOrg> orgs = this.salesOrgService.findAllChildrenBySalesOrgCodes(orgCodes);
        if (CollectionUtils.isEmpty(orgs)) {
            return null;
        }
        return (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(orgs, SalesOrg.class,
                SalesOrgVo.class, HashSet.class, ArrayList.class);
    }

    @Override
    public Map<String, String> findByRelateSalesOrgCodeQueryDto(RelateSalesOrgCodeQueryDto dto) {
        if (Objects.isNull(dto) || CollectionUtils.isEmpty(dto.getSalesOrgCodeSet())) {
            return Maps.newHashMap();
        }
        dto.setSearchType(Optional.ofNullable(dto.getSearchType()).orElse(0));
        List<SalesOrg> orgList = this.salesOrgService.findByOrgCodes(Lists.newArrayList(dto.getSalesOrgCodeSet()));
        if (CollectionUtils.isEmpty(orgList)) {
            return Maps.newHashMap();
        }
        if (dto.getSearchType() == 0) {
            return orgList.stream().filter(a -> StringUtils.isNoneBlank(a.getSalesOrgCode(), a.getRuleCode()))
                    .collect(Collectors.toMap(SalesOrg::getSalesOrgCode, SalesOrg::getRuleCode, (a, b) -> a));
        }
        List<String> ruleCodeList = orgList.stream().filter(a -> StringUtils.isNotBlank(a.getRuleCode()))
                .map(SalesOrg::getRuleCode).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(ruleCodeList)) {
            return Maps.newHashMap();
        }

        TreeRuleCodeStrategy treeRuleCodeStrategy = treeRuleCodeStrategyHolder.getStrategy(null);
        if (dto.getSearchType() > 0) {
            Set<String> ruleCodeSet = treeRuleCodeStrategy
                    .findParentRuleCodesByRuleCodesExcludeSelf(SalesOrgCodeConstant.RULE_CODE_LENGTH, ruleCodeList);
            if (CollectionUtils.isEmpty(ruleCodeSet)) {
                ruleCodeSet = Sets.newHashSet();
            }
            ruleCodeSet.addAll(ruleCodeList);
            if (CollectionUtils.isNotEmpty(ruleCodeSet)) {
                final List<SalesOrg> cur =
                        this.salesOrgService.findByRuleCodesAndEnableStatus(Lists.newLinkedList(ruleCodeSet), null);
                if (CollectionUtils.isEmpty(cur)) {
                    return Maps.newHashMap();
                }
                return cur.stream()
                        .collect(Collectors.toMap(SalesOrg::getSalesOrgCode, SalesOrg::getRuleCode, (a, b) -> a));
            }
        } else {
            final List<SalesOrg> cur = this.salesOrgService.findChildrenByRuleCode(ruleCodeList);
            if (CollectionUtils.isEmpty(cur)) {
                return Maps.newHashMap();
            }
            return cur.stream()
                    .collect(Collectors.toMap(SalesOrg::getSalesOrgCode, SalesOrg::getRuleCode, (a, b) -> a));
        }
        return Maps.newHashMap();
    }

    /**
     * 查询指定salesOrgLevel 下的所有销售组织编码和名称(MDG)
     *
     * @param salesOrgLevel
     * @author 杜涛涛
     * @date 2022/12/5 20:32
     **/
    @Override
    public List<SalesOrgVo> findAllSalesOrgLevelBySalesOrg(String salesOrgLevel,String salesOrgName) {
        if (StringUtils.isEmpty(salesOrgLevel)) {
            throw new RuntimeException("组织层级不能为空！");
        }
        List<SalesOrg> salesOrg = salesOrgRepository.findAllSalesOrgLevelBySalesOrg(salesOrgLevel,salesOrgName);
        if (ObjectUtils.isEmpty(salesOrg)) {
            throw new RuntimeException("查询数据不不存在！");
        }
        return (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByBlankList(salesOrg,
                SalesOrg.class, SalesOrgVo.class, HashSet.class, ArrayList.class);
    }


    /**
     * 根据所属销售机构编码查询所属销售机构名称和所属销售机构部门名称和销售组名称
     *
     * @param salesAgencyCode
     * @return
     */
    @Override
    public List<String> findBySalesOrgName(String salesAgencyCode) {
        Validate.notEmpty(salesAgencyCode, "所属销售机构编码不能为空");
        List<SalesOrg> salesOrg = this.salesOrgRepository.findBySalesAgencyCode(salesAgencyCode);
        if (ObjectUtils.isEmpty(salesOrg)) {
            return null;
        }
        return salesOrg.stream().map(SalesOrg::getSalesOrgName).collect(Collectors.toList());
    }


    @Override
    public List<SalesOrgVo> findByErpCodeList(List<String> salesOrgErpCodeList) {
        List<SalesOrg> entityList = this.salesOrgRepository.findByErpCodeList(salesOrgErpCodeList);
        return (List<SalesOrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, SalesOrg.class, SalesOrgVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 根据销售编码查询销售名称
     *
     * @param source2Code
     * @return
     */
    @Override
    public List<String> findByMdgSalesOrgCode(String source2Code) {
        Validate.notEmpty(source2Code, "销售编码来源不能为空");
        List<SalesOrg> customerEntity = this.salesOrgRepository.findByMdgSalesOrgCode(source2Code);
        if (ObjectUtils.isEmpty(customerEntity)) {
            return null;
        }
        return customerEntity.stream().map(SalesOrg::getSalesOrgName).collect(Collectors.toList());
    }

    /**
     * 根据erp编码查询销售组织全部信息
     *
     * @param erpCodes erp编码列表
     * @return Map<String, SalesOrgVo>
     */
    @Override
    public Map<String, SalesOrgVo> getSalesMapByErpCodes(List<List<String>> erpCodes) {
        Map<String, SalesOrgVo> map = new HashMap<>();
        if (CollectionUtils.isEmpty(erpCodes)) {
            return map;
        }
        List<SalesOrgVo> list = this.salesOrgRepository.getSalesMapByErpCodes(erpCodes, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(list)) {
            return map;
        }
        return list.stream().collect(Collectors.toMap(SalesOrgVo::getErpCode, Function.identity()));
    }

    @Override
    public Map<String, SalesOrgVo> getSalesMapByTpmCodes(List<String> tpmCodes) {
        Map<String, SalesOrgVo> map = new HashMap<>();
        if (CollectionUtils.isEmpty(tpmCodes)) {
            return map;
        }
        List<List<String>> codes = Lists.partition(tpmCodes, 500);
        List<SalesOrgVo> list = this.salesOrgRepository.getSalesMapByTpmCodes(codes, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(list)) {
            return map;
        }
        return list.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, Function.identity()));
    }

    /**
     * @param salesOrgCodes sap编码
     * @return
     */
    @Override
    public Map<String, String> findSapCodesBySalesOrgCodes(List<String> salesOrgCodes) {
        if (CollectionUtils.isEmpty(salesOrgCodes)) {
            return Maps.newHashMap();
        }
        return salesOrgService.findSapCodesByOrgCodes(salesOrgCodes);
    }

    /**
     * 根据销售组织编码集合查询对应名称
     *
     * @param salesOrgCodes 销售组织编码
     * @return Map<String, String>
     */
    @Override
    public Map<String, String> findNameBySalesOrgCodes(List<List<String>> salesOrgCodes) {
        Map<String, String> map = new HashMap<>();
        if (CollectionUtils.isEmpty(salesOrgCodes)) {
            return map;
        }
        List<SalesOrgVo> list = this.salesOrgRepository.findNameBySalesOrgCodes(salesOrgCodes, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(list)) {
            return map;
        }
        return list.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, SalesOrgVo::getSalesOrgName));
    }


    /**
     * 构建返回字段
     *
     * @param vo
     */
    private void build(SalesOrgVo vo) {
        SalesOrgVo parent = vo.getParent();
        if (parent != null) {
            vo.setParentName(parent.getSalesOrgName());
        }
    }

    /**
     * 构建返回字段
     *
     * @param vos
     */
    private void build(List<SalesOrgVo> vos) {
        vos = vos.stream().map(item -> {
            this.build(item);
            return item;
        }).collect(Collectors.toList());
    }

    /**
     * erp编码查询名称
     * @param erpCodes
     * @return {@link Map}<{@link String}, {@link String}>
     */
    @Override
    public Map<String, String> findNameByErpCodes(List<String> erpCodes) {
        return this.salesOrgRepository.findNameByErpCodes(erpCodes);
    }

    /**
     * 根据编码或者名称查询
     * @param codeOrName
     * @return {@link List}<{@link SalesOrgVo}>
     */
    @Override
    public List<SalesOrgVo> findByCodeOrName(List<String> codeOrName) {
        return this.salesOrgRepository.findByCodeOrName(codeOrName);
    }

    /**
     * 业务单元编辑时，同步更新销售组织上的业务单元
     *
     * @param businessUnitCode 业务单元
     * @param salesOrgCodes    销售组织编码
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void editSalesOrgBusinessUnit(String businessUnitCode, List<String> salesOrgCodes) {
        if(StringUtils.isBlank(businessUnitCode) || CollectionUtils.isEmpty(salesOrgCodes)){
            return;
        }
        for (String salesOrgCode : salesOrgCodes) {
            this.salesOrgRepository.lambdaUpdate().set(SalesOrg::getBusinessUnitCode,businessUnitCode)
                    .eq(SalesOrg::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                    .eq(SalesOrg::getTenantCode,TenantUtils.getTenantCode())
                    .likeRight(SalesOrg::getSalesOrgCode,salesOrgCode)
                    .update();
        }
    }
}
