package com.biz.crm.mdm.business.supplier.local.controller;

import java.util.List;

import com.biz.crm.mdm.business.supplier.sdk.dto.SupplierSelectDto;
import com.biz.crm.mdm.business.supplier.sdk.vo.SupplierBankVo;
import com.biz.crm.mn.common.base.dto.CommonSelectDto;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.supplier.local.service.SupplierService;
import com.biz.crm.mdm.business.supplier.sdk.dto.SupplierDto;
import com.biz.crm.mdm.business.supplier.sdk.service.SupplierVoService;
import com.biz.crm.mdm.business.supplier.sdk.vo.SupplierVo;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

/**
 * 供应商管理表(Supplier)表控制层
 *
 * @author makejava
 * @date 2022-11-23 15:26:25
 */
@Slf4j
@Api(tags = "供应商管理表")
@RestController
@RequestMapping(value = {"/v1/supplier/supplier"})
public class SupplierController {

    @Autowired(required = false)
    private SupplierService supplierService;

    @Autowired(required = false)
    private SupplierVoService supplierVoService;

    /**
     * 分页条件查询
     *
     * @param pageable      分页参数
     * @param paginationDto 查询条件
     * @return 供应商列表
     */
    @ApiOperation(value = "查询分页列表")
    @GetMapping(value = {"/findByConditions"})
    public Result<Page<SupplierVo>> findByConditions(@PageableDefault(50) Pageable pageable,
                                                     SupplierDto paginationDto) {
        try {
            Page<SupplierVo> result = this.supplierVoService.findByConditions(pageable, paginationDto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * feign分页条件查询
     *
     * @param pageable      分页参数
     * @param paginationDto 查询条件
     * @return 供应商列表
     */
    @ApiOperation(value = "查询分页列表")
    @PostMapping(value = {"findSupplierByConditions"})
    Result<Page<SupplierVo>> findSupplierByConditions(@RequestBody SupplierDto paginationDto) {
        try {
            Pageable pageable = PageRequest.of(paginationDto.getPage(), paginationDto.getSize());
            Page<SupplierVo> result = this.supplierVoService.findByConditions(pageable, paginationDto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量拉取 MDG 供应商数据
     *
     * @param dto 请求参数
     * @return 供应商列表
     */
    @ApiOperation(value = "批量拉取 MDG 供应商数据")
    @PostMapping(value = {"/pullSupplierList"})
    public Result<?> pullSupplierList(@RequestBody MasterDataMdgBaseDto dto) {
        try {
            this.supplierService.pullSupplierList(dto);
            return Result.ok("批量拉取 MDG 供应商数据成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量拉取 MDG 供应商银行数据
     *
     * @param dto 请求参数
     * @return 供应商列表
     */
    @ApiOperation(value = "批量拉取 MDG 供应商银行数据")
    @PostMapping(value = {"/pullSupplierBankList"})
    public Result<?> pullSupplierBankList(@RequestBody MasterDataMdgBaseDto dto) {
        try {
            this.supplierService.pullSupplierBankList(dto);
            return Result.ok("批量拉取 MDG 供应商数据成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页条件查询
     *
     * @param paginationDto 查询条件
     * @return 供应商列表
     */
    @ApiOperation(value = "条件查询列表")
    @PostMapping(value = {"/findByConditions"})
    public Result<List<SupplierVo>> findByConditions(@RequestBody SupplierDto paginationDto) {
        try {
            List<SupplierVo> result = this.supplierVoService.findByConditions(paginationDto);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据供应商编码查询供应商
     *
     * @param supplierCodes
     * @return
     */
    @ApiOperation(value = "条件查询列表")
    @PostMapping(value = {"/findBySupplierCodes"})
    public Result<List<SupplierVo>> findBySupplierCodes(@RequestBody List<String> supplierCodes) {
        try {
            List<SupplierVo> result = this.supplierVoService.findBySupplierCodes(supplierCodes);
            return Result.ok(result);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 按id查询详情供应商
     *
     * @param pageable 分页参数
     * @param supplierDto 请求参数
     * @return 供应商VO
     */
    @ApiOperation(value = "详情")
    @GetMapping(value = {"/findDetailById"})
    public Result<Page<SupplierBankVo>> findDetailById(@PageableDefault(50) Pageable pageable,
                                                       SupplierDto supplierDto) {
        try {
            Page<SupplierBankVo> page = this.supplierVoService.findDetailById(pageable, supplierDto);
            return Result.ok(page);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 按code查询详情供应商
     * @param pageable
     * @param supplierDto
     * @return
     */
    @ApiOperation(value = "按code查询详情供应商")
    @GetMapping(value = {"/findDetailByCode"})
    public Result<Page<SupplierBankVo>> findDetailByCode(@PageableDefault(50) Pageable pageable,
                                                       SupplierDto supplierDto) {
        try {
            Page<SupplierBankVo> page = this.supplierVoService.findDetailByCode(pageable, supplierDto);
            return Result.ok(page);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 创建供应商
     *
     * @param supplierVo 供应商VO
     * @return 创建成功的供应商VO
     */
    @ApiOperation(value = "创建")
    @PostMapping(value = "")
    public Result<SupplierVo> create(@RequestBody SupplierVo supplierVo) {
        try {
            SupplierVo current = this.supplierVoService.create(supplierVo);
            return Result.ok(current);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 更新供应商信息
     *
     * @param supplierVo new供应商VO
     * @return 更新结果
     */
    @ApiOperation(value = "更新")
    @PatchMapping(value = "")
    public Result<SupplierVo> update(@RequestBody SupplierVo supplierVo) {
        try {
            SupplierVo current = this.supplierVoService.update(supplierVo);
            return Result.ok(current);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 按id集合启用
     *
     * @param ids 主键主键列表
     */
    @ApiOperation(value = "启用")
    @PatchMapping("/enable")
    public Result<?> enable(@RequestBody List<String> ids) {
        try {
            this.supplierVoService.enableBatch(ids);
            return Result.ok("启用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 按id集合禁用
     *
     * @param ids 禁用主键列表
     */
    @ApiOperation(value = "禁用")
    @PatchMapping("/disable")
    public Result<?> disable(@RequestBody List<String> ids) {
        try {
            this.supplierVoService.disableBatch(ids);
            return Result.ok("禁用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 逻辑删除
     *
     * @param ids 删除主键列表
     */
    @ApiOperation(value = "逻辑删除")
    @DeleteMapping("/delete")
    public Result<?> delete(@RequestParam("ids") List<String> ids) {
        try {
            this.supplierVoService.updateDelFlagByIds(ids);
            return Result.ok("删除成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 供应商下拉
     *
     * @param dto 终端渠道查询参数
     */
    @ApiOperation(value = "供应商下拉")
    @PostMapping("findSupplierSelectList")
    public Result<List<CommonSelectVo>> findSupplierSelectList(@ApiParam(name = "CommonSelectDto", value = "查询参数") @RequestBody CommonSelectDto dto) {
        try {
            List<CommonSelectVo> list = supplierService.findSupplierSelectList(dto);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 供应商下拉(可排除部分数据)
     *
     * @param dto 查询参数
     * @return Result<List<CommonSelectVo>>
     */
    @ApiOperation(value = "供应商下拉(可排除部分数据)")
    @PostMapping("findSupplierSelectListByOut")
    public Result<List<CommonSelectVo>> findSupplierSelectListByOut(@ApiParam(name = "dto", value = "查询参数") @RequestBody SupplierSelectDto dto) {
        try {
            List<CommonSelectVo> list = supplierService.findSupplierSelectListByOut(dto);
            return Result.ok(list);
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

}
