package com.biz.crm.mdm.business.supplier.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.supplier.local.entity.SupplierBankEntity;
import com.biz.crm.mdm.business.supplier.local.entity.SupplierEntity;
import com.biz.crm.mdm.business.supplier.local.repository.SupplierBankRepository;
import com.biz.crm.mdm.business.supplier.local.repository.SupplierRepository;
import com.biz.crm.mdm.business.supplier.local.service.SupplierService;
import com.biz.crm.mdm.business.supplier.sdk.constants.SupplierConstant;
import com.biz.crm.mdm.business.supplier.sdk.dto.SupplierSelectDto;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import com.biz.crm.mn.common.base.dto.CommonSelectDto;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgSupplierBankVo;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgSupplierVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年12月05日 21:39
 */
@Slf4j
@Service
public class SupplierServiceImpl implements SupplierService {

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;

    @Autowired(required = false)
    private SupplierRepository supplierRepository;

    @Autowired(required = false)
    private SupplierBankRepository supplierBankRepository;

    /**
     * 批量拉取 MDG 供应商数据
     *
     * @param dto
     * @author huojia
     * @date 2022/12/5 21:58
     **/
    @Override
    public void pullSupplierList(MasterDataMdgBaseDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }
        boolean lock = true;
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        if (!StringUtils.isEmpty(dto.getDs())) {
            dto.setDs(lockKey);
        }
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            List<MasterDataMdgSupplierVo> masterDataMdgSupplierVos = masterDataMdgService.pullSupplierList(dto);
            if (CollectionUtils.isEmpty(masterDataMdgSupplierVos)) {
                return;
            }
            // 数据校验
            List<SupplierEntity> pullList = this.supplierValidate(masterDataMdgSupplierVos);
            List<String> supplierCodeList = pullList.stream()
                    .map(SupplierEntity::getSupplierCode).collect(Collectors.toList());
            // 区分更新、新增操作
            List<SupplierEntity> bySupplierCodes = supplierRepository.findBySupplierCodes(supplierCodeList);
            if (CollectionUtils.isEmpty(bySupplierCodes)) {
                this.saveOrUpdateBatch(pullList, null);
                return;
            }
            // 区分更新、新增
            List<SupplierEntity> saveList = new ArrayList<>();
            List<SupplierEntity> updateList = new ArrayList<>();
            Map<String, SupplierEntity> map = bySupplierCodes.stream().collect(Collectors.toMap(SupplierEntity::getSupplierCode, Function.identity()));
            pullList.forEach(pull -> {
                if (map.containsKey(pull.getSupplierCode())) {
                    SupplierEntity supplierEntity = map.get(pull.getSupplierCode());
                    supplierEntity.setSupplierName(pull.getSupplierName());
                    supplierEntity.setContactPhone(pull.getContactPhone());
                    updateList.add(supplierEntity);
                } else {
                    saveList.add(pull);
                }
            });
            this.saveOrUpdateBatch(saveList, updateList);
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }
    /**
     * 批量新增数据（MDG用）
     * @param saveList
     * @param updateList
     * @author huojia
     * @date 2023/1/3 17:38
     **/
    @Transactional(rollbackFor = Exception.class)
    void saveOrUpdateBatch(List<SupplierEntity> saveList, List<SupplierEntity> updateList) {
        if (!CollectionUtils.isEmpty(saveList)) {
            this.supplierRepository.saveBatch(saveList);
        }
        if (!CollectionUtils.isEmpty(updateList)) {
            this.supplierRepository.updateBatchById(updateList);
        }
    }

    /**
     * 数据校验
     *
     * @param masterDataMdgSupplierVos
     * @return java.util.List<com.biz.crm.mdm.business.supplier.local.entity.SupplierEntity>
     * @author huojia
     * @date 2022/12/6 21:37
     **/
    private List<SupplierEntity> supplierValidate(List<MasterDataMdgSupplierVo> masterDataMdgSupplierVos) {
        // 编码去重
        Map<String, List<MasterDataMdgSupplierVo>> codeMap = new HashMap<>(1000000);
        List<SupplierEntity> pullList = new ArrayList<>();
        masterDataMdgSupplierVos.forEach(masterDataMdgSupplierVo -> {
            if (StringUtils.isEmpty(masterDataMdgSupplierVo.getLifnr())) {
                throw new RuntimeException(JSON.toJSONString("供应商编码为空：" + JSON.toJSONString(masterDataMdgSupplierVo)));
            }
            SupplierEntity supplierEntity = new SupplierEntity();
            List<MasterDataMdgSupplierVo> supplierVos = Optional.ofNullable(codeMap.get(masterDataMdgSupplierVo.getLifnr())).orElse(new ArrayList<>());
            supplierVos.add(masterDataMdgSupplierVo);
            if (supplierVos.size() > 1) {
                throw new RuntimeException("数据：" + JSON.toJSONString(supplierVos) + "重复拉取，请检查！");
            }
            codeMap.put(masterDataMdgSupplierVo.getLifnr(), supplierVos);
            supplierEntity.setSupplierCode(masterDataMdgSupplierVo.getLifnr());
            supplierEntity.setSupplierName(masterDataMdgSupplierVo.getName1());
            supplierEntity.setContactPhone(masterDataMdgSupplierVo.getTelf1());
            supplierEntity.setTenantCode(TenantUtils.getTenantCode());
            supplierEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            pullList.add(supplierEntity);
        });
        return pullList;
    }

    /**
     * 批量拉取供应商银行数据
     *
     * @param dto
     * @author huojia
     * @date 2022/12/6 22:04
     **/
    @Override
    public void pullSupplierBankList(MasterDataMdgBaseDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }
        boolean lock = true;
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        if (!StringUtils.isEmpty(dto.getDs())) {
            dto.setDs(lockKey);
        }
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            List<MasterDataMdgSupplierBankVo> masterDataMdgSupplierBankVos = masterDataMdgService.pullSupplierBankList(dto);
            if (CollectionUtils.isEmpty(masterDataMdgSupplierBankVos)) {
                return;
            }
            List<String> supplierCodeList = masterDataMdgSupplierBankVos.stream()
                    .map(MasterDataMdgSupplierBankVo::getLifnr)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
            List<SupplierEntity> supplierEntities = supplierRepository.findBySupplierCodes(supplierCodeList);
            if (CollectionUtils.isEmpty(supplierEntities)) {
                return;
            }
            List<SupplierBankEntity> pullList = new ArrayList<>();
            masterDataMdgSupplierBankVos.forEach(masterDataMdgSupplierBankVo -> {
                if (StringUtils.isEmpty(masterDataMdgSupplierBankVo.getLifnr())) {
                    throw new RuntimeException("供应商编码为空：" + JSON.toJSONString(masterDataMdgSupplierBankVo));
                }
                SupplierBankEntity supplierBankEntity = new SupplierBankEntity();
                // 银行数据维度不确定
                /*List<MasterDataMdgSupplierBankVo> bankVos = Optional.ofNullable(codeMap.get(masterDataMdgSupplierBankVo.getLifnr())).orElse(new ArrayList<>());
                bankVos.add(masterDataMdgSupplierBankVo);
                if (bankVos.size() > 1) {
                    log.info("数据：" + JSON.toJSONString(bankVos) + "重复拉取，请检查！");
                }
                codeMap.put(masterDataMdgSupplierBankVo.getLifnr(), bankVos);*/
                /*if (!supplierMap.containsKey(masterDataMdgSupplierBankVo.getLifnr())) {
                    throw new RuntimeException("供应商不存在：" + JSON.toJSONString(masterDataMdgSupplierBankVo));
                }*/
                supplierBankEntity.setSupplierCode(masterDataMdgSupplierBankVo.getLifnr());
                supplierBankEntity.setBankCard(masterDataMdgSupplierBankVo.getBankn());
                supplierBankEntity.setBankAccount(masterDataMdgSupplierBankVo.getBanka());
                supplierBankEntity.setUnionPayBankCard(masterDataMdgSupplierBankVo.getBankl());
                supplierBankEntity.setContactName(masterDataMdgSupplierBankVo.getKoinh());
                supplierBankEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                supplierBankEntity.setTenantCode(TenantUtils.getTenantCode());
                supplierBankEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                pullList.add(supplierBankEntity);
            });
            this.saveOrUpdateMdgBatch(supplierCodeList, pullList);
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }

    /**
     * 批量操作数据
     *
     * @param supplierCodeList
     * @param pullList
     * @author huojia
     * @date 2023/1/3 17:41
     **/
    @Transactional(rollbackFor = Exception.class)
    void saveOrUpdateMdgBatch(List<String> supplierCodeList, List<SupplierBankEntity> pullList) {
        if (!CollectionUtils.isEmpty(supplierCodeList)) {
            supplierBankRepository.delBySupplierCodeList(supplierCodeList);
        }
        if (!CollectionUtils.isEmpty(pullList)) {
            supplierBankRepository.saveBatch(pullList);
        }
    }

    /**
     * 按年月日加锁
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    public boolean lock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取供应商加锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(SupplierConstant.SUPPLIER_LOCK + yearMonthDay, TimeUnit.HOURS, 12);
    }

    /**
     * 按年月日解锁
     *
     * @param yearMonthDay
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    public void unLock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取供应商解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(SupplierConstant.SUPPLIER_LOCK + yearMonthDay);
    }

    /**
     * 供应商下拉
     *
     * @param dto 终端渠道查询参数
     */
    @Override
    public List<CommonSelectVo> findSupplierSelectList(CommonSelectDto dto) {
        return supplierRepository.findSupplierSelectList(dto);
    }

    /**
     * 供应商下拉(可排除部分数据)
     *
     * @param dto 查询参数
     * @return List<CommonSelectVo>
     */
    @Override
    public List<CommonSelectVo> findSupplierSelectListByOut(SupplierSelectDto dto) {
        if (Objects.isNull(dto)) {
            dto = new SupplierSelectDto();
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.supplierRepository.findSupplierSelectListByOut(dto);
    }
}
