package com.biz.crm.mdm.business.supplier.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.supplier.local.entity.SupplierBankEntity;
import com.biz.crm.mdm.business.supplier.local.entity.SupplierEntity;
import com.biz.crm.mdm.business.supplier.local.repository.SupplierBankRepository;
import com.biz.crm.mdm.business.supplier.local.repository.SupplierRepository;
import com.biz.crm.mdm.business.supplier.sdk.dto.SupplierDto;
import com.biz.crm.mdm.business.supplier.sdk.dto.SupplierLogEventDto;
import com.biz.crm.mdm.business.supplier.sdk.event.SupplierEventListener;
import com.biz.crm.mdm.business.supplier.sdk.event.log.SupplierLogEventListener;
import com.biz.crm.mdm.business.supplier.sdk.service.SupplierVoService;
import com.biz.crm.mdm.business.supplier.sdk.vo.SupplierBankVo;
import com.biz.crm.mdm.business.supplier.sdk.vo.SupplierVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * 供应商管理表(Supplier)表服务实现类
 *
 * @author makejava
 * @date 2022-11-23 15:26:25
 */
@Slf4j
@Service("supplierService")
public class SupplierVoServiceImpl implements SupplierVoService {

    @Autowired(required = false)
    private SupplierBankRepository supplierBankRepository;

    @Autowired(required = false)
    private SupplierRepository supplierRepository;

    @Autowired(required = false)
    private List<SupplierEventListener> eventListeners;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private NebulaNetEventClient nebulaNetEventClient;

    /**
     * 分页条件查询
     *
     * @param pageable 分页参数
     * @param dto      查询条件
     * @return 供应商列表
     */
    @Override
    public Page<SupplierVo> findByConditions(Pageable pageable, SupplierDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new SupplierDto());
        Page<SupplierVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.supplierRepository.findByConditions(page, dto);
    }

    @Override
    public List<SupplierVo> findByConditions(SupplierDto dto) {
        if (dto == null) {
            return Lists.newArrayList();
        }
        List<SupplierEntity> supplierEntities = this.supplierRepository.findByConditions(dto);

        if (CollectionUtils.isEmpty(supplierEntities)) {
            return Lists.newArrayList();
        }
        return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(supplierEntities, SupplierEntity.class, SupplierVo.class, HashSet.class, ArrayList.class));
    }

    @Override
    public List<SupplierVo> findBySupplierCodes(List<String> supplierCodes) {

        if (CollectionUtils.isEmpty(supplierCodes)) {
            return Lists.newArrayList();
        }
        List<SupplierEntity> supplierEntities = this.supplierRepository.findBySupplierCodes(supplierCodes);
        if (CollectionUtils.isEmpty(supplierEntities)) {
            return Lists.newArrayList();
        }
        return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(supplierEntities, SupplierEntity.class, SupplierVo.class, HashSet.class, ArrayList.class));
    }

    /**
     * 分页查询供应商银行信息
     *
     * @param pageable
     * @param dto
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.mdm.business.supplier.sdk.vo.SupplierBankVo>
     * @author huojia
     * @date 2022/12/30 9:58
     **/
    @Override
    public Page<SupplierBankVo> findDetailById(Pageable pageable, SupplierDto dto) {
        if (StringUtils.isBlank(dto.getId())) {
            return null;
        }
        SupplierEntity supplier = this.supplierRepository.findById(dto.getId());
        if (supplier == null) {
            return null;
        }
        dto.setSupplierCode(supplier.getSupplierCode());
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new SupplierDto());
        Page<SupplierVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.supplierBankRepository.findByConditions(page, dto);
    }

    @Override
    public Page<SupplierBankVo> findDetailByCode(Pageable pageable, SupplierDto dto) {
        if (StringUtils.isBlank(dto.getSupplierCode())) {
            return null;
        }
        SupplierEntity supplier = this.supplierRepository.findBySupplierCode(dto.getSupplierCode());
        if (supplier == null) {
            return null;
        }
        dto.setSupplierCode(supplier.getSupplierCode());
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new SupplierDto());
        Page<SupplierVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.supplierBankRepository.findByConditions(page, dto);
    }


    /**
     * 创建供应商
     *
     * @param supplierVo 供应商VO
     * @return 创建成功的供应商VO
     */
    @Override
    @Transactional
    public SupplierVo create(SupplierVo supplierVo) {
        this.createValidation(supplierVo);
        supplierVo.setTenantCode(TenantUtils.getTenantCode());
        supplierVo.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        supplierVo.setEnableStatus(EnableStatusEnum.ENABLE.getCode());

        SupplierEntity supplier = nebulaToolkitService.copyObjectByWhiteList(supplierVo, SupplierEntity.class, HashSet.class, ArrayList.class);
        this.supplierRepository.saveOrUpdate(supplier);

        supplierVo.setId(supplier.getId());

        //crud监听事件
        if (!CollectionUtils.isEmpty(eventListeners)) {
            for (SupplierEventListener supplierEventListener : eventListeners) {
                supplierEventListener.onCreate(supplierVo);
            }
        }

        // 发送通知
        SupplierLogEventDto logEventDto = new SupplierLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(supplierVo);
        SerializableBiConsumer<SupplierLogEventListener, SupplierLogEventDto> onCreate =
                SupplierLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, SupplierLogEventListener.class, onCreate);
        return supplierVo;
    }


    /**
     * 更新供应商信息
     *
     * @param supplierVo new供应商VO
     * @return 更新结果
     */
    @Override
    @Transactional
    public SupplierVo update(SupplierVo supplierVo) {
        this.updateValidation(supplierVo);
        String currentId = supplierVo.getId();
        SupplierEntity current = supplierRepository.findById(currentId);
        current = Validate.notNull(current, "修改信息不存在");
        SupplierVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(current, SupplierVo.class, HashSet.class, ArrayList.class);


        current.setSupplierName(supplierVo.getSupplierName());
        current.setBankCard(supplierVo.getBankCard());
        current.setBankAccount(supplierVo.getBankAccount());
        current.setUnionPayBankCard(supplierVo.getUnionPayBankCard());
        current.setContactName(supplierVo.getContactName());
        current.setContactPhone(supplierVo.getContactPhone());

        this.supplierRepository.saveOrUpdate(current);
        //crud监听事件
        if (!CollectionUtils.isEmpty(eventListeners)) {
            for (SupplierEventListener supplierEventListener : eventListeners) {
                supplierEventListener.onUpdate(oldVo, supplierVo);
            }
        }

        // 发送修改通知
        SupplierLogEventDto logEventDto = new SupplierLogEventDto();
        logEventDto.setOriginal(oldVo);
        logEventDto.setNewest(supplierVo);
        SerializableBiConsumer<SupplierLogEventListener, SupplierLogEventDto> onUpdate =
                SupplierLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, SupplierLogEventListener.class, onUpdate);
        return supplierVo;
    }


    /**
     * 按id集合启用
     *
     * @param ids 主键主键列表
     */
    @Override
    @Transactional
    public void enableBatch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.supplierRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    }


    /**
     * 按id集合禁用
     *
     * @param ids 禁用主键列表
     */
    @Override
    @Transactional
    public void disableBatch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.supplierRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    }


    /**
     * 逻辑删除
     *
     * @param ids 删除主键列表
     */
    @Override
    @Transactional
    public void updateDelFlagByIds(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
        this.supplierRepository.updateDelFlagByIds(ids);
    }


    /**
     * 创建供应商时做的验证
     *
     * @param supplierVo 供应商vo
     */
    private void createValidation(SupplierVo supplierVo) {

        this.validation(supplierVo);
    }

    /**
     * 修改供应商时做的验证
     *
     * @param supplierVo 供应商vo
     */
    private void updateValidation(SupplierVo supplierVo) {

        this.validation(supplierVo);
    }

    /**
     * 公共的验证逻辑
     *
     * @param supplierVo 供应商vo
     */
    private void validation(SupplierVo supplierVo) {
        Validate.notEmpty(supplierVo.getSupplierCode(), "供应商编码不能为空");
        Validate.notEmpty(supplierVo.getSupplierName(), "供应商名称不能为空");
    }
}
