package com.biz.crm.mdm.business.table.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.mdm.business.table.local.entity.MdmColumnConfigEntity;
import com.biz.crm.mdm.business.table.local.mapper.ColumnConfigMapper;
import com.biz.crm.mdm.business.table.sdk.dto.ColumnConfigPaginationDto;
import com.biz.crm.mdm.business.table.sdk.vo.ColumnConfigVo;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * @author hefan
 */
@Component
public class ColumnConfigRepository extends ServiceImpl<ColumnConfigMapper, MdmColumnConfigEntity> {

  @Autowired(required = false)
  private ColumnConfigMapper columnConfigMapper;

  /**
   * 根据 parentCode，functionCodeDB，修改functionCode
   *
   * @param parentCode
   * @param functionCodeDB
   * @param functionCode
   */
  public void updateFunctionCodeByParentCodeAndFunctionCode(String parentCode, String functionCodeDB, String functionCode) {
    this.lambdaUpdate()
        .eq(MdmColumnConfigEntity::getParentCode, parentCode)
        .eq(MdmColumnConfigEntity::getFunctionCode, functionCodeDB)
        .set(MdmColumnConfigEntity::getFunctionCode, functionCode)
        .update();
  }

  /**
   * 据 parentCode集合，functionCode集合删除
   *
   * @param parentCodeList
   * @param functionCodeList
   */
  public void deleteByParentCodeInAndFunctionCodeIn(List<String> parentCodeList, List<String> functionCodeList) {
    this.lambdaUpdate()
        .in(MdmColumnConfigEntity::getParentCode, parentCodeList)
        .in(MdmColumnConfigEntity::getFunctionCode, functionCodeList)
        .remove();
  }

  /**
   * 分页 动态条件 查询
   *
   * @param pageable
   * @param columnConfigPaginationDto
   * @return
   */
  public Page<ColumnConfigVo> findByCondition(Pageable pageable, ColumnConfigPaginationDto columnConfigPaginationDto) {
    Page<ColumnConfigVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.columnConfigMapper.findByCondition(page, columnConfigPaginationDto);
  }

  /**
   * 根据 parentCode functionCode 查询
   *
   * @param parentCode
   * @param functionCode
   * @return
   */
  public List<MdmColumnConfigEntity> findByParentCodeAndFunctionCode(String parentCode, String functionCode) {
    return this.lambdaQuery()
        .eq(MdmColumnConfigEntity::getParentCode, parentCode)
        .eq(MdmColumnConfigEntity::getFunctionCode, functionCode)
        .list();
  }

  /**
   * 查询一条数据
   * 动态条件：parentCode、functionCode、field
   *
   * @param parentCode
   * @param functionCode
   * @param field
   * @return
   */
  public MdmColumnConfigEntity findByParentCodeAndFunctionCodeAndField(String parentCode, String functionCode, String field) {
    MdmColumnConfigEntity one = this.lambdaQuery()
        .eq(!StringUtils.isBlank(parentCode), MdmColumnConfigEntity::getParentCode, parentCode)
        .eq(!StringUtils.isBlank(functionCode), MdmColumnConfigEntity::getFunctionCode, functionCode)
        .eq(!StringUtils.isBlank(field), MdmColumnConfigEntity::getField, field)
        .one();
    return one;
  }

  /**
   * 根据 parentCode functionCode 删除
   *
   * @param parentCode
   * @param functionCode
   */
  public void deleteByParentCodeAndFunctionCode(String parentCode, String functionCode) {
    this.lambdaUpdate()
        .eq(MdmColumnConfigEntity::getParentCode, parentCode)
        .eq(MdmColumnConfigEntity::getFunctionCode, functionCode)
        .remove();
  }

  /**
   * 根据 parentCode functionCode field 删除
   *
   * @param parentCode
   * @param functionCode
   * @param field
   */
  public void deleteByParentCodeAndFunctionCodeAndField(String parentCode, String functionCode, String field) {
    this.lambdaUpdate()
        .eq(MdmColumnConfigEntity::getParentCode, parentCode)
        .eq(MdmColumnConfigEntity::getFunctionCode, functionCode)
        .eq(MdmColumnConfigEntity::getField, field)
        .remove();
  }

  /**
   * 根据 ID 集合 查询
   *
   * @param ids
   * @return
   */
  public List<MdmColumnConfigEntity> findByIdIn(List<String> ids) {
    List<MdmColumnConfigEntity> list = this.lambdaQuery()
        .in(MdmColumnConfigEntity::getId, ids)
        .list();
    return list;
  }

  /**
   * 根据 ID集合 修改 enableStatus
   *
   * @param ids
   * @param enableStatusCode
   */
  public void updateEnableStatusByIdIn(List<String> ids, String enableStatusCode) {
    this.lambdaUpdate()
        .in(MdmColumnConfigEntity::getId, ids)
        .set(MdmColumnConfigEntity::getEnableStatus, enableStatusCode)
        .update();
  }

  /**
   * 根据 parentCode functionCode 【columnExport】查询 字段配置
   *   【】代表可选
   *    方法名Opt 代码可选
   * @param parentCode
   * @param functionCode
   * @param columnExport
   * @return
   */
  public List<MdmColumnConfigEntity> findByParentCodeAndFunctionCodeAndColumnExportOpt(String parentCode, String functionCode, String columnExport) {
    List<MdmColumnConfigEntity> list = this.lambdaQuery()
        .eq(MdmColumnConfigEntity::getParentCode, parentCode)
        .eq(MdmColumnConfigEntity::getFunctionCode, functionCode)
        .eq(!StringUtils.isEmpty(columnExport), MdmColumnConfigEntity::getColumnExport, columnExport)
        .select(MdmColumnConfigEntity::getField, MdmColumnConfigEntity::getTitle, MdmColumnConfigEntity::getFormorder, MdmColumnConfigEntity::getDictCode, MdmColumnConfigEntity::getColumnExport)
        .list();
    return list;
  }

  /**
   * 发现由父代码
   *
   * @param parentCode 父代码
   * @return {@link List}<{@link ColumnConfigVo}>
   */
  public List<MdmColumnConfigEntity> findByParentCode(String parentCode) {
    return this.lambdaQuery()
        .eq(MdmColumnConfigEntity::getParentCode, parentCode)
        .list();
  }
}
