package com.biz.crm.mdm.business.table.local.service.internal;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.mdm.business.dictionary.sdk.constant.DictConstant;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.table.local.entity.MdmFunctionSubEntity;
import com.biz.crm.mdm.business.table.local.repository.FunctionSubRepository;
import com.biz.crm.mdm.business.table.sdk.dto.PermissionObjDto;
import com.biz.crm.mdm.business.table.sdk.service.PermissionObjVoService;
import com.biz.crm.mdm.business.table.sdk.vo.PermissionObjVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author hefan
 */
@Slf4j
@Service
public class PermissionObjVoServiceImpl implements PermissionObjVoService {

  @Autowired(required = false)
  private DictToolkitService dictToolkitService;
  @Autowired(required = false)
  private FunctionSubRepository functionSubRepository;

  @Override
  public List<PermissionObjVo> findByParentCodeAndFunctionCode(String parentCode, String functionCode) {
    /**
     * 查询了字典里的权限，还是树形的。
     * 查询了functionSub
     * --functionSub没有配置权限，权限树所有都设置select=0
     * --functionSub配置了权限，权限树根据functionSub配置，设置为包含 select=1，不包含 select=0
     */
    List<DictDataVo> permissionTree = this.dictToolkitService.tree(DictConstant.PERMISSION_OBJ);
    if (CollectionUtils.isEmpty(permissionTree)) {
      return new ArrayList<>();
    }
    MdmFunctionSubEntity mdmFunctionSubEntity = this.functionSubRepository.findPermissionObjByParentCodeAndFunctionCode(parentCode,functionCode);
    log.info("[数据权限][已配置的权限对象]:{}", mdmFunctionSubEntity);
    if (mdmFunctionSubEntity == null || StringUtils.isBlank(mdmFunctionSubEntity.getPermissionObj())) {
      return permissionTree.stream().map(x -> {
        PermissionObjVo permissionObjVo = new PermissionObjVo();
        permissionObjVo.setSelect(BooleanEnum.FALSE.getNumStr());
        permissionObjVo.setPermissionObjCode(x.getDictCode());
        permissionObjVo.setPermissionObjName(x.getDictValue());
        permissionObjVo.setRemarks(x.getDictValue());
        return permissionObjVo;
      }).collect(Collectors.toList());
    }
    List<String> permissionObjList = Arrays.asList(mdmFunctionSubEntity.getPermissionObj().split(","));
    return permissionTree
        .stream()
        .map(x -> {
          PermissionObjVo permissionObjVo = new PermissionObjVo();
          if (permissionObjList.contains(x.getDictCode())) {
            permissionObjVo.setSelect(BooleanEnum.TRUE.getNumStr());
          } else {
            permissionObjVo.setSelect(BooleanEnum.FALSE.getNumStr());
          }
          permissionObjVo.setPermissionObjCode(x.getDictCode());
          permissionObjVo.setPermissionObjName(x.getDictValue());
          permissionObjVo.setRemarks(x.getDictValue());
          return permissionObjVo;
        }).collect(Collectors.toList());
  }

  @Transactional
  @Override
  public void create(PermissionObjDto permissionObjDto) {
    /**
     * 查询数据库内是否存在数据
     * --入参中没有权限信息，将数据库内数据的权限置空
     * --入参中有权限信息，修改数据库信息
     */
    String functionCode = permissionObjDto.getFunctionCode();
    Validate.notBlank(functionCode, "功能列表编码不能为空");
    String parentCode = permissionObjDto.getParentCode();
    Validate.notBlank(parentCode, "菜单编码不能为空");
    Integer count = this.functionSubRepository.countByParentCodeAndFunctionCode(parentCode, functionCode);
    Validate.isTrue(count != null && count > 0, "功能列表不存在");
    List<String> permissionObjCodes = permissionObjDto.getPermissionObjCodes();
    if (CollectionUtils.isEmpty(permissionObjCodes)) {
      this.functionSubRepository.updatePermissionObjByParentCodeAndFunctionCode(parentCode, functionCode, null);
      return;
    }
    String collect = StringUtils.join(permissionObjCodes, ",");
    this.functionSubRepository.updatePermissionObjByParentCodeAndFunctionCode(parentCode, functionCode, collect);
  }
}
