package com.biz.crm.mdm.business.terminal.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalSupply;
import com.biz.crm.mdm.business.terminal.local.mapper.TerminalSupplyMapper;
import com.biz.crm.mdm.business.terminal.sdk.enums.TerminalSupplyTypeEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Set;
import org.springframework.stereotype.Component;

/**
 * 终端业务员(repository)
 *
 * @author sunx
 * @since 2021-10-18 17:49:34
 */
@Component
public class TerminalSupplyRepository extends ServiceImpl<TerminalSupplyMapper, TerminalSupply> {

  /**
   * 根据终端编码集合获取供货明细数据
   *
   * @param terminalCodeList
   * @return
   */
  public List<TerminalSupply> findByTerminalCodes(List<String> terminalCodeList) {
    return this.lambdaQuery()
        .eq(TerminalSupply::getTenantCode, TenantUtils.getTenantCode())
        .in(TerminalSupply::getTerminalCode, terminalCodeList)
        .list();
  }

  /**
   * 根据终端编码集合删除供货关系信息
   *
   * @param terminalCodeList
   */
  public void deleteByTerminalCodes(List<String> terminalCodeList) {
    LambdaQueryWrapper<TerminalSupply> lambdaQuery = Wrappers.lambdaQuery();
    lambdaQuery
        .eq(TerminalSupply::getTenantCode, TenantUtils.getTenantCode())
        .in(TerminalSupply::getTerminalCode, terminalCodeList);
    baseMapper.delete(lambdaQuery);
  }

  /**
   * 根据岗位编码+终端编码获取匹配的供货关系信息
   *
   * @param positionCodes
   * @param terminalCodeList
   * @return
   */
  public List<TerminalSupply> findByPositionCodesAndTerminalCodes(
      List<String> positionCodes, List<String> terminalCodeList) {
    return this.lambdaQuery()
        .eq(TerminalSupply::getTenantCode, TenantUtils.getTenantCode())
        .eq(TerminalSupply::getSupplyType, TerminalSupplyTypeEnum.DEFAULT.getValue())
        .in(TerminalSupply::getPositionCode, positionCodes)
        .in(TerminalSupply::getTerminalCode, terminalCodeList)
        .list();
  }

  /**
   * 根据客户用户和终端编码获取供货关系信息
   *
   * @param userName
   * @param terminalCodeList
   * @return
   */
  public List<TerminalSupply> findByCustomerUserNameAndTerminalCodes(
      String userName, List<String> terminalCodeList) {
    return this.lambdaQuery()
        .eq(TerminalSupply::getTenantCode, TenantUtils.getTenantCode())
        .eq(TerminalSupply::getSupplyType, TerminalSupplyTypeEnum.CUSTOMER_USER.getValue())
        .eq(TerminalSupply::getUserName, userName)
        .in(TerminalSupply::getTerminalCode, terminalCodeList)
        .list();
  }

  /**
   * 根据客户编码和终端编码获取供货关系信息
   *
   * @param customerCode
   * @param terminalCodeList
   * @param supplyType
   * @return
   */
  public List<TerminalSupply> findByCustomerCodeAndTerminalCodesAndSupplyType(
      String customerCode, List<String> terminalCodeList, String supplyType) {
    return this.lambdaQuery()
        .eq(TerminalSupply::getTenantCode, TenantUtils.getTenantCode())
        .eq(StringUtils.isNotBlank(supplyType), TerminalSupply::getSupplyType, supplyType)
        .eq(TerminalSupply::getCustomerCode, customerCode)
        .in(TerminalSupply::getTerminalCode, terminalCodeList)
        .list();
  }

  /**
   * 根据id集合获取供货明细
   *
   * @param supplyIdList
   * @return
   */
  public List<TerminalSupply> findByIds(List<String> supplyIdList) {
    return this.baseMapper.selectBatchIds(supplyIdList);
  }

  /**
   * 通过用户名 供货关系类型 查询
   *
   * @param userNameList
   * @param value
   */
  public List<TerminalSupply> findByUserNames(List<String> userNameList, String value) {
    return this.lambdaQuery()
        .eq(TerminalSupply::getSupplyType, value)
        .in(TerminalSupply::getUserName, userNameList)
        .eq(TerminalSupply::getTenantCode, TenantUtils.getTenantCode())
        .list();
  }

  /**
   * 获取终端关联商品的供货经销商编码
   *
   * @param terminalCode
   * @param productCodeSet
   * @param productLevelCodeSet
   * @return
   */
  public Set<String> findTerminalSupplyCustomerCodeSet(
      String terminalCode, Set<String> productCodeSet, Set<String> productLevelCodeSet) {
    Set<String> set = Sets.newHashSet();
    if (CollectionUtils.isNotEmpty(productCodeSet)) {
      Set<String> curSet =
          this.baseMapper.findTerminalSupplyCustomerCodeSetByProduct(
              terminalCode, StringUtils.EMPTY, "1", productCodeSet, TenantUtils.getTenantCode());
      if (CollectionUtils.isNotEmpty(curSet)) {
        set.addAll(curSet);
      }
    }
    if (CollectionUtils.isNotEmpty(productLevelCodeSet)) {
      Set<String> curSet =
          this.baseMapper.findTerminalSupplyCustomerCodeSetByProductLevel(
              terminalCode,
              StringUtils.EMPTY,
              "2",
              productLevelCodeSet,
              TenantUtils.getTenantCode());
      if (CollectionUtils.isNotEmpty(curSet)) {
        set.addAll(curSet);
      }
    }
    Set<String> curSet =
        this.baseMapper.findTerminalSupplyCustomerCodeSetByNoProduct(
            terminalCode, TenantUtils.getTenantCode());
    if (CollectionUtils.isNotEmpty(curSet)) {
      set.addAll(curSet);
    }
    return set;
  }

  /**
   * 根据客户编码查找供货关系信息
   *
   * @param customerCodeSet
   * @return
   */
  public List<TerminalSupply> findByCustomerCodes(Set<String> customerCodeSet) {
    return this.lambdaQuery()
        .eq(TerminalSupply::getTenantCode, TenantUtils.getTenantCode())
        .in(TerminalSupply::getCustomerCode, customerCodeSet)
        .list();
  }

  /**
   * 根据岗位编码查找供货关系信息
   *
   * @param positionCodeSet
   * @return
   */
  public List<TerminalSupply> findByPositionCodes(Set<String> positionCodeSet) {
    return this.lambdaQuery()
        .eq(TerminalSupply::getTenantCode, TenantUtils.getTenantCode())
        .in(TerminalSupply::getPositionCode, positionCodeSet)
        .list();
  }
}
